/*
 *  Copyright (C) 2006-2025, Thomas Maier-Komor
 *
 *  This is the source code of xjobs.
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>

#include "colortty.h"
#include "log.h"
#include "support.h"

verbose_t Verbose = Status;

extern void finalize(void);

static int Log = STDERR_FILENO;
static const char *LogName = 0;


void set_log(const char *fn)
{
	int newlog = open(fn,O_WRONLY|O_CREAT|O_APPEND|O_CLOEXEC,0644);
	if (newlog == -1) {
		warn("unable to open logfile %s: %s\n",fn,strerror(errno));
		return;
	}
	int err = fcntl(newlog,F_SETFD,FD_CLOEXEC);
	assert(err != -1);
	if (Log != STDERR_FILENO)
		(void) close(Log);
	Log = newlog;
	LogName = fn;
}


int write_log(verbose_t v, const char *s, size_t l)
{
	if (Verbose < v)
		return Write(Log,s,l);
	return 0;
}


static void vprint_log(verbose_t v, const char *pfx, const char *msg, va_list val)
{
	if (Verbose >= v) {
		char buf[4096], *b = buf;
		size_t pl = 0;
		if ((Log == STDERR_FILENO) && (0 != pfx)) {
			pl = strlen(pfx);
			memcpy(b,pfx,pl);
			b += pl;
		}
		b += vsnprintf(b,sizeof(buf)-(b-buf),msg,val);
		if (b - buf > sizeof(buf)) {
			b = buf+sizeof(buf)-3;
			*b++ = '.';
			*b++ = '.';
			*b++ = '.';
		}
		if (v == Status)
			Write(STDERR_FILENO,buf,b-buf);
		else
			Write(Log,buf+pl,b-buf-pl);
		if ((v < Status) && (STDERR_FILENO != Log))
			Write(STDERR_FILENO,buf,b-buf);
	}
}


void dbug(const char *msg, ...)
{
	va_list val;
	va_start(val,msg);
	vprint_log(Debug,PfxDebug,msg,val);
	va_end(val);
}


void info(const char *msg, ...)
{
	va_list val;
	va_start(val,msg);
	vprint_log(Info,PfxInfo,msg,val);
	va_end(val);
}


void status(const char *msg, ...)
{
	va_list val;
	va_start(val,msg);
	vprint_log(Status,PfxStart,msg,val);
	va_end(val);
}


void warn(const char *msg, ...)
{
	va_list val;
	va_start(val,msg);
	vprint_log(Warning,PfxWarn,msg,val);
	va_end(val);
}


void error(const char *msg, ...)
{
	va_list val;
	va_start(val,msg);
	vprint_log(Error,PfxError,msg,val);
	va_end(val);
	finalize();
}

