// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { Clipboard, SessionContextDialogs } from '@jupyterlab/apputils';
import { Cell, CodeCellModel } from '@jupyterlab/cells';
import { CodeEditorWrapper } from '@jupyterlab/codeeditor';
import { CodeMirrorEditorFactory, CodeMirrorMimeTypeService, EditorExtensionRegistry, EditorLanguageRegistry, ybinding } from '@jupyterlab/codemirror';
import { Context } from '@jupyterlab/docregistry';
import { DEFAULT_OUTPUTS as TEST_OUTPUTS, defaultRenderMime as testRenderMime } from '@jupyterlab/rendermime/lib/testutils';
import { ServiceManager } from '@jupyterlab/services';
import { ServiceManagerMock } from '@jupyterlab/services/lib/testutils';
import { UUID } from '@lumino/coreutils';
import * as defaultContent from './default.json';
import { NotebookModel } from './model';
import { NotebookModelFactory } from './modelfactory';
import { NotebookPanel } from './panel';
import { Notebook, StaticNotebook } from './widget';
import { NotebookWidgetFactory } from './widgetfactory';
export const DEFAULT_CONTENT = defaultContent;
/**
 * Create and initialize context for a notebook.
 */
export async function initNotebookContext(options = {}) {
    var _a, _b, _c;
    const factory = Private.notebookFactory;
    const manager = options.manager || Private.getManager();
    const path = options.path || UUID.uuid4() + '.ipynb';
    console.debug('Initializing notebook context for', path, 'kernel:', options.startKernel);
    const startKernel = options.startKernel === undefined ? false : options.startKernel;
    await manager.ready;
    const context = new Context({
        sessionDialogs: new SessionContextDialogs(),
        manager,
        factory,
        path,
        kernelPreference: {
            shouldStart: startKernel,
            canStart: startKernel,
            shutdownOnDispose: true,
            name: (_a = manager.kernelspecs.specs) === null || _a === void 0 ? void 0 : _a.default
        }
    });
    await context.initialize(true);
    if (startKernel) {
        await context.sessionContext.initialize();
        await ((_c = (_b = context.sessionContext.session) === null || _b === void 0 ? void 0 : _b.kernel) === null || _c === void 0 ? void 0 : _c.info);
    }
    return context;
}
/**
 * The default notebook content.
 */
export var NBTestUtils;
(function (NBTestUtils) {
    NBTestUtils.DEFAULT_OUTPUTS = TEST_OUTPUTS;
    NBTestUtils.defaultEditorConfig = { ...StaticNotebook.defaultEditorConfig };
    const editorServices = (function () {
        const languages = new EditorLanguageRegistry();
        EditorLanguageRegistry.getDefaultLanguages()
            .filter(lang => ['Python'].includes(lang.name))
            .forEach(lang => {
            languages.addLanguage(lang);
        });
        const extensions = new EditorExtensionRegistry();
        EditorExtensionRegistry.getDefaultExtensions()
            .filter(ext => ['autoClosingBrackets'].includes(ext.name))
            .forEach(ext => {
            extensions.addExtension(ext);
        });
        extensions.addExtension({
            name: 'binding',
            factory: ({ model }) => EditorExtensionRegistry.createImmutableExtension(ybinding({ ytext: model.sharedModel.ysource }))
        });
        const factoryService = new CodeMirrorEditorFactory({
            languages,
            extensions
        });
        const mimeTypeService = new CodeMirrorMimeTypeService(languages);
        return {
            factoryService,
            mimeTypeService
        };
    })();
    NBTestUtils.editorFactory = editorServices.factoryService.newInlineEditor.bind(editorServices.factoryService);
    NBTestUtils.mimeTypeService = editorServices.mimeTypeService;
    /**
     * Get a copy of the default rendermime instance.
     */
    function defaultRenderMime() {
        return testRenderMime();
    }
    NBTestUtils.defaultRenderMime = defaultRenderMime;
    NBTestUtils.clipboard = Clipboard.getInstance();
    /**
     * Create a base cell content factory.
     */
    function createBaseCellFactory() {
        return new Cell.ContentFactory({ editorFactory: NBTestUtils.editorFactory });
    }
    NBTestUtils.createBaseCellFactory = createBaseCellFactory;
    /**
     * Create a new code cell content factory.
     */
    function createCodeCellFactory() {
        return new Cell.ContentFactory({ editorFactory: NBTestUtils.editorFactory });
    }
    NBTestUtils.createCodeCellFactory = createCodeCellFactory;
    /**
     * Create a cell editor widget.
     */
    function createCellEditor(model) {
        return new CodeEditorWrapper({
            model: model !== null && model !== void 0 ? model : new CodeCellModel(),
            factory: NBTestUtils.editorFactory
        });
    }
    NBTestUtils.createCellEditor = createCellEditor;
    /**
     * Create a default notebook content factory.
     */
    function createNotebookFactory() {
        return new Notebook.ContentFactory({ editorFactory: NBTestUtils.editorFactory });
    }
    NBTestUtils.createNotebookFactory = createNotebookFactory;
    /**
     * Create a default notebook panel content factory.
     */
    function createNotebookPanelFactory() {
        return new NotebookPanel.ContentFactory({ editorFactory: NBTestUtils.editorFactory });
    }
    NBTestUtils.createNotebookPanelFactory = createNotebookPanelFactory;
    /**
     * Create a notebook widget.
     */
    function createNotebook() {
        return new Notebook({
            rendermime: defaultRenderMime(),
            contentFactory: createNotebookFactory(),
            mimeTypeService: NBTestUtils.mimeTypeService,
            notebookConfig: {
                ...StaticNotebook.defaultNotebookConfig,
                windowingMode: 'none'
            }
        });
    }
    NBTestUtils.createNotebook = createNotebook;
    /**
     * Create a notebook panel widget.
     */
    function createNotebookPanel(context) {
        return new NotebookPanel({
            content: createNotebook(),
            context
        });
    }
    NBTestUtils.createNotebookPanel = createNotebookPanel;
    /**
     * Populate a notebook with default content.
     */
    function populateNotebook(notebook) {
        const model = new NotebookModel();
        model.fromJSON(DEFAULT_CONTENT);
        notebook.model = model;
    }
    NBTestUtils.populateNotebook = populateNotebook;
    function createNotebookWidgetFactory(toolbarFactory) {
        return new NotebookWidgetFactory({
            name: 'notebook',
            fileTypes: ['notebook'],
            rendermime: defaultRenderMime(),
            toolbarFactory,
            contentFactory: createNotebookPanelFactory(),
            mimeTypeService: NBTestUtils.mimeTypeService,
            editorConfig: NBTestUtils.defaultEditorConfig
        });
    }
    NBTestUtils.createNotebookWidgetFactory = createNotebookWidgetFactory;
    /**
     * Create a context for a file.
     */
    async function createMockContext(startKernel = false) {
        const path = UUID.uuid4() + '.txt';
        const manager = new ServiceManagerMock();
        const factory = new NotebookModelFactory({});
        const context = new Context({
            sessionDialogs: new SessionContextDialogs(),
            manager,
            factory,
            path,
            kernelPreference: {
                shouldStart: startKernel,
                canStart: startKernel,
                autoStartDefault: startKernel
            }
        });
        await context.initialize(true);
        await context.sessionContext.initialize();
        return context;
    }
    NBTestUtils.createMockContext = createMockContext;
})(NBTestUtils || (NBTestUtils = {}));
/**
 * A namespace for private data.
 */
var Private;
(function (Private) {
    let manager;
    Private.notebookFactory = new NotebookModelFactory();
    /**
     * Get or create the service manager singleton.
     */
    function getManager() {
        if (!manager) {
            manager = new ServiceManager({ standby: 'never' });
        }
        return manager;
    }
    Private.getManager = getManager;
})(Private || (Private = {}));
//# sourceMappingURL=testutils.js.map