/***  mmd.c: Morrow Micro-Decision
 
    Created by Patrick Linstruth (patrick@deltecent.com)
 
    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    PETER SCHORN BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
    IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

    Except as contained in this notice, the name of Patrick Linstruth shall not
    be used in advertising or otherwise to promote the sale, use or other dealings
    in this Software without prior written authorization from Patrick Linstruth.

    These functions support the Morrow Micro-Decision Models 2 and 3.

    There are two devices defined, MMD and MMDM. The MMD device supports 4
    floppy drives and the console serial port.

    The Micro-Decision's second serial (modem) port uses I/O ports 0xfe and 0xff.
    Z80 or 8080 programs can also communicate with the SIMH pseudo device via
    port 0xfe. Due to this conflict, the MMDM device is disabled by default. In
    order to use the modem port, execute "SET MMDM ENABLE". Note that when the
    MMDM device is enabled it will not be possible to use the SIMH pseudo device
    and CP/M utilities such as R.COM and W.COM.

    Units:

    MMD0 - Drive A
    MMD1 - Drive B
    MMD2 - Drive C
    MMD3 - Drive D
    MMD4 - Console Port
    MMDM - Modem Port

    ** NOTES **

    PORT              FUNCTION
    ----              --------
     F5               MOTOR CHECK / DIAG / CENTRONICS STATUS
                      Bit 0,1 Motor Mask (read)
                      Bit 2   *Index (read)
                      Bit 3   Printer Ack (read)
                      Bit 4   Printer Busy (read)
                      Bit 5   *Diag (read)
                      Bit 6   Unknown
                      Bit 7   *Printer Strobe (write)

    DISK FORMAT
    -----------
    40 Cylinders
    5 Sectors/Track
    1024 Bytes per Track
   
    Tracks 0-1 Bootable Image (10K)

    Double sided disks have 80 tracks, with even numbered tracks on side 0
    and odd numbered tracks on side 1.

    Track = (2 * Cylinder) + Side

    MTAB contains one 9 byte entry for each logical drive.
    The bytes of each entry are defined as follows:

    Byte 0          DSKDEF0:
                    Bit 0-2 Motor control bit
                    Bit 3-4 Double sided mode:
                                    00=Even tracks on side 0,
                                       Odd tracks on side 1.
                                    01=1st 40 (or 80) tracks
                                       on side 0, remaining
                                       tracks on side 1.
                                    10=Both sides are treated
                                       as a single track with
                                       twice as many sectors.
                    Bit 5 Double sided drive if = 1.
                    Bit 6 Unused.
                    Bit 7 Foreign drive format if = 1.

    Byte 1          DSKDEF1:
                    Bit 0-1 Physical drive address.
                    Bit 2   Double sided media if = 1.
                    Bit 3-4 Sector size:
                                    00=128
                                    01=256
                                    10=512
                                    11=1024.
                    Bit 5   Tracks: 0=40; 1=80.
                    Bit 6   Density: 0=single; 1=double.
                    Bit 7   Virtual drive: 1=virtual.

    Byte 2          Motor on wait time in increments of 4 ms.

    Byte 3          Head settle time (after seek) in increments
                    of 4 ms.

    Byte 4-5        The two parameter bytes for the FDC specify
                    command: Byte 4 = SRT/HUT
                             Byte 5 = HLT/ND
                             ND must be 1.

    Byte 6          EOT byte for FDC read or write commands.

    Byte 7          GPL byte for FDC read or write commands.

    Byte 8          Current track.

*/

/* #define DBG_MSG */

#include "altairz80_defs.h"
#include "sim_imd.h"
#include "sim_tmxr.h"

/********/
/* SIMH */
/********/

#ifdef DBG_MSG
#define DBG_PRINT(args) sim_printf args
#else
#define DBG_PRINT(args)
#endif

extern uint32 vectorInterrupt;            /* Interrupt Request */
extern uint8 dataBus[MAX_INT_VECTORS];    /* Data Bus Value    */
extern uint32 PCX;

extern uint32 sim_map_resource(uint32 baseaddr, uint32 size, uint32 resource_type,
                               int32 (*routine)(const int32, const int32, const int32), const char* name, uint8 unmap);

/* Debug flags */
#define VERBOSE_MSG         (1 << 0)
#define ERROR_MSG           (1 << 1)
#define RBUF_MSG            (1 << 2)
#define WBUF_MSG            (1 << 3)
#define FDC_MSG             (1 << 4)
#define STATUS_MSG          (1 << 5)

/* Debug Flags */
static DEBTAB mmd_dt[] = {
    { "VERBOSE",    VERBOSE_MSG,        "Verbose messages"      },
    { "ERROR",      ERROR_MSG,          "Error messages"        },
    { "FDC",        FDC_MSG,            "FDC Command messages"  },
    { "RBUF",       RBUF_MSG,           "Read Buffer messages"  },
    { "WBUF",       WBUF_MSG,           "Write Buffer messages" },
    { "STATUS",     STATUS_MSG,         "Status messages"       },
    { NULL,         0                                           }
};

/*****************************/
/* Local function prototypes */
/*****************************/

static t_stat mmd_reset(DEVICE *dptr);
static t_stat mmdm_reset(DEVICE *dptr);
static t_stat mmd_svc(UNIT *uptr);
static t_stat mmdm_svc(UNIT *uptr);
static t_stat mmd_sio1_svc(UNIT *uptr);
static t_stat mmd_attach(UNIT *uptr, CONST char *cptr);
static t_stat mmdm_attach(UNIT *uptr, CONST char *cptr);
static t_stat mmd_detach(UNIT *uptr);
static t_stat mmdm_detach(UNIT *uptr);
static t_stat mmd_config_line(DEVICE *dev, TMLN *tmln, int baud);
static t_stat mmd_boot(int32 unitno, DEVICE *dptr);
static const char* mmd_description(DEVICE *dptr);
static t_stat mmd_set_diag(UNIT *uptr, int32 val, CONST char *cptr, void *desc);
static t_stat mmd_show_diag(FILE *st, UNIT *uptr, int32 val, CONST void *desc);
static t_stat mmd_set_rom(UNIT *uptr, int32 val, CONST char *cptr, void *desc);
static t_stat mmd_show_rom(FILE *st, UNIT *uptr, int32 val, CONST void *desc);
static int32 mmddev(int32 Addr, int32 rw, int32 data);
static int32 mmdrom(int32 Addr, int32 rw, int32 data);
static int32 mmdmem(int32 Addr, int32 rw, int32 data);
static void intZ80(void);
static uint8 MMD_Dev_Read(uint32 Addr);
static uint8 MMD_FDC_Read_Data(void);
static uint8 MMD_Dev_Write(uint32 Addr, int32 data);
static void MMD_FDC_Write_Data(int32 Data);
static void MMD_Command_New(int32 Data);
static void MMD_Command_Stack(int32 Data);
static void MMD_Command_Proc(void);
static uint8 MMD_Result_Stack(void);
static uint8 MMD_Exec_Format(void);
static uint8 MMD_Exec_Read(void);
static uint8 MMD_Exec_Write(int32 Data);
static uint8 MMD_Term_Count(void);
static uint8 MMD_Seek_Sector(uint8 drive);
static uint8 MMD_Read_Sector(uint8 drive);
static uint8 MMD_Write_Sector(uint8 drive);
static void MMD_Dump_Sector(void);
static const char * MMD_Command_String(int32 command);

/***********/
/* ROM/RAM */
/***********/

#define MMD_ROM_BASE   0x0000
#define MMD_ROM_SIZE   4096
#define MMD_ROM_MASK   (MMD_ROM_SIZE-1)

static uint8 mmd_mem[MMD_ROM_SIZE];    /* 4K Banked Memory */

/* MD2 1.3 ROM is 2048 bytes */
static uint8 mmd_rom_13[MMD_ROM_SIZE] = {
    0xc3, 0x1d, 0x00, 0xc3, 0xba, 0x01, 0xc3, 0x10,
    0x04, 0xc3, 0xd3, 0x01, 0xc3, 0xe0, 0x01, 0xc3,
    0x64, 0x02, 0xc3, 0x93, 0x03, 0xc3, 0xdd, 0x03,
    0xc3, 0x96, 0x01, 0xd4, 0x05, 0xed, 0x46, 0x31,
    0x00, 0x00, 0xfd, 0x21, 0xf3, 0x05, 0xcd, 0x5b,
    0x00, 0x11, 0x0f, 0x07, 0xcd, 0xb1, 0x01, 0xcd,
    0x81, 0x00, 0xcd, 0xac, 0x00, 0xcd, 0x90, 0x00,
    0xcd, 0x00, 0xfc, 0x11, 0x8a, 0x07, 0xcd, 0xba,
    0x01, 0xcd, 0x78, 0x03, 0xfe, 0x0d, 0x20, 0xf9,
    0x11, 0x78, 0x00, 0xcd, 0xba, 0x01, 0xc3, 0x76,
    0x01, 0xcd, 0xba, 0x01, 0x11, 0xf3, 0x06, 0xcd,
    0xba, 0x01, 0x76, 0x3e, 0x80, 0xd3, 0xfd, 0xd3,
    0xff, 0xd3, 0xfd, 0xd3, 0xff, 0x3e, 0x40, 0xd3,
    0xfd, 0xd3, 0xff, 0x3e, 0xce, 0xd3, 0xfd, 0xd3,
    0xff, 0x3e, 0x37, 0xd3, 0xfd, 0xd3, 0xff, 0xc9,
    0x0d, 0x0a, 0x00, 0x3a, 0xa0, 0x00, 0xc5, 0xf6,
    0x04, 0x21, 0x00, 0x00, 0x01, 0xfd, 0x07, 0xcd,
    0x9f, 0x00, 0x11, 0xd9, 0x07, 0x20, 0xc2, 0xc9,
    0x21, 0x00, 0xfc, 0x01, 0xf8, 0x03, 0xcd, 0x9f,
    0x00, 0x11, 0xc6, 0x07, 0x20, 0xb3, 0xc9, 0xaf,
    0xf5, 0xf1, 0xae, 0x23, 0x0b, 0xf5, 0x78, 0xb1,
    0x20, 0xf7, 0xf1, 0xc9, 0x21, 0xc4, 0x00, 0x11,
    0x00, 0xfc, 0x01, 0xb2, 0x00, 0xed, 0xb0, 0x01,
    0x45, 0x03, 0xd5, 0xe1, 0x13, 0x36, 0xff, 0xed,
    0xb0, 0x36, 0x1f, 0xc9, 0xdb, 0xf6, 0x3a, 0x00,
    0x00, 0xfe, 0xc3, 0xd3, 0xf6, 0x20, 0x03, 0xc3,
    0x3b, 0x00, 0x11, 0x57, 0x07, 0xcd, 0xba, 0x01,
    0xdb, 0xf6, 0x21, 0x00, 0x00, 0x01, 0xff, 0xfb,
    0xcd, 0x31, 0xfc, 0xb7, 0xd3, 0xf6, 0x11, 0xc6,
    0x07, 0xc2, 0x51, 0x00, 0x11, 0x7b, 0x07, 0xcd,
    0xba, 0x01, 0xc3, 0x3b, 0x00, 0x22, 0xb2, 0xfc,
    0x21, 0x84, 0xfc, 0x56, 0x23, 0xcb, 0x7e, 0x20,
    0x18, 0x5e, 0xcb, 0x73, 0xcb, 0xb3, 0x23, 0xe5,
    0x28, 0x05, 0xcd, 0x57, 0xfc, 0x18, 0x07, 0xdd,
    0x21, 0x77, 0xfc, 0xcd, 0x5b, 0xfc, 0xe1, 0x18,
    0xe2, 0xaf, 0xc9, 0xdd, 0x21, 0x79, 0xfc, 0x2a,
    0xb2, 0xfc, 0x7b, 0x85, 0x6f, 0x7c, 0xce, 0x00,
    0x67, 0xcd, 0x75, 0xfc, 0x23, 0x23, 0x23, 0x78,
    0xbc, 0xd8, 0x20, 0xf5, 0x79, 0xbd, 0xd8, 0x18,
    0xf0, 0xdd, 0xe9, 0x72, 0xc9, 0x7e, 0xba, 0x20,
    0x01, 0xc9, 0xf1, 0xf1, 0xf1, 0x3e, 0xff, 0xc9,
    0xff, 0x00, 0x00, 0x01, 0xff, 0x02, 0x00, 0x41,
    0xff, 0x42, 0xff, 0x40, 0x00, 0x00, 0xff, 0x01,
    0x00, 0x02, 0x00, 0x42, 0x00, 0x40, 0xff, 0x41,
    0xaa, 0x00, 0x55, 0x02, 0xaa, 0x01, 0xaa, 0x41,
    0x55, 0x01, 0xaa, 0x40, 0x55, 0x00, 0x55, 0x42,
    0x55, 0x41, 0x55, 0x40, 0x00, 0x80, 0x21, 0xd4,
    0x05, 0x11, 0x00, 0xff, 0x01, 0x42, 0x00, 0xed,
    0xb0, 0xdd, 0x21, 0x00, 0xff, 0xfd, 0x21, 0x1f,
    0xff, 0xcd, 0x33, 0x04, 0xcd, 0x64, 0x02, 0xdd,
    0x7e, 0x07, 0xb7, 0xca, 0x00, 0xfe, 0x11, 0xd3,
    0x06, 0xc3, 0x51, 0x00, 0xdb, 0xfd, 0xcb, 0x47,
    0x28, 0xfa, 0x79, 0xd3, 0xfc, 0xc9, 0xdb, 0xfd,
    0xcb, 0x4f, 0x28, 0xfa, 0xdb, 0xfc, 0xe6, 0x7f,
    0xc9, 0x06, 0x32, 0x3e, 0x0a, 0xcd, 0xc3, 0x01,
    0x10, 0xf9, 0x1a, 0xb7, 0xc8, 0xcd, 0xc3, 0x01,
    0x13, 0x18, 0xf7, 0x4f, 0xd5, 0xe5, 0xfd, 0x6e,
    0x16, 0xfd, 0x66, 0x17, 0xcd, 0xd2, 0x01, 0xe1,
    0xd1, 0xc9, 0xe9, 0x21, 0x93, 0x03, 0xdd, 0x75,
    0x0a, 0xdd, 0x74, 0x0b, 0x0e, 0x06, 0x18, 0x0b,
    0x21, 0xdd, 0x03, 0xdd, 0x75, 0x0a, 0xdd, 0x74,
    0x0b, 0x0e, 0x05, 0xdd, 0x7e, 0x00, 0xcd, 0xb8,
    0x04, 0x23, 0x7e, 0xe6, 0x40, 0xb1, 0xdd, 0x77,
    0x16, 0xdd, 0x36, 0x15, 0x09, 0x3e, 0x18, 0xa6,
    0x0f, 0x0f, 0x0f, 0xdd, 0x77, 0x1b, 0x06, 0x80,
    0x11, 0x80, 0x00, 0xfe, 0x00, 0x28, 0x11, 0x06,
    0xff, 0x11, 0x00, 0x00, 0xfe, 0x01, 0x28, 0x08,
    0x16, 0x01, 0xfe, 0x02, 0x28, 0x02, 0x16, 0x03,
    0xdd, 0x70, 0x1e, 0xdd, 0xcb, 0x0d, 0x7e, 0x20,
    0x06, 0xdd, 0x73, 0x0c, 0xdd, 0x72, 0x0d, 0xaf,
    0xcb, 0x56, 0xdd, 0x46, 0x01, 0xdd, 0x4e, 0x02,
    0x28, 0x03, 0xcb, 0x18, 0x17, 0xdd, 0x77, 0x19,
    0x07, 0x07, 0xdd, 0x77, 0x09, 0xdd, 0x70, 0x08,
    0xdd, 0x70, 0x18, 0xdd, 0x71, 0x1a, 0x3e, 0x03,
    0xa6, 0xdd, 0xb6, 0x09, 0xdd, 0x77, 0x17, 0x11,
    0x05, 0x00, 0x19, 0x7e, 0xdd, 0x77, 0x1c, 0x23,
    0x7e, 0xdd, 0x77, 0x1d, 0xdd, 0x46, 0x04, 0x11,
    0x00, 0x03, 0xc5, 0xd5, 0x3e, 0x08, 0xcd, 0x35,
    0x04, 0xcd, 0x09, 0x03, 0xfd, 0xe5, 0xcd, 0xc9,
    0x02, 0xcd, 0xfd, 0x02, 0xfd, 0xe1, 0x06, 0x07,
    0xdd, 0xe5, 0xe1, 0x11, 0x0e, 0x00, 0x19, 0xcd,
    0x21, 0x05, 0xed, 0xa2, 0x20, 0xf9, 0xd1, 0xc1,
    0xdd, 0x7e, 0x0e, 0xe6, 0x40, 0x28, 0x2d, 0xcd,
    0x83, 0x05, 0x3e, 0x01, 0xb9, 0x28, 0x15, 0x3e,
    0x06, 0xb9, 0xcc, 0x0a, 0x04, 0x3e, 0x02, 0xb9,
    0xcc, 0x30, 0x04, 0x1c, 0x3e, 0x0b, 0xb8, 0xcc,
    0x2c, 0x04, 0x10, 0xb6, 0xdd, 0x7e, 0x07, 0xfe,
    0xff, 0x28, 0x0a, 0xcd, 0x17, 0x03, 0x38, 0x05,
    0xb7, 0xc2, 0x64, 0x02, 0x4f, 0xdd, 0x71, 0x07,
    0xc9, 0x0e, 0xfb, 0xdd, 0xe5, 0xe1, 0x11, 0x15,
    0x00, 0x19, 0x46, 0x23, 0xcd, 0x11, 0x05, 0xcd,
    0x18, 0x05, 0xed, 0xa3, 0x20, 0xf9, 0xdd, 0x46,
    0x0c, 0xdd, 0x5e, 0x0d, 0xcb, 0xbb, 0xdd, 0x56,
    0x03, 0x15, 0xdd, 0x6e, 0x0a, 0xdd, 0x66, 0x0b,
    0xe5, 0xfd, 0xe1, 0xdd, 0x6e, 0x05, 0xdd, 0x66,
    0x06, 0xc9, 0xcb, 0x7f, 0xc0, 0xdb, 0xfa, 0xcb,
    0x6f, 0x28, 0xf7, 0xd5, 0x11, 0x00, 0x00, 0xfd,
    0xe9, 0x21, 0xcf, 0x05, 0xd5, 0x16, 0x00, 0xcb,
    0x2b, 0x19, 0xd1, 0x7e, 0xd3, 0xf7, 0xc9, 0x79,
    0x07, 0x5f, 0x16, 0x00, 0x21, 0x14, 0x06, 0x19,
    0x5e, 0x23, 0x56, 0xc5, 0xd5, 0x11, 0x88, 0x06,
    0xcd, 0xba, 0x01, 0x3e, 0x41, 0xdd, 0x86, 0x00,
    0xcd, 0xc3, 0x01, 0x11, 0x7b, 0x00, 0xcd, 0xba,
    0x01, 0xd1, 0xcd, 0xba, 0x01, 0xcd, 0x52, 0x03,
    0xc1, 0xc9, 0xcd, 0xba, 0x01, 0xcd, 0x78, 0x03,
    0xfe, 0x0d, 0x20, 0xf9, 0x11, 0x78, 0x00, 0xc3,
    0xba, 0x01, 0x11, 0x9f, 0x06, 0xcd, 0xba, 0x01,
    0xcd, 0x78, 0x03, 0xe6, 0xdf, 0x5f, 0xfe, 0x41,
    0x37, 0x28, 0x08, 0xfe, 0x52, 0x28, 0x04, 0xd6,
    0x49, 0x20, 0xed, 0xf5, 0x7b, 0xcd, 0xc3, 0x01,
    0x11, 0x78, 0x00, 0xcd, 0xba, 0x01, 0xf1, 0xc9,
    0xe5, 0xfd, 0x6e, 0x18, 0xfd, 0x66, 0x19, 0xcd,
    0xd2, 0x01, 0xe1, 0xc9, 0xe6, 0x20, 0x28, 0x36,
    0xdb, 0xfa, 0xfe, 0xf0, 0x28, 0x0b, 0x1b, 0x7b,
    0xb2, 0x28, 0x2d, 0xdb, 0xfa, 0xfe, 0xf0, 0x20,
    0xeb, 0xed, 0xa2, 0xd1, 0xd5, 0x7b, 0xb2, 0x20,
    0x01, 0x05, 0xfb, 0x76, 0xdb, 0xfa, 0xe6, 0x20,
    0x28, 0x14, 0xed, 0xa2, 0x20, 0xf4, 0x1d, 0xf2,
    0x9d, 0x03, 0xd1, 0x15, 0xd5, 0xf2, 0x9d, 0x03,
    0xfb, 0x76, 0xdb, 0xf8, 0xed, 0xa2, 0xd1, 0xc9,
    0x3e, 0x08, 0xd3, 0xf7, 0xdb, 0xf5, 0xdb, 0xfa,
    0xcb, 0x6f, 0x20, 0xf8, 0x18, 0xf0, 0xe6, 0x20,
    0x28, 0xec, 0xdb, 0xfa, 0xfe, 0xb0, 0x28, 0x0b,
    0x1b, 0x7b, 0xb2, 0x28, 0xe3, 0xdb, 0xfa, 0xfe,
    0xb0, 0x20, 0xeb, 0xed, 0xa3, 0xd1, 0xd5, 0x7b,
    0xb2, 0x20, 0x01, 0x05, 0xfb, 0x76, 0xdb, 0xfa,
    0xe6, 0x20, 0x28, 0xca, 0xed, 0xa3, 0x20, 0xf4,
    0x1d, 0xf2, 0xe7, 0x03, 0xd1, 0x15, 0xd5, 0xf2,
    0xe7, 0x03, 0xfb, 0x76, 0xdb, 0xf8, 0xed, 0xa3,
    0xd1, 0xc9, 0x1d, 0x15, 0xc0, 0x06, 0x01, 0xc9,
    0xfd, 0xcb, 0x0b, 0x4e, 0x20, 0x04, 0xfd, 0xcb,
    0x0b, 0x86, 0xdd, 0x7e, 0x00, 0xf5, 0xfd, 0x7e,
    0x00, 0xdd, 0x77, 0x00, 0xcd, 0x33, 0x04, 0xf1,
    0xdd, 0x77, 0x00, 0xc9, 0x1e, 0x00, 0x18, 0x03,
    0xcd, 0x0a, 0x04, 0x3e, 0x00, 0xc5, 0xd5, 0x4f,
    0xdd, 0x7e, 0x00, 0xf5, 0xcd, 0xb8, 0x04, 0x7e,
    0xe6, 0x07, 0x5f, 0x23, 0xf1, 0xfd, 0xbe, 0x11,
    0x28, 0x1b, 0xfd, 0x77, 0x11, 0xcb, 0x7e, 0xc5,
    0x28, 0x0f, 0xfd, 0xbe, 0x13, 0xfd, 0x77, 0x13,
    0xf5, 0xd5, 0xe5, 0xc4, 0xa5, 0x04, 0xe1, 0xd1,
    0xf1, 0xcd, 0x4d, 0x05, 0xc1, 0xcb, 0x76, 0x28,
    0x0f, 0x3e, 0x13, 0xcb, 0x6e, 0x28, 0x02, 0x3e,
    0x27, 0xdd, 0xbe, 0x08, 0x30, 0x02, 0xcb, 0xdb,
    0x3e, 0x07, 0xd3, 0xf7, 0xdb, 0xf5, 0xe6, 0x07,
    0x57, 0xa3, 0xf5, 0x7a, 0xb3, 0xd3, 0xf8, 0xfd,
    0x77, 0x12, 0xcd, 0xc9, 0x04, 0xdd, 0x7e, 0x08,
    0xcb, 0x59, 0x28, 0x03, 0xba, 0x28, 0x06, 0xcd,
    0x2a, 0x05, 0xf1, 0x37, 0xf5, 0xf1, 0x23, 0xcd,
    0xd1, 0x04, 0xd1, 0xc1, 0xc9, 0x3c, 0xcb, 0xf7,
    0xfd, 0x5e, 0x0f, 0xfd, 0x56, 0x10, 0x12, 0xfd,
    0x5e, 0x0d, 0xfd, 0x56, 0x0e, 0xc3, 0x42, 0x03,
    0xfd, 0xe5, 0xe1, 0x11, 0x1a, 0x00, 0x19, 0x57,
    0x07, 0x07, 0x07, 0x82, 0x5f, 0x16, 0x00, 0x19,
    0xc9, 0xe5, 0x11, 0x07, 0x00, 0x19, 0x56, 0xe1,
    0xc9, 0xfd, 0x46, 0x12, 0x0e, 0xf8, 0x56, 0x1e,
    0x00, 0x23, 0x30, 0x28, 0xed, 0x5e, 0x3e, 0x00,
    0xed, 0x47, 0x20, 0x0c, 0xfb, 0x1b, 0x7a, 0xb3,
    0x28, 0x08, 0x0e, 0xf8, 0xed, 0x41, 0x18, 0xf5,
    0x56, 0xfb, 0xed, 0x41, 0x18, 0xfc, 0xed, 0x46,
    0x33, 0x33, 0xcd, 0x76, 0x05, 0x7a, 0xbe, 0x30,
    0x04, 0x56, 0x18, 0x01, 0xc0, 0xed, 0x41, 0xed,
    0x41, 0x00, 0x1b, 0x7a, 0xb3, 0xc2, 0x05, 0x05,
    0xc9, 0xdb, 0xfa, 0xcb, 0x67, 0x20, 0xfa, 0xc9,
    0xdb, 0xfa, 0xe6, 0xc0, 0xfe, 0x80, 0x20, 0xf8,
    0xc9, 0xdb, 0xfa, 0xe6, 0xc0, 0xfe, 0xc0, 0x20,
    0xf8, 0xc9, 0xcd, 0x18, 0x05, 0x3e, 0x07, 0xb1,
    0xd3, 0xfb, 0xcd, 0x18, 0x05, 0x3e, 0x03, 0xa6,
    0xcb, 0x59, 0x28, 0x03, 0xdd, 0xb6, 0x09, 0xd3,
    0xfb, 0xcb, 0x59, 0xc8, 0xcd, 0x18, 0x05, 0xdd,
    0x7e, 0x08, 0xd3, 0xfb, 0xc9, 0xe5, 0x23, 0x23,
    0x23, 0xcd, 0x18, 0x05, 0x3e, 0x03, 0xd3, 0xfb,
    0x01, 0xfb, 0x02, 0xcd, 0x18, 0x05, 0xed, 0xa3,
    0x20, 0xf9, 0xe1, 0xc9, 0xcd, 0x18, 0x05, 0x3e,
    0x08, 0xd3, 0xfb, 0xcd, 0x21, 0x05, 0xdb, 0xfb,
    0xcd, 0x21, 0x05, 0xdb, 0xfb, 0xc9, 0xcd, 0x64,
    0x05, 0xe5, 0xd5, 0x11, 0x05, 0x00, 0x19, 0x77,
    0xd1, 0xe1, 0xc9, 0x0e, 0x01, 0xdd, 0xcb, 0x0f,
    0x4e, 0xc0, 0x0c, 0xdd, 0xcb, 0x10, 0x4e, 0xc0,
    0xdd, 0xcb, 0x10, 0x66, 0xc0, 0x0c, 0xdd, 0xcb,
    0x10, 0x6e, 0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x6e,
    0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x7e, 0xc0, 0xdd,
    0xcb, 0x0f, 0x56, 0xc0, 0xdd, 0xcb, 0x0f, 0x46,
    0xc0, 0xdd, 0xcb, 0x10, 0x76, 0xc0, 0xdd, 0xcb,
    0x10, 0x46, 0xc0, 0x0c, 0xdd, 0xcb, 0x0e, 0x5e,
    0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x66, 0xc0, 0x0c,
    0xdd, 0xcb, 0x0e, 0x66, 0xc0, 0x0c, 0xc9, 0x08,
    0x09, 0x07, 0x0a, 0x06, 0x00, 0x00, 0x01, 0x01,
    0x14, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x93, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x09, 0x46, 0x00, 0x00, 0x00, 0x01, 0x03,
    0x05, 0x1c, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
    0x00, 0x9c, 0x01, 0xa6, 0x01, 0x01, 0xd8, 0x58,
    0x05, 0x6f, 0x03, 0x05, 0x1c, 0xff, 0x35, 0x06,
    0x28, 0x06, 0x46, 0x06, 0x46, 0x06, 0x52, 0x06,
    0x68, 0x06, 0x5d, 0x06, 0x68, 0x06, 0x79, 0x06,
    0x57, 0x72, 0x6f, 0x6e, 0x67, 0x20, 0x74, 0x72,
    0x61, 0x63, 0x6b, 0xae, 0x00, 0x57, 0x72, 0x69,
    0x74, 0x65, 0x20, 0x70, 0x72, 0x6f, 0x74, 0x65,
    0x63, 0x74, 0x65, 0x64, 0xae, 0x00, 0x44, 0x61,
    0x74, 0x61, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72,
    0xae, 0x00, 0x4e, 0x6f, 0x74, 0x20, 0x66, 0x6f,
    0x75, 0x6e, 0x64, 0xae, 0x00, 0x4c, 0x6f, 0x73,
    0x74, 0x20, 0x64, 0x61, 0x74, 0x61, 0xae, 0x00,
    0x44, 0x72, 0x69, 0x76, 0x65, 0x20, 0x6e, 0x6f,
    0x74, 0x20, 0x72, 0x65, 0x61, 0x64, 0x79, 0xae,
    0x00, 0x55, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e,
    0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0xae, 0x00,
    0x0d, 0x0a, 0x44, 0x69, 0x73, 0x6b, 0x20, 0x65,
    0x72, 0x72, 0x6f, 0x72, 0x20, 0x6f, 0x6e, 0x20,
    0x64, 0x72, 0x69, 0x76, 0x65, 0xa0, 0x00, 0x0d,
    0x0a, 0x54, 0x79, 0x70, 0x65, 0x20, 0x52, 0x20,
    0x74, 0x6f, 0x20, 0x74, 0x72, 0x79, 0x20, 0x61,
    0x67, 0x61, 0x69, 0x6e, 0x2c, 0x20, 0x41, 0x20,
    0x74, 0x6f, 0x20, 0x61, 0x62, 0x6f, 0x72, 0x74,
    0x2c, 0x20, 0x6f, 0x72, 0x20, 0x49, 0x20, 0x74,
    0x6f, 0x20, 0x69, 0x67, 0x6e, 0x6f, 0x72, 0x65,
    0x3a, 0xa0, 0x00, 0x0a, 0x45, 0x72, 0x72, 0x6f,
    0x72, 0x20, 0x6f, 0x6e, 0x20, 0x43, 0x50, 0x2f,
    0x4d, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
    0x20, 0x64, 0x69, 0x73, 0x6b, 0x65, 0x74, 0x74,
    0x65, 0x2e, 0x00, 0x0d, 0x0a, 0x50, 0x75, 0x73,
    0x68, 0x20, 0x72, 0x65, 0x73, 0x65, 0x74, 0x20,
    0x74, 0x6f, 0x20, 0x74, 0x72, 0x79, 0x20, 0x61,
    0x67, 0x61, 0x69, 0x6e, 0x2e, 0xa0, 0x00, 0x0d,
    0x4d, 0x69, 0x63, 0x72, 0x6f, 0x2d, 0x44, 0x65,
    0x63, 0x69, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x2d,
    0x2d, 0x20, 0x52, 0x4f, 0x4d, 0x20, 0x52, 0x65,
    0x76, 0x2e, 0x20, 0x31, 0x2e, 0x33, 0x0d, 0x0a,
    0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
    0x74, 0x20, 0x31, 0x39, 0x38, 0x32, 0x20, 0x4d,
    0x6f, 0x72, 0x72, 0x6f, 0x77, 0x20, 0x44, 0x65,
    0x73, 0x69, 0x67, 0x6e, 0x73, 0x2c, 0x20, 0x49,
    0x6e, 0x63, 0x2e, 0x0d, 0x0a, 0x0a, 0x00, 0x54,
    0x65, 0x73, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x6d,
    0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20, 0x2d, 0x2d,
    0x20, 0x70, 0x6c, 0x65, 0x61, 0x73, 0x65, 0x20,
    0x73, 0x74, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x79,
    0x3a, 0x20, 0x00, 0x4d, 0x65, 0x6d, 0x6f, 0x72,
    0x79, 0x20, 0x4f, 0x2e, 0x4b, 0x2e, 0x0d, 0x0a,
    0x0a, 0x00, 0x49, 0x6e, 0x73, 0x65, 0x72, 0x74,
    0x20, 0x43, 0x50, 0x2f, 0x4d, 0x20, 0x73, 0x79,
    0x73, 0x74, 0x65, 0x6d, 0x20, 0x64, 0x69, 0x73,
    0x6b, 0x65, 0x74, 0x74, 0x65, 0x20, 0x69, 0x6e,
    0x20, 0x44, 0x72, 0x69, 0x76, 0x65, 0x20, 0x41,
    0x20, 0x61, 0x6e, 0x64, 0x20, 0x70, 0x72, 0x65,
    0x73, 0x73, 0x20, 0x5b, 0x52, 0x45, 0x54, 0x55,
    0x52, 0x4e, 0x5d, 0x20, 0x07, 0x00, 0x07, 0x07,
    0x52, 0x41, 0x4d, 0x20, 0x4d, 0x65, 0x6d, 0x6f,
    0x72, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72,
    0x00, 0x07, 0x07, 0x52, 0x4f, 0x4d, 0x20, 0x4d,
    0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20, 0x65, 0x72,
    0x72, 0x6f, 0x72, 0x00, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x13, 0xff, 0xff, 0xff,
    };

/* MD3 2.3 ROM is 4096 bytes */
static uint8 mmd_rom_23[MMD_ROM_SIZE] = {
    0xc3, 0x26, 0x00, 0xc3, 0xc3, 0x01, 0xc3, 0x5d,
    0x05, 0xc3, 0x01, 0x03, 0xc3, 0x0e, 0x03, 0xc3,
    0x92, 0x03, 0xc3, 0xd3, 0x04, 0xc3, 0x2a, 0x05,
    0xc3, 0x57, 0x01, 0xb1, 0x0e, 0xc3, 0x98, 0x01,
    0xc3, 0xde, 0x01, 0xc3, 0xf1, 0x01, 0xed, 0x46,
    0x31, 0x00, 0x00, 0xfd, 0x21, 0xd0, 0x0e, 0xcd,
    0x85, 0x00, 0xcd, 0x53, 0x00, 0x11, 0x37, 0x0c,
    0xcd, 0xba, 0x01, 0xcd, 0xaa, 0x00, 0xcd, 0xe9,
    0x00, 0xcd, 0xca, 0x00, 0xcd, 0x00, 0xfc, 0xdb,
    0xfc, 0xdb, 0xf5, 0xcb, 0x6f, 0xcc, 0x01, 0x08,
    0xc3, 0x03, 0x01, 0x3e, 0x80, 0xd3, 0xfd, 0xd3,
    0xff, 0xd3, 0xfd, 0xd3, 0xff, 0x3e, 0x40, 0xd3,
    0xfd, 0xd3, 0xff, 0x3e, 0xce, 0xd3, 0xfd, 0xd3,
    0xff, 0x3e, 0x37, 0xd3, 0xfd, 0xd3, 0xff, 0xc9,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x72,
    0x06, 0x91, 0x9e, 0x3f, 0x0a, 0x3e, 0x3e, 0xd3,
    0xf3, 0x3e, 0xff, 0xd3, 0xf0, 0x3e, 0xff, 0xd3,
    0xf0, 0x3e, 0x7e, 0xd3, 0xf3, 0x3e, 0x0d, 0xd3,
    0xf1, 0x3e, 0x00, 0xd3, 0xf1, 0x3e, 0xbe, 0xd3,
    0xf3, 0x3e, 0x68, 0xd3, 0xf2, 0x3e, 0x00, 0xd3,
    0xf2, 0xc9, 0x21, 0x00, 0x00, 0x01, 0xfd, 0x0f,
    0xcd, 0xdc, 0x00, 0xc8, 0x11, 0xf4, 0x0c, 0xcd,
    0xc3, 0x01, 0x18, 0x00, 0xdb, 0xf5, 0xcb, 0x6f,
    0xca, 0x47, 0x00, 0x11, 0x1b, 0x0c, 0xcd, 0xc3,
    0x01, 0x76, 0x21, 0x00, 0xfc, 0x01, 0xf8, 0x03,
    0xcd, 0xdc, 0x00, 0xc8, 0x11, 0x07, 0x0d, 0xcd,
    0xc3, 0x01, 0x18, 0xe0, 0xaf, 0xf5, 0xf1, 0xae,
    0x23, 0x0b, 0xf5, 0x78, 0xb1, 0x20, 0xf7, 0xf1,
    0xc9, 0x21, 0x3f, 0x0a, 0x11, 0x00, 0xfc, 0x01,
    0x00, 0x01, 0xed, 0xb0, 0x01, 0xf7, 0x02, 0xd5,
    0xe1, 0x13, 0x36, 0xff, 0xed, 0xb0, 0x3a, 0x82,
    0x00, 0x77, 0xc9, 0xcd, 0x5d, 0x01, 0x3a, 0xea,
    0x0e, 0xe6, 0x03, 0xcb, 0x57, 0x28, 0x01, 0x0f,
    0x4f, 0x3a, 0xeb, 0x0e, 0xe6, 0x03, 0x47, 0x04,
    0x3e, 0x08, 0x07, 0x10, 0xfd, 0xb1, 0x11, 0x00,
    0x40, 0xd3, 0xf7, 0x1b, 0x7a, 0xb3, 0x28, 0x13,
    0xdb, 0xf5, 0xcb, 0x57, 0x28, 0xf5, 0x1b, 0x7a,
    0xb3, 0x28, 0x08, 0xdb, 0xf5, 0xcb, 0x57, 0x20,
    0xf5, 0x18, 0x0f, 0x11, 0xb8, 0x0c, 0xcd, 0xc3,
    0x01, 0xcd, 0xb6, 0x04, 0x11, 0xab, 0x0e, 0xcd,
    0xc3, 0x01, 0xcd, 0x70, 0x05, 0xcd, 0x92, 0x03,
    0xdd, 0x7e, 0x07, 0xb7, 0xca, 0x00, 0xfe, 0x11,
    0xfb, 0x0b, 0xc3, 0xbc, 0x00, 0x21, 0xb1, 0x0e,
    0x11, 0x00, 0xff, 0x01, 0x5d, 0x00, 0xed, 0xb0,
    0xdd, 0x21, 0x00, 0xff, 0xfd, 0x21, 0x1f, 0xff,
    0xc9, 0x06, 0xff, 0x18, 0x02, 0x06, 0xfd, 0x78,
    0x41, 0x4f, 0xed, 0x78, 0xcb, 0x47, 0x28, 0xfa,
    0x78, 0x0d, 0xed, 0x79, 0xc9, 0x06, 0xff, 0x18,
    0x02, 0x06, 0xfd, 0x48, 0xed, 0x78, 0xcb, 0x4f,
    0x28, 0xfa, 0x0d, 0xed, 0x78, 0xe6, 0x7f, 0xc9,
    0xdb, 0xf5, 0xcb, 0x67, 0x20, 0xfa, 0x79, 0xd3,
    0xf4, 0x3e, 0x80, 0xd3, 0xf5, 0xaf, 0xd3, 0xf5,
    0x21, 0x40, 0x00, 0xdb, 0xf5, 0xcb, 0x5f, 0x28,
    0x07, 0x2b, 0x7d, 0xb4, 0x20, 0xf5, 0x3d, 0xc9,
    0xaf, 0xc9, 0x06, 0x32, 0x3e, 0x0a, 0xcd, 0xcc,
    0x01, 0x10, 0xf9, 0x1a, 0xb7, 0xc8, 0xcd, 0xcc,
    0x01, 0x13, 0x18, 0xf7, 0x4f, 0xc5, 0xd5, 0xe5,
    0xfd, 0x6e, 0x16, 0xfd, 0x66, 0x17, 0xcd, 0xdd,
    0x01, 0xe1, 0xd1, 0xc1, 0xc9, 0xe9, 0xfd, 0x36,
    0x04, 0x00, 0xfd, 0xcb, 0x0b, 0xde, 0xfd, 0xcb,
    0x0b, 0xd6, 0xfd, 0x36, 0x0a, 0x02, 0xc3, 0x56,
    0x02, 0xfd, 0xcb, 0x0b, 0x9e, 0xfd, 0x71, 0x0a,
    0x79, 0xfe, 0x02, 0x20, 0x18, 0xfd, 0x7e, 0x08,
    0xfd, 0x77, 0x04, 0xfd, 0x7e, 0x00, 0xfd, 0x77,
    0x05, 0xfd, 0x7e, 0x01, 0xfd, 0x77, 0x06, 0xfd,
    0x7e, 0x02, 0xfd, 0x77, 0x07, 0xfd, 0x7e, 0x04,
    0xb7, 0x28, 0x33, 0xfd, 0x35, 0x04, 0xfd, 0x7e,
    0x00, 0xfd, 0xbe, 0x05, 0x20, 0x28, 0xfd, 0x7e,
    0x01, 0xfd, 0xbe, 0x06, 0x20, 0x20, 0xfd, 0x7e,
    0x02, 0xfd, 0xbe, 0x07, 0x20, 0x18, 0xfd, 0x34,
    0x07, 0xfd, 0x7e, 0x09, 0xfd, 0xbe, 0x07, 0x30,
    0x07, 0xfd, 0x34, 0x06, 0xfd, 0x36, 0x07, 0x01,
    0xfd, 0xcb, 0x0b, 0x96, 0x18, 0x08, 0xfd, 0x36,
    0x04, 0x00, 0xfd, 0xcb, 0x0b, 0xd6, 0xdd, 0x36,
    0x07, 0x00, 0xfd, 0x7e, 0x00, 0xcd, 0x33, 0x06,
    0x23, 0x7e, 0xfd, 0x5e, 0x02, 0x1d, 0x53, 0xe6,
    0x18, 0x06, 0x00, 0xfe, 0x00, 0x28, 0x14, 0x06,
    0x01, 0xfe, 0x08, 0x28, 0x0c, 0x06, 0x03, 0xfe,
    0x10, 0x28, 0x04, 0x06, 0x07, 0xcb, 0x2b, 0xcb,
    0x2b, 0xcb, 0x2b, 0x1c, 0xfd, 0x73, 0x03, 0x78,
    0xa2, 0xfd, 0x77, 0x0c, 0xfd, 0xcb, 0x0b, 0x46,
    0xfd, 0xcb, 0x0b, 0xc6, 0x28, 0x22, 0xfd, 0x7e,
    0x00, 0xdd, 0xbe, 0x00, 0x20, 0x10, 0xfd, 0x7e,
    0x01, 0xdd, 0xbe, 0x01, 0x20, 0x08, 0xfd, 0x7e,
    0x03, 0xdd, 0xbe, 0x02, 0x28, 0x2a, 0xaf, 0xfd,
    0xcb, 0x0b, 0x4e, 0xc4, 0x0e, 0x03, 0xb7, 0xc0,
    0xfd, 0x7e, 0x00, 0xdd, 0x77, 0x00, 0xfd, 0x7e,
    0x01, 0xdd, 0x77, 0x01, 0xfd, 0x7e, 0x03, 0xdd,
    0x77, 0x02, 0xaf, 0xfd, 0xcb, 0x0b, 0x56, 0xc4,
    0x01, 0x03, 0xb7, 0xc0, 0xfd, 0xcb, 0x0b, 0x8e,
    0xfd, 0x6e, 0x0c, 0x26, 0x00, 0x29, 0x29, 0x29,
    0x29, 0x29, 0x29, 0x29, 0xdd, 0x5e, 0x05, 0xdd,
    0x56, 0x06, 0x19, 0xfd, 0x5e, 0x14, 0xfd, 0x56,
    0x15, 0x01, 0x80, 0x00, 0xfd, 0xcb, 0x0b, 0x5e,
    0x3e, 0x00, 0xc0, 0xfd, 0xcb, 0x0b, 0xce, 0xeb,
    0xc9, 0x21, 0xd3, 0x04, 0xdd, 0x75, 0x0a, 0xdd,
    0x74, 0x0b, 0x0e, 0x06, 0x18, 0x0b, 0x21, 0x2a,
    0x05, 0xdd, 0x75, 0x0a, 0xdd, 0x74, 0x0b, 0x0e,
    0x05, 0xdd, 0x7e, 0x00, 0xcd, 0x33, 0x06, 0x23,
    0x7e, 0xe6, 0x40, 0xb1, 0xdd, 0x77, 0x16, 0xdd,
    0x36, 0x15, 0x09, 0x3e, 0x18, 0xa6, 0x0f, 0x0f,
    0x0f, 0xdd, 0x77, 0x1b, 0x06, 0x80, 0x11, 0x80,
    0x00, 0xfe, 0x00, 0x28, 0x11, 0x06, 0xff, 0x11,
    0x00, 0x00, 0xfe, 0x01, 0x28, 0x08, 0x16, 0x01,
    0xfe, 0x02, 0x28, 0x02, 0x16, 0x03, 0xdd, 0x70,
    0x1e, 0xdd, 0xcb, 0x0d, 0x7e, 0x20, 0x06, 0xdd,
    0x73, 0x0c, 0xdd, 0x72, 0x0d, 0xaf, 0xcb, 0x56,
    0xdd, 0x46, 0x01, 0xdd, 0x4e, 0x02, 0x28, 0x03,
    0xcb, 0x18, 0x17, 0xdd, 0x77, 0x19, 0x07, 0x07,
    0xdd, 0x77, 0x09, 0xdd, 0x70, 0x08, 0xdd, 0x70,
    0x18, 0xdd, 0x71, 0x1a, 0x3e, 0x03, 0xa6, 0xdd,
    0xb6, 0x09, 0xdd, 0x77, 0x17, 0x11, 0x05, 0x00,
    0x19, 0x7e, 0xdd, 0x77, 0x1c, 0x23, 0x7e, 0xdd,
    0x77, 0x1d, 0xdd, 0x46, 0x04, 0x16, 0x03, 0xc5,
    0xd5, 0xcd, 0x74, 0x05, 0xfd, 0xe5, 0xcd, 0x0f,
    0x04, 0xcd, 0x43, 0x04, 0xfd, 0xe1, 0x06, 0x07,
    0xdd, 0xe5, 0xe1, 0x11, 0x0e, 0x00, 0x19, 0xcd,
    0xa6, 0x06, 0xed, 0xa2, 0x20, 0xf9, 0xd1, 0xc1,
    0xdd, 0x7e, 0x0e, 0xe6, 0xc0, 0x28, 0x35, 0xcd,
    0x0d, 0x07, 0x3e, 0x01, 0xb9, 0x28, 0x1d, 0x3e,
    0x06, 0xb9, 0xcc, 0x57, 0x05, 0x3e, 0x02, 0xb9,
    0xcc, 0x6c, 0x05, 0x3e, 0x05, 0xb9, 0xcc, 0xfa,
    0x03, 0xdd, 0x7e, 0x04, 0x0f, 0x3c, 0xb8, 0xcc,
    0x70, 0x05, 0x10, 0xb3, 0xdd, 0x7e, 0x07, 0xfe,
    0xfe, 0x28, 0x0a, 0xcd, 0x4f, 0x04, 0x38, 0x05,
    0xb7, 0xc2, 0x92, 0x03, 0x4f, 0xdd, 0x71, 0x07,
    0x79, 0xc9, 0xdd, 0x7e, 0x08, 0xfe, 0x03, 0xd0,
    0xf5, 0xdd, 0x36, 0x08, 0x03, 0xcd, 0x74, 0x05,
    0xf1, 0xdd, 0x77, 0x08, 0xc3, 0x6c, 0x05, 0x0e,
    0xfb, 0xdd, 0xe5, 0xe1, 0x11, 0x15, 0x00, 0x19,
    0x46, 0x23, 0xcd, 0x90, 0x06, 0xcd, 0x9d, 0x06,
    0xed, 0xa3, 0x20, 0xf9, 0xdd, 0x46, 0x0c, 0xdd,
    0x5e, 0x0d, 0xcb, 0xbb, 0xdd, 0x56, 0x03, 0x15,
    0xdd, 0x6e, 0x0a, 0xdd, 0x66, 0x0b, 0xe5, 0xfd,
    0xe1, 0xdd, 0x6e, 0x05, 0xdd, 0x66, 0x06, 0xc9,
    0xcb, 0x7f, 0xc0, 0xdb, 0xfa, 0xcb, 0x6f, 0x28,
    0xf7, 0xd5, 0x11, 0x00, 0x00, 0xfd, 0xe9, 0x79,
    0x07, 0x5f, 0x16, 0x00, 0x21, 0x0c, 0x0f, 0x19,
    0x5e, 0x23, 0x56, 0xc5, 0xd5, 0x11, 0xb0, 0x0b,
    0xcd, 0xc3, 0x01, 0x3e, 0x41, 0xdd, 0x86, 0x00,
    0xcd, 0xcc, 0x01, 0x11, 0xae, 0x0e, 0xcd, 0xc3,
    0x01, 0xd1, 0xcd, 0xc3, 0x01, 0x3e, 0xff, 0xdd,
    0xbe, 0x07, 0x37, 0xc4, 0x90, 0x04, 0xc1, 0xc9,
    0xcd, 0xc3, 0x01, 0xcd, 0xb6, 0x04, 0xfe, 0x0d,
    0x20, 0xf9, 0x11, 0xab, 0x0e, 0xc3, 0xc3, 0x01,
    0x11, 0xc7, 0x0b, 0xcd, 0xc3, 0x01, 0xcd, 0xb6,
    0x04, 0xe6, 0xdf, 0x5f, 0xfe, 0x41, 0x37, 0x28,
    0x08, 0xfe, 0x52, 0x28, 0x04, 0xd6, 0x49, 0x20,
    0xed, 0xf5, 0x7b, 0xcd, 0xcc, 0x01, 0x11, 0xab,
    0x0e, 0xcd, 0xc3, 0x01, 0xf1, 0xc9, 0xc5, 0xe5,
    0xfd, 0x6e, 0x18, 0xfd, 0x66, 0x19, 0xcd, 0xdd,
    0x01, 0xe1, 0xc1, 0xc9, 0xe6, 0x20, 0x28, 0x36,
    0xdb, 0xfa, 0xfe, 0xf0, 0x28, 0x0b, 0x1b, 0x7b,
    0xb2, 0x28, 0x2d, 0xdb, 0xfa, 0xfe, 0xf0, 0x20,
    0xeb, 0xed, 0xa2, 0xd1, 0xd5, 0x7b, 0xb2, 0x20,
    0x01, 0x05, 0xfb, 0x76, 0xdb, 0xfa, 0xe6, 0x20,
    0x28, 0x14, 0xed, 0xa2, 0x20, 0xf4, 0x1d, 0xf2,
    0xdd, 0x04, 0xd1, 0x15, 0xd5, 0xf2, 0xdd, 0x04,
    0xfb, 0x76, 0xdb, 0xf7, 0xed, 0xa2, 0xd1, 0xc9,
    0xd1, 0xe1, 0xfd, 0xe1, 0xfd, 0xe5, 0xc5, 0x0e,
    0xf7, 0xfd, 0x7e, 0x12, 0x47, 0xcb, 0xdf, 0xed,
    0x79, 0xdb, 0xfa, 0xcb, 0x6f, 0x20, 0xfa, 0xed,
    0x41, 0xc1, 0xe9, 0xe6, 0x20, 0x28, 0xdf, 0xdb,
    0xfa, 0xfe, 0xb0, 0x28, 0x0b, 0x1b, 0x7b, 0xb2,
    0x28, 0xd6, 0xdb, 0xfa, 0xfe, 0xb0, 0x20, 0xeb,
    0xed, 0xa3, 0xd1, 0xd5, 0x7b, 0xb2, 0x20, 0x01,
    0x05, 0xfb, 0x76, 0xdb, 0xfa, 0xe6, 0x20, 0x28,
    0xbd, 0xed, 0xa3, 0x20, 0xf4, 0x1d, 0xf2, 0x34,
    0x05, 0xd1, 0x15, 0xd5, 0xf2, 0x34, 0x05, 0xfb,
    0x76, 0xdb, 0xf7, 0xed, 0xa3, 0xd1, 0xc9, 0x04,
    0x15, 0xc0, 0x06, 0x01, 0xc9, 0xfd, 0xcb, 0x0b,
    0x4e, 0x20, 0x04, 0xfd, 0xcb, 0x0b, 0x86, 0xfd,
    0x36, 0x01, 0x00, 0xc9, 0xcd, 0x57, 0x05, 0xc8,
    0x3e, 0x00, 0x18, 0x02, 0x3e, 0x08, 0xc5, 0xd5,
    0x4f, 0xdd, 0x7e, 0x00, 0xf5, 0xcd, 0x33, 0x06,
    0x7e, 0xe6, 0x07, 0xcb, 0x57, 0x28, 0x01, 0x0f,
    0x5f, 0x23, 0xf1, 0xfd, 0xbe, 0x11, 0x28, 0x1b,
    0xfd, 0x77, 0x11, 0xcb, 0x7e, 0xc5, 0x28, 0x0f,
    0xfd, 0xbe, 0x13, 0xfd, 0x77, 0x13, 0xf5, 0xd5,
    0xe5, 0xc4, 0x20, 0x06, 0xe1, 0xd1, 0xf1, 0xcd,
    0xd2, 0x06, 0xc1, 0xcb, 0x76, 0x28, 0x0f, 0x3e,
    0x13, 0xcb, 0x6e, 0x28, 0x02, 0x3e, 0x27, 0xdd,
    0xbe, 0x08, 0x30, 0x02, 0xcb, 0xd3, 0xdb, 0xf5,
    0xe6, 0x03, 0x57, 0xa3, 0xf5, 0x7e, 0xe6, 0x03,
    0x47, 0x04, 0x3e, 0x08, 0x07, 0x10, 0xfd, 0xb2,
    0xb3, 0xd3, 0xf7, 0xfd, 0x77, 0x12, 0xcd, 0x44,
    0x06, 0xdd, 0x7e, 0x08, 0xcb, 0x59, 0x28, 0x0a,
    0xb7, 0x20, 0x04, 0xcb, 0x99, 0x18, 0x03, 0xba,
    0x28, 0x06, 0xcd, 0xaf, 0x06, 0xf1, 0x37, 0xf5,
    0xf1, 0xc5, 0xe5, 0xcd, 0x4c, 0x06, 0xe1, 0xc1,
    0xcb, 0x59, 0xcc, 0x00, 0x06, 0xd1, 0xc1, 0xc9,
    0xcd, 0x9d, 0x06, 0x3e, 0x04, 0xd3, 0xfb, 0xcd,
    0x9d, 0x06, 0x3e, 0x03, 0xa6, 0xd3, 0xfb, 0xcd,
    0xa6, 0x06, 0xdb, 0xfb, 0xe6, 0x10, 0xc0, 0xcd,
    0xaf, 0x06, 0x3e, 0xff, 0xb7, 0x37, 0x18, 0x2c,
    0x3c, 0xcb, 0xf7, 0xfd, 0x5e, 0x0f, 0xfd, 0x56,
    0x10, 0x12, 0xfd, 0x5e, 0x0d, 0xfd, 0x56, 0x0e,
    0xc3, 0x80, 0x04, 0xfd, 0xe5, 0xe1, 0x11, 0x1a,
    0x00, 0x19, 0x57, 0x07, 0x07, 0x07, 0x82, 0x5f,
    0x16, 0x00, 0x19, 0xc9, 0xe5, 0x11, 0x07, 0x00,
    0x19, 0x56, 0xe1, 0xc9, 0x23, 0xfd, 0x46, 0x12,
    0x0e, 0xf7, 0x56, 0x1e, 0x00, 0x23, 0x30, 0x28,
    0xed, 0x5e, 0x3e, 0x00, 0xed, 0x47, 0x20, 0x0c,
    0xfb, 0x1b, 0x7a, 0xb3, 0x28, 0x08, 0x0e, 0xf7,
    0xed, 0x41, 0x18, 0xf5, 0x56, 0xfb, 0xed, 0x41,
    0x18, 0xfc, 0xed, 0x46, 0x33, 0x33, 0xcd, 0x00,
    0x07, 0x7a, 0xbe, 0x30, 0x04, 0x56, 0x18, 0x01,
    0xc0, 0xed, 0x41, 0xed, 0x41, 0x00, 0x1b, 0x7a,
    0xb3, 0xc2, 0x81, 0x06, 0xc9, 0xcd, 0xe9, 0x06,
    0xdb, 0xfa, 0xe6, 0x0f, 0x20, 0xf7, 0xdb, 0xfa,
    0xcb, 0x67, 0x20, 0xf4, 0xc9, 0xdb, 0xfa, 0xe6,
    0xc0, 0xfe, 0x80, 0x20, 0xf8, 0xc9, 0xdb, 0xfa,
    0xe6, 0xc0, 0xfe, 0xc0, 0x20, 0xf8, 0xc9, 0xcd,
    0x9d, 0x06, 0x3e, 0x07, 0xb1, 0xd3, 0xfb, 0xcd,
    0x9d, 0x06, 0x3e, 0x03, 0xa6, 0xcb, 0x59, 0x28,
    0x03, 0xdd, 0xb6, 0x09, 0xd3, 0xfb, 0xcb, 0x59,
    0xc8, 0xcd, 0x9d, 0x06, 0xdd, 0x7e, 0x08, 0xd3,
    0xfb, 0xc9, 0xe5, 0x23, 0x23, 0x23, 0xcd, 0x9d,
    0x06, 0x3e, 0x03, 0xd3, 0xfb, 0x01, 0xfb, 0x02,
    0xcd, 0x9d, 0x06, 0xed, 0xa3, 0x20, 0xf9, 0xe1,
    0xc9, 0xcd, 0x9d, 0x06, 0x3e, 0x08, 0xd3, 0xfb,
    0xcd, 0xa6, 0x06, 0xdb, 0xfb, 0xe6, 0xc0, 0xfe,
    0x80, 0xc8, 0xcd, 0xa6, 0x06, 0xdb, 0xfb, 0xc9,
    0xcd, 0xe9, 0x06, 0xe5, 0xd5, 0x11, 0x05, 0x00,
    0x19, 0x77, 0xd1, 0xe1, 0xc9, 0x0e, 0x01, 0xdd,
    0xcb, 0x0f, 0x4e, 0xc0, 0x0c, 0xdd, 0xcb, 0x10,
    0x4e, 0xc0, 0xdd, 0xcb, 0x10, 0x66, 0xc0, 0x0c,
    0xdd, 0xcb, 0x10, 0x6e, 0xc0, 0x0c, 0xdd, 0xcb,
    0x0f, 0x6e, 0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x7e,
    0xc0, 0xdd, 0xcb, 0x0f, 0x56, 0xc0, 0xdd, 0xcb,
    0x0f, 0x46, 0xc0, 0xdd, 0xcb, 0x10, 0x76, 0xc0,
    0xdd, 0xcb, 0x10, 0x46, 0xc0, 0x0c, 0xdd, 0xcb,
    0x0e, 0x5e, 0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x66,
    0xc0, 0x0c, 0xdd, 0xcb, 0x0e, 0x66, 0xc0, 0x0c,
    0xdd, 0xcb, 0x0e, 0x7e, 0xc0, 0x0c, 0xc9, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x13, 0x22, 0x23, 0xff,
    0xff, 0x11, 0x41, 0x0d, 0xcd, 0xc3, 0x01, 0xcd,
    0xb6, 0x04, 0xfe, 0x31, 0xfa, 0x07, 0x08, 0xfe,
    0x3a, 0xf2, 0x07, 0x08, 0xf5, 0xcd, 0xcc, 0x01,
    0x11, 0xab, 0x0e, 0xcd, 0xc3, 0x01, 0xf1, 0xd6,
    0x30, 0xfe, 0x04, 0xf2, 0x29, 0x08, 0x32, 0xfe,
    0xff, 0x3d, 0x07, 0x4f, 0xaf, 0x47, 0x21, 0x3b,
    0x08, 0x09, 0x5e, 0x23, 0x56, 0xeb, 0xcd, 0xdd,
    0x01, 0x18, 0xc6, 0x4d, 0x08, 0x4d, 0x08, 0x4d,
    0x08, 0xa3, 0x08, 0xcd, 0x08, 0x21, 0x09, 0xc9,
    0x09, 0x1a, 0x0a, 0x3d, 0x0a, 0x3e, 0x20, 0x32,
    0xfd, 0xff, 0x3e, 0x5f, 0x47, 0x3a, 0xfd, 0xff,
    0xcd, 0x80, 0x08, 0x05, 0x28, 0x08, 0x3c, 0xf2,
    0x58, 0x08, 0x3e, 0x20, 0x18, 0xf2, 0xcd, 0x9e,
    0x08, 0xc0, 0x3e, 0x0d, 0xcd, 0x80, 0x08, 0x3e,
    0x0a, 0xcd, 0x80, 0x08, 0x3a, 0xfd, 0xff, 0x3c,
    0xfa, 0x4d, 0x08, 0x32, 0xfd, 0xff, 0x18, 0xd2,
    0xf5, 0xc5, 0x4f, 0x3a, 0xfe, 0xff, 0xfe, 0x01,
    0x20, 0x05, 0xcd, 0x75, 0x01, 0x18, 0x0c, 0xfe,
    0x02, 0x20, 0x05, 0xcd, 0x71, 0x01, 0x18, 0x03,
    0xcd, 0x98, 0x01, 0xc1, 0xf1, 0xc9, 0xdb, 0xfd,
    0xcb, 0x4f, 0xc9, 0xdb, 0xfe, 0xaf, 0x57, 0x4f,
    0xcd, 0x71, 0x01, 0xcd, 0x85, 0x01, 0xba, 0x20,
    0x0f, 0xcd, 0x9e, 0x08, 0x20, 0x05, 0x7a, 0x3c,
    0xf2, 0xa6, 0x08, 0x11, 0x49, 0x0e, 0x18, 0x03,
    0x11, 0x3f, 0x0e, 0xcd, 0xc3, 0x01, 0x11, 0x34,
    0x0e, 0xcd, 0xc3, 0x01, 0xc9, 0x11, 0x85, 0x0c,
    0xcd, 0xc3, 0x01, 0xcd, 0xee, 0x08, 0x21, 0x00,
    0x00, 0x01, 0xff, 0xfb, 0xcd, 0x2a, 0xfc, 0xb7,
    0x28, 0x03, 0xcd, 0xf6, 0x08, 0xcd, 0xfe, 0x08,
    0xcd, 0x9e, 0x08, 0x28, 0xe9, 0xc9, 0xaf, 0x32,
    0xfe, 0xff, 0x32, 0xfd, 0xff, 0xc9, 0x3a, 0xfd,
    0xff, 0x3c, 0x32, 0xfd, 0xff, 0xc9, 0x11, 0x15,
    0x0e, 0xcd, 0xc3, 0x01, 0x3a, 0xfe, 0xff, 0x3c,
    0x32, 0xfe, 0xff, 0xcd, 0xb3, 0xfc, 0x11, 0xae,
    0x0e, 0xcd, 0xc3, 0x01, 0x3a, 0xfd, 0xff, 0xcd,
    0xb3, 0xfc, 0x11, 0x24, 0x0e, 0xcd, 0xc3, 0x01,
    0xc9, 0xcd, 0x3e, 0x09, 0xdd, 0x36, 0x01, 0x27,
    0xdd, 0x36, 0x03, 0x05, 0xcd, 0x74, 0x09, 0xcd,
    0xee, 0x08, 0xcd, 0x9b, 0x09, 0xcd, 0xac, 0x09,
    0xcd, 0x9e, 0x08, 0x28, 0xf5, 0xc9, 0xcd, 0x5d,
    0x01, 0x11, 0x53, 0x0e, 0xcd, 0xc3, 0x01, 0xcd,
    0xb6, 0x04, 0xe6, 0xdf, 0xd6, 0x41, 0x38, 0xf7,
    0xfe, 0x04, 0x30, 0xf3, 0xdd, 0x77, 0x00, 0xf5,
    0x11, 0x6e, 0x0e, 0xcd, 0xc3, 0x01, 0xf1, 0xc6,
    0x41, 0xcd, 0xcc, 0x01, 0x11, 0xde, 0x0c, 0xcd,
    0xc3, 0x01, 0xcd, 0xb6, 0x04, 0x11, 0xab, 0x0e,
    0xcd, 0xc3, 0x01, 0xc9, 0x01, 0x5f, 0xaa, 0x21,
    0x00, 0x80, 0xe5, 0x11, 0x02, 0x80, 0x71, 0x23,
    0x70, 0x2b, 0x01, 0x00, 0x08, 0xed, 0xb0, 0x01,
    0xb6, 0x6d, 0x71, 0x23, 0x70, 0x2b, 0x01, 0x00,
    0x0c, 0xed, 0xb0, 0xc1, 0xdd, 0x71, 0x05, 0xdd,
    0x70, 0x06, 0xc9, 0x11, 0x96, 0x0e, 0xcd, 0xc3,
    0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0x0e, 0x03,
    0xcd, 0xc0, 0x09, 0xc9, 0x11, 0xa3, 0x0e, 0xcd,
    0xc3, 0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0x01,
    0x03, 0xcd, 0xc0, 0x09, 0xcd, 0xfe, 0x08, 0xc9,
    0xdd, 0x7e, 0x07, 0xfe, 0x00, 0xc4, 0xf6, 0x08,
    0xc9, 0xcd, 0x3e, 0x09, 0x06, 0x00, 0x0e, 0x27,
    0xdd, 0x36, 0x05, 0x00, 0xdd, 0x36, 0x06, 0x80,
    0xdd, 0x36, 0x03, 0x01, 0xcd, 0xee, 0x08, 0xc5,
    0xcd, 0xfd, 0x09, 0xc1, 0xcd, 0xed, 0x09, 0xcd,
    0x9e, 0x08, 0x28, 0xf3, 0xc9, 0x04, 0x0d, 0xfa,
    0xf3, 0x09, 0xc9, 0x78, 0x41, 0x4f, 0xc5, 0xcd,
    0xfe, 0x08, 0xc1, 0x18, 0xf0, 0xc5, 0xdd, 0x70,
    0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0x01, 0x03,
    0xcd, 0xc0, 0x09, 0xc1, 0xdd, 0x71, 0x01, 0xdd,
    0x36, 0x07, 0xff, 0xcd, 0x01, 0x03, 0xcd, 0xc0,
    0x09, 0xc9, 0xcd, 0x5d, 0x01, 0xaf, 0xdd, 0x77,
    0x00, 0xdd, 0x77, 0x01, 0xdd, 0x36, 0x03, 0x01,
    0xdd, 0x36, 0x05, 0x00, 0xdd, 0x36, 0x06, 0x80,
    0xdd, 0x36, 0x07, 0xfe, 0xcd, 0x01, 0x03, 0xcd,
    0x9e, 0x08, 0x28, 0xf4, 0xc9, 0xf1, 0xc9, 0xdb,
    0xf6, 0x3a, 0x00, 0x00, 0xfe, 0xc3, 0xd3, 0xf6,
    0x20, 0x03, 0xc3, 0x47, 0x00, 0x11, 0x85, 0x0c,
    0xcd, 0xc3, 0x01, 0x21, 0x00, 0x00, 0x01, 0xff,
    0xfb, 0xcd, 0x2a, 0xfc, 0xb7, 0xc2, 0xd4, 0x00,
    0x11, 0xa9, 0x0c, 0xcd, 0xc3, 0x01, 0xc3, 0x47,
    0x00, 0xdb, 0xf6, 0x22, 0x00, 0xfd, 0x21, 0xd2,
    0xfc, 0x56, 0x23, 0xcb, 0x7e, 0x20, 0x18, 0x5e,
    0xcb, 0x73, 0xcb, 0xb3, 0x23, 0xe5, 0x28, 0x05,
    0xcd, 0x54, 0xfc, 0x18, 0x07, 0xdd, 0x21, 0x74,
    0xfc, 0xcd, 0x58, 0xfc, 0xe1, 0x18, 0xe2, 0xaf,
    0xd3, 0xf6, 0xc9, 0xdd, 0x21, 0x76, 0xfc, 0x2a,
    0x00, 0xfd, 0x7b, 0x85, 0x6f, 0x7c, 0xce, 0x00,
    0x67, 0xcd, 0x72, 0xfc, 0x23, 0x23, 0x23, 0x78,
    0xbc, 0xd8, 0x20, 0xf5, 0x79, 0xbd, 0xd8, 0x18,
    0xf0, 0xdd, 0xe9, 0x72, 0xc9, 0x7e, 0xba, 0x20,
    0x01, 0xc9, 0xd3, 0xf6, 0xdb, 0xf5, 0xcb, 0x6f,
    0xcc, 0x8a, 0xfc, 0xf1, 0xf1, 0xf1, 0x3e, 0xff,
    0xc9, 0xd5, 0x11, 0x1a, 0x0d, 0xcd, 0xc3, 0x01,
    0xdb, 0xf6, 0x7e, 0xd3, 0xf6, 0xcd, 0xb3, 0xfc,
    0x11, 0x24, 0x0d, 0xcd, 0xc3, 0x01, 0xd1, 0x7a,
    0xcd, 0xb3, 0xfc, 0x11, 0x2f, 0x0d, 0xcd, 0xc3,
    0x01, 0x7c, 0xcd, 0xb3, 0xfc, 0x7d, 0xcd, 0xb3,
    0xfc, 0xc9, 0xf5, 0x0f, 0x0f, 0x0f, 0x0f, 0xe6,
    0x0f, 0xcd, 0xc4, 0xfc, 0xf1, 0xe6, 0x0f, 0xcd,
    0xc4, 0xfc, 0xc9, 0xfe, 0x0a, 0xfa, 0xcb, 0xfc,
    0xc6, 0x07, 0xc6, 0x30, 0x4f, 0xcd, 0xcc, 0x01,
    0xc9, 0xff, 0x00, 0x00, 0x01, 0xff, 0x02, 0x00,
    0x41, 0xff, 0x42, 0xff, 0x40, 0x00, 0x00, 0xff,
    0x01, 0x00, 0x02, 0x00, 0x42, 0x00, 0x40, 0xff,
    0x41, 0xaa, 0x00, 0x55, 0x02, 0xaa, 0x01, 0xaa,
    0x41, 0x55, 0x01, 0xaa, 0x40, 0x55, 0x00, 0x55,
    0x42, 0x55, 0x41, 0x55, 0x40, 0x00, 0x80, 0x57,
    0x72, 0x6f, 0x6e, 0x67, 0x20, 0x74, 0x72, 0x61,
    0x63, 0x6b, 0xae, 0x00, 0x57, 0x72, 0x69, 0x74,
    0x65, 0x20, 0x70, 0x72, 0x6f, 0x74, 0x65, 0x63,
    0x74, 0x65, 0x64, 0xae, 0x00, 0x44, 0x61, 0x74,
    0x61, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0xae,
    0x00, 0x4e, 0x6f, 0x74, 0x20, 0x66, 0x6f, 0x75,
    0x6e, 0x64, 0xae, 0x00, 0x4c, 0x6f, 0x73, 0x74,
    0x20, 0x64, 0x61, 0x74, 0x61, 0xae, 0x00, 0x44,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x6e, 0x6f, 0x74,
    0x20, 0x72, 0x65, 0x61, 0x64, 0x79, 0xae, 0x00,
    0x49, 0x6e, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20,
    0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e, 0x64, 0xae,
    0x00, 0x55, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e,
    0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0xae, 0x00,
    0x0d, 0x0a, 0x44, 0x69, 0x73, 0x6b, 0x20, 0x65,
    0x72, 0x72, 0x6f, 0x72, 0x20, 0x6f, 0x6e, 0x20,
    0x64, 0x72, 0x69, 0x76, 0x65, 0xa0, 0x00, 0x0d,
    0x0a, 0x54, 0x79, 0x70, 0x65, 0x20, 0x52, 0x20,
    0x74, 0x6f, 0x20, 0x74, 0x72, 0x79, 0x20, 0x61,
    0x67, 0x61, 0x69, 0x6e, 0x2c, 0x20, 0x41, 0x20,
    0x74, 0x6f, 0x20, 0x61, 0x62, 0x6f, 0x72, 0x74,
    0x2c, 0x20, 0x6f, 0x72, 0x20, 0x49, 0x20, 0x74,
    0x6f, 0x20, 0x69, 0x67, 0x6e, 0x6f, 0x72, 0x65,
    0x3a, 0xa0, 0x00, 0x0a, 0x45, 0x72, 0x72, 0x6f,
    0x72, 0x20, 0x6f, 0x6e, 0x20, 0x43, 0x50, 0x2f,
    0x4d, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
    0x20, 0x64, 0x69, 0x73, 0x6b, 0x65, 0x74, 0x74,
    0x65, 0x2e, 0x00, 0x0d, 0x0a, 0x50, 0x75, 0x73,
    0x68, 0x20, 0x72, 0x65, 0x73, 0x65, 0x74, 0x20,
    0x74, 0x6f, 0x20, 0x74, 0x72, 0x79, 0x20, 0x61,
    0x67, 0x61, 0x69, 0x6e, 0x2e, 0xa0, 0x00, 0x0d,
    0x4d, 0x69, 0x63, 0x72, 0x6f, 0x2d, 0x44, 0x65,
    0x63, 0x69, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x2d,
    0x2d, 0x20, 0x52, 0x4f, 0x4d, 0x20, 0x52, 0x65,
    0x76, 0x2e, 0x20, 0x32, 0x2e, 0x33, 0x0d, 0x0a,
    0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
    0x74, 0x20, 0x31, 0x39, 0x38, 0x32, 0x2c, 0x20,
    0x31, 0x39, 0x38, 0x33, 0x20, 0x4d, 0x6f, 0x72,
    0x72, 0x6f, 0x77, 0x20, 0x44, 0x65, 0x73, 0x69,
    0x67, 0x6e, 0x73, 0x2c, 0x20, 0x49, 0x6e, 0x63,
    0x2e, 0x0d, 0x0a, 0x0a, 0x00, 0x54, 0x65, 0x73,
    0x74, 0x69, 0x6e, 0x67, 0x20, 0x6d, 0x65, 0x6d,
    0x6f, 0x72, 0x79, 0x20, 0x2d, 0x2d, 0x20, 0x70,
    0x6c, 0x65, 0x61, 0x73, 0x65, 0x20, 0x73, 0x74,
    0x61, 0x6e, 0x64, 0x20, 0x62, 0x79, 0x3a, 0x20,
    0x00, 0x4d, 0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20,
    0x4f, 0x2e, 0x4b, 0x2e, 0x0d, 0x0a, 0x0a, 0x00,
    0x49, 0x6e, 0x73, 0x65, 0x72, 0x74, 0x20, 0x43,
    0x50, 0x2f, 0x4d, 0x20, 0x73, 0x79, 0x73, 0x74,
    0x65, 0x6d, 0x20, 0x64, 0x69, 0x73, 0x6b, 0x65,
    0x74, 0x74, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x44,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x41, 0x20, 0x61,
    0x6e, 0x64, 0x20, 0x70, 0x72, 0x65, 0x73, 0x73,
    0x20, 0x5b, 0x52, 0x45, 0x54, 0x55, 0x52, 0x4e,
    0x5d, 0x20, 0x07, 0x00, 0x07, 0x07, 0x52, 0x4f,
    0x4d, 0x20, 0x4d, 0x65, 0x6d, 0x6f, 0x72, 0x79,
    0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00, 0x07,
    0x07, 0x52, 0x41, 0x4d, 0x20, 0x4d, 0x65, 0x6d,
    0x6f, 0x72, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f,
    0x72, 0x00, 0x07, 0x07, 0x0d, 0x0a, 0x52, 0x65,
    0x61, 0x64, 0x20, 0x00, 0x20, 0x65, 0x78, 0x70,
    0x65, 0x63, 0x74, 0x65, 0x64, 0x20, 0x00, 0x20,
    0x61, 0x74, 0x20, 0x52, 0x41, 0x4d, 0x20, 0x6c,
    0x6f, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
    0x00, 0x0d, 0x0a, 0x44, 0x69, 0x61, 0x67, 0x6e,
    0x6f, 0x73, 0x74, 0x69, 0x63, 0x73, 0x20, 0x4d,
    0x65, 0x6e, 0x75, 0x0d, 0x0a, 0x0a, 0x31, 0x29,
    0x20, 0x50, 0x6f, 0x72, 0x74, 0x20, 0x31, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x5c, 0x0d, 0x0a, 0x32, 0x29, 0x20, 0x50,
    0x6f, 0x72, 0x74, 0x20, 0x32, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x3e, 0x20, 0x42, 0x61, 0x72, 0x62, 0x65, 0x72,
    0x2d, 0x50, 0x6f, 0x6c, 0x65, 0x20, 0x70, 0x61,
    0x74, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x74, 0x65,
    0x73, 0x74, 0x0d, 0x0a, 0x33, 0x29, 0x20, 0x43,
    0x65, 0x6e, 0x74, 0x72, 0x6f, 0x6e, 0x69, 0x63,
    0x73, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x20, 0x2f,
    0x0d, 0x0a, 0x34, 0x29, 0x20, 0x4c, 0x6f, 0x6f,
    0x70, 0x20, 0x62, 0x61, 0x63, 0x6b, 0x20, 0x6f,
    0x6e, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x20, 0x32,
    0x0d, 0x0a, 0x35, 0x29, 0x20, 0x52, 0x41, 0x4d,
    0x20, 0x74, 0x65, 0x73, 0x74, 0x0d, 0x0a, 0x36,
    0x29, 0x20, 0x46, 0x44, 0x43, 0x20, 0x52, 0x2f,
    0x57, 0x0d, 0x0a, 0x37, 0x29, 0x20, 0x46, 0x44,
    0x43, 0x20, 0x73, 0x65, 0x65, 0x6b, 0x20, 0x74,
    0x65, 0x73, 0x74, 0x0d, 0x0a, 0x38, 0x29, 0x20,
    0x56, 0x46, 0x4f, 0x20, 0x74, 0x65, 0x73, 0x74,
    0x0d, 0x0a, 0x39, 0x29, 0x20, 0x42, 0x6f, 0x6f,
    0x74, 0x0d, 0x0a, 0x0a, 0x45, 0x6e, 0x74, 0x65,
    0x72, 0x20, 0x23, 0x3a, 0x00, 0x0d, 0x0a, 0x45,
    0x6e, 0x64, 0x20, 0x6f, 0x66, 0x20, 0x70, 0x61,
    0x73, 0x73, 0x20, 0x00, 0x20, 0x65, 0x72, 0x72,
    0x6f, 0x72, 0x73, 0x20, 0x73, 0x6f, 0x20, 0x66,
    0x61, 0x72, 0x2e, 0x00, 0x6c, 0x6f, 0x6f, 0x70,
    0x20, 0x74, 0x65, 0x73, 0x74, 0x2e, 0x00, 0x0d,
    0x0a, 0x46, 0x61, 0x69, 0x6c, 0x65, 0x64, 0x20,
    0x00, 0x0d, 0x0a, 0x50, 0x61, 0x73, 0x73, 0x65,
    0x64, 0x20, 0x00, 0x45, 0x6e, 0x74, 0x65, 0x72,
    0x20, 0x64, 0x72, 0x69, 0x76, 0x65, 0x20, 0x74,
    0x6f, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x28,
    0x41, 0x2d, 0x44, 0x29, 0x3a, 0x00, 0x0d, 0x0a,
    0x49, 0x6e, 0x73, 0x65, 0x72, 0x74, 0x20, 0x61,
    0x20, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x74,
    0x65, 0x64, 0x20, 0x64, 0x69, 0x73, 0x6b, 0x65,
    0x74, 0x74, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x64,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x00, 0x0d, 0x0a,
    0x57, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2e,
    0x2e, 0x2e, 0x00, 0x72, 0x65, 0x61, 0x64, 0x69,
    0x6e, 0x67, 0x00, 0x0d, 0x0a, 0x00, 0x3a, 0xa0,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x14, 0x00, 0xfe,
    0x00, 0x00, 0x00, 0xd3, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x46,
    0x00, 0x00, 0x00, 0x01, 0x03, 0x05, 0x1c, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x75, 0x01,
    0x89, 0x01, 0x01, 0xd8, 0x58, 0x05, 0x6f, 0x03,
    0x05, 0x1c, 0xff, 0x02, 0x59, 0x58, 0x05, 0x6f,
    0x03, 0x05, 0x1c, 0xff, 0x04, 0x5a, 0x58, 0x05,
    0x6f, 0x03, 0x05, 0x1c, 0xff, 0x04, 0x5b, 0x58,
    0x05, 0x6f, 0x03, 0x05, 0x1c, 0xff, 0x4c, 0x0b,
    0x3f, 0x0b, 0x5d, 0x0b, 0x5d, 0x0b, 0x69, 0x0b,
    0x7f, 0x0b, 0x74, 0x0b, 0x7f, 0x0b, 0x90, 0x0b,
    0xa1, 0x0b, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    };

/* MD3 2.4 ROM is 4096 bytes */
static uint8 mmd_rom_24[MMD_ROM_SIZE] = {
    0xc3, 0x26, 0x00, 0xc3, 0xc3, 0x01, 0xc3, 0x70,
    0x05, 0xc3, 0x01, 0x03, 0xc3, 0x0e, 0x03, 0xc3,
    0x92, 0x03, 0xc3, 0xe6, 0x04, 0xc3, 0x3d, 0x05,
    0xc3, 0x57, 0x01, 0xb1, 0x0e, 0xc3, 0x98, 0x01,
    0xc3, 0xde, 0x01, 0xc3, 0xf1, 0x01, 0xed, 0x46,
    0x31, 0x00, 0x00, 0xfd, 0x21, 0xd0, 0x0e, 0xcd,
    0x85, 0x00, 0xcd, 0x53, 0x00, 0x11, 0x37, 0x0c,
    0xcd, 0xba, 0x01, 0xcd, 0xaa, 0x00, 0xcd, 0xe9,
    0x00, 0xcd, 0xca, 0x00, 0xcd, 0x00, 0xfc, 0xdb,
    0xfc, 0xdb, 0xf5, 0xcb, 0x6f, 0xcc, 0x01, 0x08,
    0xc3, 0x03, 0x01, 0x3e, 0x80, 0xd3, 0xfd, 0xd3,
    0xff, 0xd3, 0xfd, 0xd3, 0xff, 0x3e, 0x40, 0xd3,
    0xfd, 0xd3, 0xff, 0x3e, 0xce, 0xd3, 0xfd, 0xd3,
    0xff, 0x3e, 0x37, 0xd3, 0xfd, 0xd3, 0xff, 0xc9,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x81,
    0x06, 0xdd, 0x9e, 0x3f, 0x0a, 0x3e, 0x3e, 0xd3,
    0xf3, 0x3e, 0xff, 0xd3, 0xf0, 0x3e, 0xff, 0xd3,
    0xf0, 0x3e, 0x7e, 0xd3, 0xf3, 0x3e, 0x0d, 0xd3,
    0xf1, 0x3e, 0x00, 0xd3, 0xf1, 0x3e, 0xbe, 0xd3,
    0xf3, 0x3e, 0x68, 0xd3, 0xf2, 0x3e, 0x00, 0xd3,
    0xf2, 0xc9, 0x21, 0x00, 0x00, 0x01, 0xfd, 0x0f,
    0xcd, 0xdc, 0x00, 0xc8, 0x11, 0xf4, 0x0c, 0xcd,
    0xc3, 0x01, 0x18, 0x00, 0xdb, 0xf5, 0xcb, 0x6f,
    0xca, 0x47, 0x00, 0x11, 0x1b, 0x0c, 0xcd, 0xc3,
    0x01, 0x76, 0x21, 0x00, 0xfc, 0x01, 0xf8, 0x03,
    0xcd, 0xdc, 0x00, 0xc8, 0x11, 0x07, 0x0d, 0xcd,
    0xc3, 0x01, 0x18, 0xe0, 0xaf, 0xf5, 0xf1, 0xae,
    0x23, 0x0b, 0xf5, 0x78, 0xb1, 0x20, 0xf7, 0xf1,
    0xc9, 0x21, 0x3f, 0x0a, 0x11, 0x00, 0xfc, 0x01,
    0x00, 0x01, 0xed, 0xb0, 0x01, 0xf7, 0x02, 0xd5,
    0xe1, 0x13, 0x36, 0xff, 0xed, 0xb0, 0x3a, 0x82,
    0x00, 0x77, 0xc9, 0xcd, 0x5d, 0x01, 0x3a, 0xea,
    0x0e, 0xe6, 0x03, 0xcb, 0x57, 0x28, 0x01, 0x0f,
    0x4f, 0x3a, 0xeb, 0x0e, 0xe6, 0x03, 0x47, 0x04,
    0x3e, 0x08, 0x07, 0x10, 0xfd, 0xb1, 0x11, 0x00,
    0x60, 0xd3, 0xf7, 0x1b, 0x7a, 0xb3, 0x28, 0x13,
    0xdb, 0xf5, 0xcb, 0x57, 0x28, 0xf5, 0x1b, 0x7a,
    0xb3, 0x28, 0x08, 0xdb, 0xf5, 0xcb, 0x57, 0x20,
    0xf5, 0x18, 0x0f, 0x11, 0xb8, 0x0c, 0xcd, 0xc3,
    0x01, 0xcd, 0xc9, 0x04, 0x11, 0xab, 0x0e, 0xcd,
    0xc3, 0x01, 0xcd, 0x7f, 0x05, 0xcd, 0x92, 0x03,
    0xdd, 0x7e, 0x07, 0xb7, 0xca, 0x00, 0xfe, 0x11,
    0xfb, 0x0b, 0xc3, 0xbc, 0x00, 0x21, 0xb1, 0x0e,
    0x11, 0x00, 0xff, 0x01, 0x5d, 0x00, 0xed, 0xb0,
    0xdd, 0x21, 0x00, 0xff, 0xfd, 0x21, 0x1f, 0xff,
    0xc9, 0x06, 0xff, 0x18, 0x02, 0x06, 0xfd, 0x78,
    0x41, 0x4f, 0xed, 0x78, 0xcb, 0x47, 0x28, 0xfa,
    0x78, 0x0d, 0xed, 0x79, 0xc9, 0x06, 0xff, 0x18,
    0x02, 0x06, 0xfd, 0x48, 0xed, 0x78, 0xcb, 0x4f,
    0x28, 0xfa, 0x0d, 0xed, 0x78, 0xe6, 0x7f, 0xc9,
    0xdb, 0xf5, 0xcb, 0x67, 0x20, 0xfa, 0x79, 0xd3,
    0xf4, 0x3e, 0x80, 0xd3, 0xf5, 0xaf, 0xd3, 0xf5,
    0x21, 0x40, 0x00, 0xdb, 0xf5, 0xcb, 0x5f, 0x28,
    0x07, 0x2b, 0x7d, 0xb4, 0x20, 0xf5, 0x3d, 0xc9,
    0xaf, 0xc9, 0x06, 0x32, 0x3e, 0x0a, 0xcd, 0xcc,
    0x01, 0x10, 0xf9, 0x1a, 0xb7, 0xc8, 0xcd, 0xcc,
    0x01, 0x13, 0x18, 0xf7, 0x4f, 0xc5, 0xd5, 0xe5,
    0xfd, 0x6e, 0x16, 0xfd, 0x66, 0x17, 0xcd, 0xdd,
    0x01, 0xe1, 0xd1, 0xc1, 0xc9, 0xe9, 0xfd, 0x36,
    0x04, 0x00, 0xfd, 0xcb, 0x0b, 0xde, 0xfd, 0xcb,
    0x0b, 0xd6, 0xfd, 0x36, 0x0a, 0x02, 0xc3, 0x56,
    0x02, 0xfd, 0xcb, 0x0b, 0x9e, 0xfd, 0x71, 0x0a,
    0x79, 0xfe, 0x02, 0x20, 0x18, 0xfd, 0x7e, 0x08,
    0xfd, 0x77, 0x04, 0xfd, 0x7e, 0x00, 0xfd, 0x77,
    0x05, 0xfd, 0x7e, 0x01, 0xfd, 0x77, 0x06, 0xfd,
    0x7e, 0x02, 0xfd, 0x77, 0x07, 0xfd, 0x7e, 0x04,
    0xb7, 0x28, 0x33, 0xfd, 0x35, 0x04, 0xfd, 0x7e,
    0x00, 0xfd, 0xbe, 0x05, 0x20, 0x28, 0xfd, 0x7e,
    0x01, 0xfd, 0xbe, 0x06, 0x20, 0x20, 0xfd, 0x7e,
    0x02, 0xfd, 0xbe, 0x07, 0x20, 0x18, 0xfd, 0x34,
    0x07, 0xfd, 0x7e, 0x09, 0xfd, 0xbe, 0x07, 0x30,
    0x07, 0xfd, 0x34, 0x06, 0xfd, 0x36, 0x07, 0x01,
    0xfd, 0xcb, 0x0b, 0x96, 0x18, 0x08, 0xfd, 0x36,
    0x04, 0x00, 0xfd, 0xcb, 0x0b, 0xd6, 0xdd, 0x36,
    0x07, 0x00, 0xfd, 0x7e, 0x00, 0xcd, 0x42, 0x06,
    0x23, 0x7e, 0xfd, 0x5e, 0x02, 0x1d, 0x53, 0xe6,
    0x18, 0x06, 0x00, 0xfe, 0x00, 0x28, 0x14, 0x06,
    0x01, 0xfe, 0x08, 0x28, 0x0c, 0x06, 0x03, 0xfe,
    0x10, 0x28, 0x04, 0x06, 0x07, 0xcb, 0x2b, 0xcb,
    0x2b, 0xcb, 0x2b, 0x1c, 0xfd, 0x73, 0x03, 0x78,
    0xa2, 0xfd, 0x77, 0x0c, 0xfd, 0xcb, 0x0b, 0x46,
    0xfd, 0xcb, 0x0b, 0xc6, 0x28, 0x22, 0xfd, 0x7e,
    0x00, 0xdd, 0xbe, 0x00, 0x20, 0x10, 0xfd, 0x7e,
    0x01, 0xdd, 0xbe, 0x01, 0x20, 0x08, 0xfd, 0x7e,
    0x03, 0xdd, 0xbe, 0x02, 0x28, 0x2a, 0xaf, 0xfd,
    0xcb, 0x0b, 0x4e, 0xc4, 0x0e, 0x03, 0xb7, 0xc0,
    0xfd, 0x7e, 0x00, 0xdd, 0x77, 0x00, 0xfd, 0x7e,
    0x01, 0xdd, 0x77, 0x01, 0xfd, 0x7e, 0x03, 0xdd,
    0x77, 0x02, 0xaf, 0xfd, 0xcb, 0x0b, 0x56, 0xc4,
    0x01, 0x03, 0xb7, 0xc0, 0xfd, 0xcb, 0x0b, 0x8e,
    0xfd, 0x6e, 0x0c, 0x26, 0x00, 0x29, 0x29, 0x29,
    0x29, 0x29, 0x29, 0x29, 0xdd, 0x5e, 0x05, 0xdd,
    0x56, 0x06, 0x19, 0xfd, 0x5e, 0x14, 0xfd, 0x56,
    0x15, 0x01, 0x80, 0x00, 0xfd, 0xcb, 0x0b, 0x5e,
    0x3e, 0x00, 0xc0, 0xfd, 0xcb, 0x0b, 0xce, 0xeb,
    0xc9, 0x21, 0xe6, 0x04, 0xdd, 0x75, 0x0a, 0xdd,
    0x74, 0x0b, 0x0e, 0x06, 0x18, 0x0b, 0x21, 0x3d,
    0x05, 0xdd, 0x75, 0x0a, 0xdd, 0x74, 0x0b, 0x0e,
    0x05, 0xdd, 0x7e, 0x00, 0xcd, 0x42, 0x06, 0x23,
    0x7e, 0xe6, 0x40, 0xb1, 0xdd, 0x77, 0x16, 0xdd,
    0x36, 0x15, 0x09, 0x3e, 0x18, 0xa6, 0x0f, 0x0f,
    0x0f, 0xdd, 0x77, 0x1b, 0x06, 0x80, 0x11, 0x80,
    0x00, 0xfe, 0x00, 0x28, 0x11, 0x06, 0xff, 0x11,
    0x00, 0x00, 0xfe, 0x01, 0x28, 0x08, 0x16, 0x01,
    0xfe, 0x02, 0x28, 0x02, 0x16, 0x03, 0xdd, 0x70,
    0x1e, 0xdd, 0xcb, 0x0d, 0x7e, 0x20, 0x06, 0xdd,
    0x73, 0x0c, 0xdd, 0x72, 0x0d, 0xaf, 0xcb, 0x56,
    0xdd, 0x46, 0x01, 0xdd, 0x4e, 0x02, 0x28, 0x03,
    0xcb, 0x18, 0x17, 0xdd, 0x77, 0x19, 0x07, 0x07,
    0xdd, 0x77, 0x09, 0xdd, 0x70, 0x08, 0xdd, 0x70,
    0x18, 0xdd, 0x71, 0x1a, 0x3e, 0x03, 0xa6, 0xdd,
    0xb6, 0x09, 0xdd, 0x77, 0x17, 0x11, 0x05, 0x00,
    0x19, 0x7e, 0xdd, 0x77, 0x1c, 0x23, 0x7e, 0xdd,
    0x77, 0x1d, 0xdd, 0x46, 0x04, 0x16, 0x03, 0xc5,
    0xd5, 0xcd, 0x83, 0x05, 0xfd, 0xe5, 0xcd, 0x22,
    0x04, 0xcd, 0x56, 0x04, 0xfd, 0xe1, 0x06, 0x07,
    0xdd, 0xe5, 0xe1, 0x11, 0x0e, 0x00, 0x19, 0xcd,
    0xb5, 0x06, 0xed, 0xa2, 0x20, 0xf9, 0xd1, 0xc1,
    0xdd, 0x7e, 0x0e, 0xe6, 0xc0, 0x28, 0x35, 0xcd,
    0x1c, 0x07, 0x3e, 0x01, 0xb9, 0x28, 0x1d, 0x3e,
    0x06, 0xb9, 0xcc, 0x6a, 0x05, 0x3e, 0x02, 0xb9,
    0xcc, 0xfa, 0x03, 0x3e, 0x05, 0xb9, 0xcc, 0x09,
    0x04, 0xdd, 0x7e, 0x04, 0x0f, 0x3c, 0xb8, 0xcc,
    0x7f, 0x05, 0x10, 0xb3, 0xdd, 0x7e, 0x07, 0xfe,
    0xfe, 0x28, 0x0a, 0xcd, 0x62, 0x04, 0x38, 0x05,
    0xb7, 0xc2, 0x92, 0x03, 0x4f, 0xdd, 0x71, 0x07,
    0x79, 0xc9, 0xcd, 0x6a, 0x05, 0xc8, 0xcd, 0x7f,
    0x05, 0xdd, 0x7e, 0x08, 0xfe, 0x03, 0xd0, 0x18,
    0x0a, 0xdd, 0x7e, 0x08, 0xfe, 0x03, 0xd0, 0xcd,
    0x6a, 0x05, 0xc8, 0xf5, 0xdd, 0x36, 0x08, 0x03,
    0xcd, 0x83, 0x05, 0xf1, 0xdd, 0x77, 0x08, 0xc3,
    0x7f, 0x05, 0x0e, 0xfb, 0xdd, 0xe5, 0xe1, 0x11,
    0x15, 0x00, 0x19, 0x46, 0x23, 0xcd, 0x9f, 0x06,
    0xcd, 0xac, 0x06, 0xed, 0xa3, 0x20, 0xf9, 0xdd,
    0x46, 0x0c, 0xdd, 0x5e, 0x0d, 0xcb, 0xbb, 0xdd,
    0x56, 0x03, 0x15, 0xdd, 0x6e, 0x0a, 0xdd, 0x66,
    0x0b, 0xe5, 0xfd, 0xe1, 0xdd, 0x6e, 0x05, 0xdd,
    0x66, 0x06, 0xc9, 0xcb, 0x7f, 0xc0, 0xdb, 0xfa,
    0xcb, 0x6f, 0x28, 0xf7, 0xd5, 0x11, 0x00, 0x00,
    0xfd, 0xe9, 0x79, 0x07, 0x5f, 0x16, 0x00, 0x21,
    0x0c, 0x0f, 0x19, 0x5e, 0x23, 0x56, 0xc5, 0xd5,
    0x11, 0xb0, 0x0b, 0xcd, 0xc3, 0x01, 0x3e, 0x41,
    0xdd, 0x86, 0x00, 0xcd, 0xcc, 0x01, 0x11, 0xae,
    0x0e, 0xcd, 0xc3, 0x01, 0xd1, 0xcd, 0xc3, 0x01,
    0x3e, 0xff, 0xdd, 0xbe, 0x07, 0x37, 0xc4, 0xa3,
    0x04, 0xc1, 0xc9, 0xcd, 0xc3, 0x01, 0xcd, 0xc9,
    0x04, 0xfe, 0x0d, 0x20, 0xf9, 0x11, 0xab, 0x0e,
    0xc3, 0xc3, 0x01, 0x11, 0xc7, 0x0b, 0xcd, 0xc3,
    0x01, 0xcd, 0xc9, 0x04, 0xe6, 0xdf, 0x5f, 0xfe,
    0x41, 0x37, 0x28, 0x08, 0xfe, 0x52, 0x28, 0x04,
    0xd6, 0x49, 0x20, 0xed, 0xf5, 0x7b, 0xcd, 0xcc,
    0x01, 0x11, 0xab, 0x0e, 0xcd, 0xc3, 0x01, 0xf1,
    0xc9, 0xc5, 0xe5, 0xfd, 0x6e, 0x18, 0xfd, 0x66,
    0x19, 0xcd, 0xdd, 0x01, 0xe1, 0xc1, 0xc9, 0xe6,
    0x20, 0x28, 0x36, 0xdb, 0xfa, 0xfe, 0xf0, 0x28,
    0x0b, 0x1b, 0x7b, 0xb2, 0x28, 0x2d, 0xdb, 0xfa,
    0xfe, 0xf0, 0x20, 0xeb, 0xed, 0xa2, 0xd1, 0xd5,
    0x7b, 0xb2, 0x20, 0x01, 0x05, 0xfb, 0x76, 0xdb,
    0xfa, 0xe6, 0x20, 0x28, 0x14, 0xed, 0xa2, 0x20,
    0xf4, 0x1d, 0xf2, 0xf0, 0x04, 0xd1, 0x15, 0xd5,
    0xf2, 0xf0, 0x04, 0xfb, 0x76, 0xdb, 0xf7, 0xed,
    0xa2, 0xd1, 0xc9, 0xd1, 0xe1, 0xfd, 0xe1, 0xfd,
    0xe5, 0xc5, 0x0e, 0xf7, 0xfd, 0x7e, 0x12, 0x47,
    0xcb, 0xdf, 0xed, 0x79, 0xdb, 0xfa, 0xcb, 0x6f,
    0x20, 0xfa, 0xed, 0x41, 0xc1, 0xe9, 0xe6, 0x20,
    0x28, 0xdf, 0xdb, 0xfa, 0xfe, 0xb0, 0x28, 0x0b,
    0x1b, 0x7b, 0xb2, 0x28, 0xd6, 0xdb, 0xfa, 0xfe,
    0xb0, 0x20, 0xeb, 0xed, 0xa3, 0xd1, 0xd5, 0x7b,
    0xb2, 0x20, 0x01, 0x05, 0xfb, 0x76, 0xdb, 0xfa,
    0xe6, 0x20, 0x28, 0xbd, 0xed, 0xa3, 0x20, 0xf4,
    0x1d, 0xf2, 0x47, 0x05, 0xd1, 0x15, 0xd5, 0xf2,
    0x47, 0x05, 0xfb, 0x76, 0xdb, 0xf7, 0xed, 0xa3,
    0xd1, 0xc9, 0x04, 0x15, 0xc0, 0x06, 0x01, 0xc9,
    0xfd, 0xcb, 0x0b, 0x4e, 0x20, 0x04, 0xfd, 0xcb,
    0x0b, 0x86, 0xfd, 0x36, 0x01, 0x00, 0xc9, 0x3e,
    0x00, 0x18, 0x02, 0x3e, 0x08, 0xc5, 0xd5, 0x4f,
    0xdd, 0x7e, 0x00, 0xf5, 0xcd, 0x42, 0x06, 0x7e,
    0xe6, 0x07, 0xcb, 0x57, 0x28, 0x01, 0x0f, 0x5f,
    0x23, 0xf1, 0xfd, 0xbe, 0x11, 0x28, 0x1b, 0xfd,
    0x77, 0x11, 0xcb, 0x7e, 0xc5, 0x28, 0x0f, 0xfd,
    0xbe, 0x13, 0xfd, 0x77, 0x13, 0xf5, 0xd5, 0xe5,
    0xc4, 0x2f, 0x06, 0xe1, 0xd1, 0xf1, 0xcd, 0xe1,
    0x06, 0xc1, 0xcb, 0x76, 0x28, 0x0f, 0x3e, 0x13,
    0xcb, 0x6e, 0x28, 0x02, 0x3e, 0x27, 0xdd, 0xbe,
    0x08, 0x30, 0x02, 0xcb, 0xd3, 0xdb, 0xf5, 0xe6,
    0x03, 0x57, 0xa3, 0xf5, 0x7e, 0xe6, 0x03, 0x47,
    0x04, 0x3e, 0x08, 0x07, 0x10, 0xfd, 0xb2, 0xb3,
    0xd3, 0xf7, 0xfd, 0x77, 0x12, 0xcd, 0x53, 0x06,
    0xdd, 0x7e, 0x08, 0xcb, 0x59, 0x28, 0x0a, 0xb7,
    0x20, 0x04, 0xcb, 0x99, 0x18, 0x03, 0xba, 0x28,
    0x06, 0xcd, 0xbe, 0x06, 0xf1, 0x37, 0xf5, 0xf1,
    0xc5, 0xe5, 0xcd, 0x5b, 0x06, 0xe1, 0xc1, 0xcb,
    0x59, 0xcc, 0x0f, 0x06, 0xd1, 0xc1, 0xc9, 0xcd,
    0xac, 0x06, 0x3e, 0x04, 0xd3, 0xfb, 0xcd, 0xac,
    0x06, 0x3e, 0x03, 0xa6, 0xd3, 0xfb, 0xcd, 0xb5,
    0x06, 0xdb, 0xfb, 0xe6, 0x10, 0xc0, 0xcd, 0xbe,
    0x06, 0x3e, 0xff, 0xb7, 0x37, 0x18, 0x2c, 0x3c,
    0xcb, 0xf7, 0xfd, 0x5e, 0x0f, 0xfd, 0x56, 0x10,
    0x12, 0xfd, 0x5e, 0x0d, 0xfd, 0x56, 0x0e, 0xc3,
    0x93, 0x04, 0xfd, 0xe5, 0xe1, 0x11, 0x1a, 0x00,
    0x19, 0x57, 0x07, 0x07, 0x07, 0x82, 0x5f, 0x16,
    0x00, 0x19, 0xc9, 0xe5, 0x11, 0x07, 0x00, 0x19,
    0x56, 0xe1, 0xc9, 0x23, 0xfd, 0x46, 0x12, 0x0e,
    0xf7, 0x56, 0x1e, 0x00, 0x23, 0x30, 0x28, 0xed,
    0x5e, 0x3e, 0x00, 0xed, 0x47, 0x20, 0x0c, 0xfb,
    0x1b, 0x7a, 0xb3, 0x28, 0x08, 0x0e, 0xf7, 0xed,
    0x41, 0x18, 0xf5, 0x56, 0xfb, 0xed, 0x41, 0x18,
    0xfc, 0xed, 0x46, 0x33, 0x33, 0xcd, 0x0f, 0x07,
    0x7a, 0xbe, 0x30, 0x04, 0x56, 0x18, 0x01, 0xc0,
    0xed, 0x41, 0xed, 0x41, 0x00, 0x1b, 0x7a, 0xb3,
    0xc2, 0x90, 0x06, 0xc9, 0xcd, 0xf8, 0x06, 0xdb,
    0xfa, 0xe6, 0x0f, 0x20, 0xf7, 0xdb, 0xfa, 0xcb,
    0x67, 0x20, 0xf4, 0xc9, 0xdb, 0xfa, 0xe6, 0xc0,
    0xfe, 0x80, 0x20, 0xf8, 0xc9, 0xdb, 0xfa, 0xe6,
    0xc0, 0xfe, 0xc0, 0x20, 0xf8, 0xc9, 0xcd, 0xac,
    0x06, 0x3e, 0x07, 0xb1, 0xd3, 0xfb, 0xcd, 0xac,
    0x06, 0x3e, 0x03, 0xa6, 0xcb, 0x59, 0x28, 0x03,
    0xdd, 0xb6, 0x09, 0xd3, 0xfb, 0xcb, 0x59, 0xc8,
    0xcd, 0xac, 0x06, 0xdd, 0x7e, 0x08, 0xd3, 0xfb,
    0xc9, 0xe5, 0x23, 0x23, 0x23, 0xcd, 0xac, 0x06,
    0x3e, 0x03, 0xd3, 0xfb, 0x01, 0xfb, 0x02, 0xcd,
    0xac, 0x06, 0xed, 0xa3, 0x20, 0xf9, 0xe1, 0xc9,
    0xcd, 0xac, 0x06, 0x3e, 0x08, 0xd3, 0xfb, 0xcd,
    0xb5, 0x06, 0xdb, 0xfb, 0xe6, 0xc0, 0xfe, 0x80,
    0xc8, 0xcd, 0xb5, 0x06, 0xdb, 0xfb, 0xc9, 0xcd,
    0xf8, 0x06, 0xe5, 0xd5, 0x11, 0x05, 0x00, 0x19,
    0x77, 0xd1, 0xe1, 0xc9, 0x0e, 0x01, 0xdd, 0xcb,
    0x0f, 0x4e, 0xc0, 0x0c, 0xdd, 0xcb, 0x10, 0x4e,
    0xc0, 0xdd, 0xcb, 0x10, 0x66, 0xc0, 0x0c, 0xdd,
    0xcb, 0x10, 0x6e, 0xc0, 0x0c, 0xdd, 0xcb, 0x0f,
    0x6e, 0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x7e, 0xc0,
    0xdd, 0xcb, 0x0f, 0x56, 0xc0, 0xdd, 0xcb, 0x0f,
    0x46, 0xc0, 0xdd, 0xcb, 0x10, 0x76, 0xc0, 0xdd,
    0xcb, 0x10, 0x46, 0xc0, 0x0c, 0xdd, 0xcb, 0x0e,
    0x5e, 0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x66, 0xc0,
    0x0c, 0xdd, 0xcb, 0x0e, 0x66, 0xc0, 0x0c, 0xdd,
    0xcb, 0x0e, 0x7e, 0xc0, 0x0c, 0xc9, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x13, 0x22, 0x24, 0xff,
    0xff, 0x11, 0x41, 0x0d, 0xcd, 0xc3, 0x01, 0xcd,
    0xc9, 0x04, 0xfe, 0x31, 0xfa, 0x07, 0x08, 0xfe,
    0x3a, 0xf2, 0x07, 0x08, 0xf5, 0xcd, 0xcc, 0x01,
    0x11, 0xab, 0x0e, 0xcd, 0xc3, 0x01, 0xf1, 0xd6,
    0x30, 0xfe, 0x04, 0xf2, 0x29, 0x08, 0x32, 0xfe,
    0xff, 0x3d, 0x07, 0x4f, 0xaf, 0x47, 0x21, 0x3b,
    0x08, 0x09, 0x5e, 0x23, 0x56, 0xeb, 0xcd, 0xdd,
    0x01, 0x18, 0xc6, 0x4d, 0x08, 0x4d, 0x08, 0x4d,
    0x08, 0xa3, 0x08, 0xcd, 0x08, 0x21, 0x09, 0xc9,
    0x09, 0x1a, 0x0a, 0x3d, 0x0a, 0x3e, 0x20, 0x32,
    0xfd, 0xff, 0x3e, 0x5f, 0x47, 0x3a, 0xfd, 0xff,
    0xcd, 0x80, 0x08, 0x05, 0x28, 0x08, 0x3c, 0xf2,
    0x58, 0x08, 0x3e, 0x20, 0x18, 0xf2, 0xcd, 0x9e,
    0x08, 0xc0, 0x3e, 0x0d, 0xcd, 0x80, 0x08, 0x3e,
    0x0a, 0xcd, 0x80, 0x08, 0x3a, 0xfd, 0xff, 0x3c,
    0xfa, 0x4d, 0x08, 0x32, 0xfd, 0xff, 0x18, 0xd2,
    0xf5, 0xc5, 0x4f, 0x3a, 0xfe, 0xff, 0xfe, 0x01,
    0x20, 0x05, 0xcd, 0x75, 0x01, 0x18, 0x0c, 0xfe,
    0x02, 0x20, 0x05, 0xcd, 0x71, 0x01, 0x18, 0x03,
    0xcd, 0x98, 0x01, 0xc1, 0xf1, 0xc9, 0xdb, 0xfd,
    0xcb, 0x4f, 0xc9, 0xdb, 0xfe, 0xaf, 0x57, 0x4f,
    0xcd, 0x71, 0x01, 0xcd, 0x85, 0x01, 0xba, 0x20,
    0x0f, 0xcd, 0x9e, 0x08, 0x20, 0x05, 0x7a, 0x3c,
    0xf2, 0xa6, 0x08, 0x11, 0x49, 0x0e, 0x18, 0x03,
    0x11, 0x3f, 0x0e, 0xcd, 0xc3, 0x01, 0x11, 0x34,
    0x0e, 0xcd, 0xc3, 0x01, 0xc9, 0x11, 0x85, 0x0c,
    0xcd, 0xc3, 0x01, 0xcd, 0xee, 0x08, 0x21, 0x00,
    0x00, 0x01, 0xff, 0xfb, 0xcd, 0x2a, 0xfc, 0xb7,
    0x28, 0x03, 0xcd, 0xf6, 0x08, 0xcd, 0xfe, 0x08,
    0xcd, 0x9e, 0x08, 0x28, 0xe9, 0xc9, 0xaf, 0x32,
    0xfe, 0xff, 0x32, 0xfd, 0xff, 0xc9, 0x3a, 0xfd,
    0xff, 0x3c, 0x32, 0xfd, 0xff, 0xc9, 0x11, 0x15,
    0x0e, 0xcd, 0xc3, 0x01, 0x3a, 0xfe, 0xff, 0x3c,
    0x32, 0xfe, 0xff, 0xcd, 0xb3, 0xfc, 0x11, 0xae,
    0x0e, 0xcd, 0xc3, 0x01, 0x3a, 0xfd, 0xff, 0xcd,
    0xb3, 0xfc, 0x11, 0x24, 0x0e, 0xcd, 0xc3, 0x01,
    0xc9, 0xcd, 0x3e, 0x09, 0xdd, 0x36, 0x01, 0x27,
    0xdd, 0x36, 0x03, 0x05, 0xcd, 0x74, 0x09, 0xcd,
    0xee, 0x08, 0xcd, 0x9b, 0x09, 0xcd, 0xac, 0x09,
    0xcd, 0x9e, 0x08, 0x28, 0xf5, 0xc9, 0xcd, 0x5d,
    0x01, 0x11, 0x53, 0x0e, 0xcd, 0xc3, 0x01, 0xcd,
    0xc9, 0x04, 0xe6, 0xdf, 0xd6, 0x41, 0x38, 0xf7,
    0xfe, 0x04, 0x30, 0xf3, 0xdd, 0x77, 0x00, 0xf5,
    0x11, 0x6e, 0x0e, 0xcd, 0xc3, 0x01, 0xf1, 0xc6,
    0x41, 0xcd, 0xcc, 0x01, 0x11, 0xde, 0x0c, 0xcd,
    0xc3, 0x01, 0xcd, 0xc9, 0x04, 0x11, 0xab, 0x0e,
    0xcd, 0xc3, 0x01, 0xc9, 0x01, 0x5f, 0xaa, 0x21,
    0x00, 0x80, 0xe5, 0x11, 0x02, 0x80, 0x71, 0x23,
    0x70, 0x2b, 0x01, 0x00, 0x08, 0xed, 0xb0, 0x01,
    0xb6, 0x6d, 0x71, 0x23, 0x70, 0x2b, 0x01, 0x00,
    0x0c, 0xed, 0xb0, 0xc1, 0xdd, 0x71, 0x05, 0xdd,
    0x70, 0x06, 0xc9, 0x11, 0x96, 0x0e, 0xcd, 0xc3,
    0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0x0e, 0x03,
    0xcd, 0xc0, 0x09, 0xc9, 0x11, 0xa3, 0x0e, 0xcd,
    0xc3, 0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0x01,
    0x03, 0xcd, 0xc0, 0x09, 0xcd, 0xfe, 0x08, 0xc9,
    0xdd, 0x7e, 0x07, 0xfe, 0x00, 0xc4, 0xf6, 0x08,
    0xc9, 0xcd, 0x3e, 0x09, 0x06, 0x00, 0x0e, 0x27,
    0xdd, 0x36, 0x05, 0x00, 0xdd, 0x36, 0x06, 0x80,
    0xdd, 0x36, 0x03, 0x01, 0xcd, 0xee, 0x08, 0xc5,
    0xcd, 0xfd, 0x09, 0xc1, 0xcd, 0xed, 0x09, 0xcd,
    0x9e, 0x08, 0x28, 0xf3, 0xc9, 0x04, 0x0d, 0xfa,
    0xf3, 0x09, 0xc9, 0x78, 0x41, 0x4f, 0xc5, 0xcd,
    0xfe, 0x08, 0xc1, 0x18, 0xf0, 0xc5, 0xdd, 0x70,
    0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0x01, 0x03,
    0xcd, 0xc0, 0x09, 0xc1, 0xdd, 0x71, 0x01, 0xdd,
    0x36, 0x07, 0xff, 0xcd, 0x01, 0x03, 0xcd, 0xc0,
    0x09, 0xc9, 0xcd, 0x5d, 0x01, 0xaf, 0xdd, 0x77,
    0x00, 0xdd, 0x77, 0x01, 0xdd, 0x36, 0x03, 0x01,
    0xdd, 0x36, 0x05, 0x00, 0xdd, 0x36, 0x06, 0x80,
    0xdd, 0x36, 0x07, 0xfe, 0xcd, 0x01, 0x03, 0xcd,
    0x9e, 0x08, 0x28, 0xf4, 0xc9, 0xf1, 0xc9, 0xdb,
    0xf6, 0x3a, 0x00, 0x00, 0xfe, 0xc3, 0xd3, 0xf6,
    0x20, 0x03, 0xc3, 0x47, 0x00, 0x11, 0x85, 0x0c,
    0xcd, 0xc3, 0x01, 0x21, 0x00, 0x00, 0x01, 0xff,
    0xfb, 0xcd, 0x2a, 0xfc, 0xb7, 0xc2, 0xd4, 0x00,
    0x11, 0xa9, 0x0c, 0xcd, 0xc3, 0x01, 0xc3, 0x47,
    0x00, 0xdb, 0xf6, 0x22, 0x00, 0xfd, 0x21, 0xd2,
    0xfc, 0x56, 0x23, 0xcb, 0x7e, 0x20, 0x18, 0x5e,
    0xcb, 0x73, 0xcb, 0xb3, 0x23, 0xe5, 0x28, 0x05,
    0xcd, 0x54, 0xfc, 0x18, 0x07, 0xdd, 0x21, 0x74,
    0xfc, 0xcd, 0x58, 0xfc, 0xe1, 0x18, 0xe2, 0xaf,
    0xd3, 0xf6, 0xc9, 0xdd, 0x21, 0x76, 0xfc, 0x2a,
    0x00, 0xfd, 0x7b, 0x85, 0x6f, 0x7c, 0xce, 0x00,
    0x67, 0xcd, 0x72, 0xfc, 0x23, 0x23, 0x23, 0x78,
    0xbc, 0xd8, 0x20, 0xf5, 0x79, 0xbd, 0xd8, 0x18,
    0xf0, 0xdd, 0xe9, 0x72, 0xc9, 0x7e, 0xba, 0x20,
    0x01, 0xc9, 0xd3, 0xf6, 0xdb, 0xf5, 0xcb, 0x6f,
    0xcc, 0x8a, 0xfc, 0xf1, 0xf1, 0xf1, 0x3e, 0xff,
    0xc9, 0xd5, 0x11, 0x1a, 0x0d, 0xcd, 0xc3, 0x01,
    0xdb, 0xf6, 0x7e, 0xd3, 0xf6, 0xcd, 0xb3, 0xfc,
    0x11, 0x24, 0x0d, 0xcd, 0xc3, 0x01, 0xd1, 0x7a,
    0xcd, 0xb3, 0xfc, 0x11, 0x2f, 0x0d, 0xcd, 0xc3,
    0x01, 0x7c, 0xcd, 0xb3, 0xfc, 0x7d, 0xcd, 0xb3,
    0xfc, 0xc9, 0xf5, 0x0f, 0x0f, 0x0f, 0x0f, 0xe6,
    0x0f, 0xcd, 0xc4, 0xfc, 0xf1, 0xe6, 0x0f, 0xcd,
    0xc4, 0xfc, 0xc9, 0xfe, 0x0a, 0xfa, 0xcb, 0xfc,
    0xc6, 0x07, 0xc6, 0x30, 0x4f, 0xcd, 0xcc, 0x01,
    0xc9, 0xff, 0x00, 0x00, 0x01, 0xff, 0x02, 0x00,
    0x41, 0xff, 0x42, 0xff, 0x40, 0x00, 0x00, 0xff,
    0x01, 0x00, 0x02, 0x00, 0x42, 0x00, 0x40, 0xff,
    0x41, 0xaa, 0x00, 0x55, 0x02, 0xaa, 0x01, 0xaa,
    0x41, 0x55, 0x01, 0xaa, 0x40, 0x55, 0x00, 0x55,
    0x42, 0x55, 0x41, 0x55, 0x40, 0x00, 0x80, 0x57,
    0x72, 0x6f, 0x6e, 0x67, 0x20, 0x74, 0x72, 0x61,
    0x63, 0x6b, 0xae, 0x00, 0x57, 0x72, 0x69, 0x74,
    0x65, 0x20, 0x70, 0x72, 0x6f, 0x74, 0x65, 0x63,
    0x74, 0x65, 0x64, 0xae, 0x00, 0x44, 0x61, 0x74,
    0x61, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0xae,
    0x00, 0x4e, 0x6f, 0x74, 0x20, 0x66, 0x6f, 0x75,
    0x6e, 0x64, 0xae, 0x00, 0x4c, 0x6f, 0x73, 0x74,
    0x20, 0x64, 0x61, 0x74, 0x61, 0xae, 0x00, 0x44,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x6e, 0x6f, 0x74,
    0x20, 0x72, 0x65, 0x61, 0x64, 0x79, 0xae, 0x00,
    0x49, 0x6e, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20,
    0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e, 0x64, 0xae,
    0x00, 0x55, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e,
    0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0xae, 0x00,
    0x0d, 0x0a, 0x44, 0x69, 0x73, 0x6b, 0x20, 0x65,
    0x72, 0x72, 0x6f, 0x72, 0x20, 0x6f, 0x6e, 0x20,
    0x64, 0x72, 0x69, 0x76, 0x65, 0xa0, 0x00, 0x0d,
    0x0a, 0x54, 0x79, 0x70, 0x65, 0x20, 0x52, 0x20,
    0x74, 0x6f, 0x20, 0x74, 0x72, 0x79, 0x20, 0x61,
    0x67, 0x61, 0x69, 0x6e, 0x2c, 0x20, 0x41, 0x20,
    0x74, 0x6f, 0x20, 0x61, 0x62, 0x6f, 0x72, 0x74,
    0x2c, 0x20, 0x6f, 0x72, 0x20, 0x49, 0x20, 0x74,
    0x6f, 0x20, 0x69, 0x67, 0x6e, 0x6f, 0x72, 0x65,
    0x3a, 0xa0, 0x00, 0x0a, 0x45, 0x72, 0x72, 0x6f,
    0x72, 0x20, 0x6f, 0x6e, 0x20, 0x43, 0x50, 0x2f,
    0x4d, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
    0x20, 0x64, 0x69, 0x73, 0x6b, 0x65, 0x74, 0x74,
    0x65, 0x2e, 0x00, 0x0d, 0x0a, 0x50, 0x75, 0x73,
    0x68, 0x20, 0x72, 0x65, 0x73, 0x65, 0x74, 0x20,
    0x74, 0x6f, 0x20, 0x74, 0x72, 0x79, 0x20, 0x61,
    0x67, 0x61, 0x69, 0x6e, 0x2e, 0xa0, 0x00, 0x0d,
    0x4d, 0x69, 0x63, 0x72, 0x6f, 0x2d, 0x44, 0x65,
    0x63, 0x69, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x2d,
    0x2d, 0x20, 0x52, 0x4f, 0x4d, 0x20, 0x52, 0x65,
    0x76, 0x2e, 0x20, 0x32, 0x2e, 0x34, 0x0d, 0x0a,
    0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
    0x74, 0x20, 0x31, 0x39, 0x38, 0x32, 0x2c, 0x20,
    0x31, 0x39, 0x38, 0x33, 0x20, 0x4d, 0x6f, 0x72,
    0x72, 0x6f, 0x77, 0x20, 0x44, 0x65, 0x73, 0x69,
    0x67, 0x6e, 0x73, 0x2c, 0x20, 0x49, 0x6e, 0x63,
    0x2e, 0x0d, 0x0a, 0x0a, 0x00, 0x54, 0x65, 0x73,
    0x74, 0x69, 0x6e, 0x67, 0x20, 0x6d, 0x65, 0x6d,
    0x6f, 0x72, 0x79, 0x20, 0x2d, 0x2d, 0x20, 0x70,
    0x6c, 0x65, 0x61, 0x73, 0x65, 0x20, 0x73, 0x74,
    0x61, 0x6e, 0x64, 0x20, 0x62, 0x79, 0x3a, 0x20,
    0x00, 0x4d, 0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20,
    0x4f, 0x2e, 0x4b, 0x2e, 0x0d, 0x0a, 0x0a, 0x00,
    0x49, 0x6e, 0x73, 0x65, 0x72, 0x74, 0x20, 0x43,
    0x50, 0x2f, 0x4d, 0x20, 0x73, 0x79, 0x73, 0x74,
    0x65, 0x6d, 0x20, 0x64, 0x69, 0x73, 0x6b, 0x65,
    0x74, 0x74, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x44,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x41, 0x20, 0x61,
    0x6e, 0x64, 0x20, 0x70, 0x72, 0x65, 0x73, 0x73,
    0x20, 0x5b, 0x52, 0x45, 0x54, 0x55, 0x52, 0x4e,
    0x5d, 0x20, 0x07, 0x00, 0x07, 0x07, 0x52, 0x4f,
    0x4d, 0x20, 0x4d, 0x65, 0x6d, 0x6f, 0x72, 0x79,
    0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00, 0x07,
    0x07, 0x52, 0x41, 0x4d, 0x20, 0x4d, 0x65, 0x6d,
    0x6f, 0x72, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f,
    0x72, 0x00, 0x07, 0x07, 0x0d, 0x0a, 0x52, 0x65,
    0x61, 0x64, 0x20, 0x00, 0x20, 0x65, 0x78, 0x70,
    0x65, 0x63, 0x74, 0x65, 0x64, 0x20, 0x00, 0x20,
    0x61, 0x74, 0x20, 0x52, 0x41, 0x4d, 0x20, 0x6c,
    0x6f, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
    0x00, 0x0d, 0x0a, 0x44, 0x69, 0x61, 0x67, 0x6e,
    0x6f, 0x73, 0x74, 0x69, 0x63, 0x73, 0x20, 0x4d,
    0x65, 0x6e, 0x75, 0x0d, 0x0a, 0x0a, 0x31, 0x29,
    0x20, 0x50, 0x6f, 0x72, 0x74, 0x20, 0x31, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x5c, 0x0d, 0x0a, 0x32, 0x29, 0x20, 0x50,
    0x6f, 0x72, 0x74, 0x20, 0x32, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x3e, 0x20, 0x42, 0x61, 0x72, 0x62, 0x65, 0x72,
    0x2d, 0x50, 0x6f, 0x6c, 0x65, 0x20, 0x70, 0x61,
    0x74, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x74, 0x65,
    0x73, 0x74, 0x0d, 0x0a, 0x33, 0x29, 0x20, 0x43,
    0x65, 0x6e, 0x74, 0x72, 0x6f, 0x6e, 0x69, 0x63,
    0x73, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x20, 0x2f,
    0x0d, 0x0a, 0x34, 0x29, 0x20, 0x4c, 0x6f, 0x6f,
    0x70, 0x20, 0x62, 0x61, 0x63, 0x6b, 0x20, 0x6f,
    0x6e, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x20, 0x32,
    0x0d, 0x0a, 0x35, 0x29, 0x20, 0x52, 0x41, 0x4d,
    0x20, 0x74, 0x65, 0x73, 0x74, 0x0d, 0x0a, 0x36,
    0x29, 0x20, 0x46, 0x44, 0x43, 0x20, 0x52, 0x2f,
    0x57, 0x0d, 0x0a, 0x37, 0x29, 0x20, 0x46, 0x44,
    0x43, 0x20, 0x73, 0x65, 0x65, 0x6b, 0x20, 0x74,
    0x65, 0x73, 0x74, 0x0d, 0x0a, 0x38, 0x29, 0x20,
    0x56, 0x46, 0x4f, 0x20, 0x74, 0x65, 0x73, 0x74,
    0x0d, 0x0a, 0x39, 0x29, 0x20, 0x42, 0x6f, 0x6f,
    0x74, 0x0d, 0x0a, 0x0a, 0x45, 0x6e, 0x74, 0x65,
    0x72, 0x20, 0x23, 0x3a, 0x00, 0x0d, 0x0a, 0x45,
    0x6e, 0x64, 0x20, 0x6f, 0x66, 0x20, 0x70, 0x61,
    0x73, 0x73, 0x20, 0x00, 0x20, 0x65, 0x72, 0x72,
    0x6f, 0x72, 0x73, 0x20, 0x73, 0x6f, 0x20, 0x66,
    0x61, 0x72, 0x2e, 0x00, 0x6c, 0x6f, 0x6f, 0x70,
    0x20, 0x74, 0x65, 0x73, 0x74, 0x2e, 0x00, 0x0d,
    0x0a, 0x46, 0x61, 0x69, 0x6c, 0x65, 0x64, 0x20,
    0x00, 0x0d, 0x0a, 0x50, 0x61, 0x73, 0x73, 0x65,
    0x64, 0x20, 0x00, 0x45, 0x6e, 0x74, 0x65, 0x72,
    0x20, 0x64, 0x72, 0x69, 0x76, 0x65, 0x20, 0x74,
    0x6f, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x28,
    0x41, 0x2d, 0x44, 0x29, 0x3a, 0x00, 0x0d, 0x0a,
    0x49, 0x6e, 0x73, 0x65, 0x72, 0x74, 0x20, 0x61,
    0x20, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x74,
    0x65, 0x64, 0x20, 0x64, 0x69, 0x73, 0x6b, 0x65,
    0x74, 0x74, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x64,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x00, 0x0d, 0x0a,
    0x57, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2e,
    0x2e, 0x2e, 0x00, 0x72, 0x65, 0x61, 0x64, 0x69,
    0x6e, 0x67, 0x00, 0x0d, 0x0a, 0x00, 0x3a, 0xa0,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x14, 0x00, 0xfe,
    0x00, 0x00, 0x00, 0xe6, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x46,
    0x00, 0x00, 0x00, 0x01, 0x03, 0x05, 0x1c, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x75, 0x01,
    0x89, 0x01, 0x01, 0xd8, 0x58, 0x05, 0x6f, 0x03,
    0x05, 0x1c, 0xff, 0x02, 0x59, 0x58, 0x05, 0x6f,
    0x03, 0x05, 0x1c, 0xff, 0x04, 0x5a, 0x58, 0x05,
    0x6f, 0x03, 0x05, 0x1c, 0xff, 0x04, 0x5b, 0x58,
    0x05, 0x6f, 0x03, 0x05, 0x1c, 0xff, 0x4c, 0x0b,
    0x3f, 0x0b, 0x5d, 0x0b, 0x5d, 0x0b, 0x69, 0x0b,
    0x7f, 0x0b, 0x74, 0x0b, 0x7f, 0x0b, 0x90, 0x0b,
    0xa1, 0x0b, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    };

/* MD3 2.5 ROM is 4096 bytes */
static uint8 mmd_rom_25[MMD_ROM_SIZE] = {
    0xc3, 0x26, 0x00, 0xc3, 0xc3, 0x01, 0xc3, 0x89,
    0x05, 0xc3, 0x01, 0x03, 0xc3, 0x0e, 0x03, 0xc3,
    0x92, 0x03, 0xc3, 0xff, 0x04, 0xc3, 0x56, 0x05,
    0xc3, 0x57, 0x01, 0xb1, 0x0e, 0xc3, 0x98, 0x01,
    0xc3, 0xde, 0x01, 0xc3, 0xf1, 0x01, 0xed, 0x46,
    0x31, 0x00, 0x00, 0xfd, 0x21, 0xd0, 0x0e, 0xcd,
    0x85, 0x00, 0xcd, 0x53, 0x00, 0x11, 0x37, 0x0c,
    0xcd, 0xba, 0x01, 0xcd, 0xaa, 0x00, 0xcd, 0xe9,
    0x00, 0xcd, 0xca, 0x00, 0xcd, 0x00, 0xfc, 0xdb,
    0xfc, 0xdb, 0xf5, 0xcb, 0x6f, 0xcc, 0x01, 0x08,
    0xc3, 0x03, 0x01, 0x3e, 0x80, 0xd3, 0xfd, 0xd3,
    0xff, 0xd3, 0xfd, 0xd3, 0xff, 0x3e, 0x40, 0xd3,
    0xfd, 0xd3, 0xff, 0x3e, 0xce, 0xd3, 0xfd, 0xd3,
    0xff, 0x3e, 0x37, 0xd3, 0xfd, 0xd3, 0xff, 0xc9,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9a,
    0x06, 0x9b, 0x9e, 0x3f, 0x0a, 0x3e, 0x3e, 0xd3,
    0xf3, 0x3e, 0xff, 0xd3, 0xf0, 0x3e, 0xff, 0xd3,
    0xf0, 0x3e, 0x7e, 0xd3, 0xf3, 0x3e, 0x0d, 0xd3,
    0xf1, 0x3e, 0x00, 0xd3, 0xf1, 0x3e, 0xbe, 0xd3,
    0xf3, 0x3e, 0x68, 0xd3, 0xf2, 0x3e, 0x00, 0xd3,
    0xf2, 0xc9, 0x21, 0x00, 0x00, 0x01, 0xfd, 0x0f,
    0xcd, 0xdc, 0x00, 0xc8, 0x11, 0xf4, 0x0c, 0xcd,
    0xc3, 0x01, 0x18, 0x00, 0xdb, 0xf5, 0xcb, 0x6f,
    0xca, 0x47, 0x00, 0x11, 0x1b, 0x0c, 0xcd, 0xc3,
    0x01, 0x76, 0x21, 0x00, 0xfc, 0x01, 0xf8, 0x03,
    0xcd, 0xdc, 0x00, 0xc8, 0x11, 0x07, 0x0d, 0xcd,
    0xc3, 0x01, 0x18, 0xe0, 0xaf, 0xf5, 0xf1, 0xae,
    0x23, 0x0b, 0xf5, 0x78, 0xb1, 0x20, 0xf7, 0xf1,
    0xc9, 0x21, 0x3f, 0x0a, 0x11, 0x00, 0xfc, 0x01,
    0x00, 0x01, 0xed, 0xb0, 0x01, 0xf7, 0x02, 0xd5,
    0xe1, 0x13, 0x36, 0xff, 0xed, 0xb0, 0x3a, 0x82,
    0x00, 0x77, 0xc9, 0xcd, 0x5d, 0x01, 0x3a, 0xea,
    0x0e, 0xe6, 0x03, 0xcb, 0x57, 0x28, 0x01, 0x0f,
    0x4f, 0x3a, 0xeb, 0x0e, 0xe6, 0x03, 0x47, 0x04,
    0x3e, 0x08, 0x07, 0x10, 0xfd, 0xb1, 0x11, 0x00,
    0x60, 0xd3, 0xf7, 0x1b, 0x7a, 0xb3, 0x28, 0x13,
    0xdb, 0xf5, 0xcb, 0x57, 0x28, 0xf5, 0x1b, 0x7a,
    0xb3, 0x28, 0x08, 0xdb, 0xf5, 0xcb, 0x57, 0x20,
    0xf5, 0x18, 0x0f, 0x11, 0xb8, 0x0c, 0xcd, 0xc3,
    0x01, 0xcd, 0xe2, 0x04, 0x11, 0xab, 0x0e, 0xcd,
    0xc3, 0x01, 0xcd, 0x98, 0x05, 0xcd, 0x92, 0x03,
    0xdd, 0x7e, 0x07, 0xb7, 0xca, 0x00, 0xfe, 0x11,
    0xfb, 0x0b, 0xc3, 0xbc, 0x00, 0x21, 0xb1, 0x0e,
    0x11, 0x00, 0xff, 0x01, 0x5d, 0x00, 0xed, 0xb0,
    0xdd, 0x21, 0x00, 0xff, 0xfd, 0x21, 0x1f, 0xff,
    0xc9, 0x06, 0xff, 0x18, 0x02, 0x06, 0xfd, 0x78,
    0x41, 0x4f, 0xed, 0x78, 0xcb, 0x47, 0x28, 0xfa,
    0x78, 0x0d, 0xed, 0x79, 0xc9, 0x06, 0xff, 0x18,
    0x02, 0x06, 0xfd, 0x48, 0xed, 0x78, 0xcb, 0x4f,
    0x28, 0xfa, 0x0d, 0xed, 0x78, 0xe6, 0x7f, 0xc9,
    0xdb, 0xf5, 0xcb, 0x67, 0x20, 0xfa, 0x79, 0xd3,
    0xf4, 0x3e, 0x80, 0xd3, 0xf5, 0xaf, 0xd3, 0xf5,
    0x21, 0x40, 0x00, 0xdb, 0xf5, 0xcb, 0x5f, 0x28,
    0x07, 0x2b, 0x7d, 0xb4, 0x20, 0xf5, 0x3d, 0xc9,
    0xaf, 0xc9, 0x06, 0x32, 0x3e, 0x0a, 0xcd, 0xcc,
    0x01, 0x10, 0xf9, 0x1a, 0xb7, 0xc8, 0xcd, 0xcc,
    0x01, 0x13, 0x18, 0xf7, 0x4f, 0xc5, 0xd5, 0xe5,
    0xfd, 0x6e, 0x16, 0xfd, 0x66, 0x17, 0xcd, 0xdd,
    0x01, 0xe1, 0xd1, 0xc1, 0xc9, 0xe9, 0xfd, 0x36,
    0x04, 0x00, 0xfd, 0xcb, 0x0b, 0xde, 0xfd, 0xcb,
    0x0b, 0xd6, 0xfd, 0x36, 0x0a, 0x02, 0xc3, 0x56,
    0x02, 0xfd, 0xcb, 0x0b, 0x9e, 0xfd, 0x71, 0x0a,
    0x79, 0xfe, 0x02, 0x20, 0x18, 0xfd, 0x7e, 0x08,
    0xfd, 0x77, 0x04, 0xfd, 0x7e, 0x00, 0xfd, 0x77,
    0x05, 0xfd, 0x7e, 0x01, 0xfd, 0x77, 0x06, 0xfd,
    0x7e, 0x02, 0xfd, 0x77, 0x07, 0xfd, 0x7e, 0x04,
    0xb7, 0x28, 0x33, 0xfd, 0x35, 0x04, 0xfd, 0x7e,
    0x00, 0xfd, 0xbe, 0x05, 0x20, 0x28, 0xfd, 0x7e,
    0x01, 0xfd, 0xbe, 0x06, 0x20, 0x20, 0xfd, 0x7e,
    0x02, 0xfd, 0xbe, 0x07, 0x20, 0x18, 0xfd, 0x34,
    0x07, 0xfd, 0x7e, 0x09, 0xfd, 0xbe, 0x07, 0x30,
    0x07, 0xfd, 0x34, 0x06, 0xfd, 0x36, 0x07, 0x01,
    0xfd, 0xcb, 0x0b, 0x96, 0x18, 0x08, 0xfd, 0x36,
    0x04, 0x00, 0xfd, 0xcb, 0x0b, 0xd6, 0xdd, 0x36,
    0x07, 0x00, 0xfd, 0x7e, 0x00, 0xcd, 0x5b, 0x06,
    0x23, 0x7e, 0xfd, 0x5e, 0x02, 0x1d, 0x53, 0xe6,
    0x18, 0x06, 0x00, 0xfe, 0x00, 0x28, 0x14, 0x06,
    0x01, 0xfe, 0x08, 0x28, 0x0c, 0x06, 0x03, 0xfe,
    0x10, 0x28, 0x04, 0x06, 0x07, 0xcb, 0x2b, 0xcb,
    0x2b, 0xcb, 0x2b, 0x1c, 0xfd, 0x73, 0x03, 0x78,
    0xa2, 0xfd, 0x77, 0x0c, 0xfd, 0xcb, 0x0b, 0x46,
    0xfd, 0xcb, 0x0b, 0xc6, 0x28, 0x22, 0xfd, 0x7e,
    0x00, 0xdd, 0xbe, 0x00, 0x20, 0x10, 0xfd, 0x7e,
    0x01, 0xdd, 0xbe, 0x01, 0x20, 0x08, 0xfd, 0x7e,
    0x03, 0xdd, 0xbe, 0x02, 0x28, 0x2a, 0xaf, 0xfd,
    0xcb, 0x0b, 0x4e, 0xc4, 0x0e, 0x03, 0xb7, 0xc0,
    0xfd, 0x7e, 0x00, 0xdd, 0x77, 0x00, 0xfd, 0x7e,
    0x01, 0xdd, 0x77, 0x01, 0xfd, 0x7e, 0x03, 0xdd,
    0x77, 0x02, 0xaf, 0xfd, 0xcb, 0x0b, 0x56, 0xc4,
    0x01, 0x03, 0xb7, 0xc0, 0xfd, 0xcb, 0x0b, 0x8e,
    0xfd, 0x6e, 0x0c, 0x26, 0x00, 0x29, 0x29, 0x29,
    0x29, 0x29, 0x29, 0x29, 0xdd, 0x5e, 0x05, 0xdd,
    0x56, 0x06, 0x19, 0xfd, 0x5e, 0x14, 0xfd, 0x56,
    0x15, 0x01, 0x80, 0x00, 0xfd, 0xcb, 0x0b, 0x5e,
    0x3e, 0x00, 0xc0, 0xfd, 0xcb, 0x0b, 0xce, 0xeb,
    0xc9, 0x21, 0xff, 0x04, 0xdd, 0x75, 0x0a, 0xdd,
    0x74, 0x0b, 0x0e, 0x06, 0x18, 0x0b, 0x21, 0x56,
    0x05, 0xdd, 0x75, 0x0a, 0xdd, 0x74, 0x0b, 0x0e,
    0x05, 0xdd, 0x7e, 0x00, 0xcd, 0x5b, 0x06, 0x23,
    0x7e, 0xe6, 0x40, 0xb1, 0xdd, 0x77, 0x16, 0xdd,
    0x36, 0x15, 0x09, 0x3e, 0x18, 0xa6, 0x0f, 0x0f,
    0x0f, 0xdd, 0x77, 0x1b, 0x06, 0x80, 0x11, 0x80,
    0x00, 0xfe, 0x00, 0x28, 0x11, 0x06, 0xff, 0x11,
    0x00, 0x00, 0xfe, 0x01, 0x28, 0x08, 0x16, 0x01,
    0xfe, 0x02, 0x28, 0x02, 0x16, 0x03, 0xdd, 0x70,
    0x1e, 0xdd, 0xcb, 0x0d, 0x7e, 0x20, 0x06, 0xdd,
    0x73, 0x0c, 0xdd, 0x72, 0x0d, 0xaf, 0xcb, 0x56,
    0xdd, 0x46, 0x01, 0xdd, 0x4e, 0x02, 0x28, 0x03,
    0xcb, 0x18, 0x17, 0xdd, 0x77, 0x19, 0x07, 0x07,
    0xdd, 0x77, 0x09, 0xdd, 0x70, 0x08, 0xdd, 0x70,
    0x18, 0xdd, 0x71, 0x1a, 0x3e, 0x03, 0xa6, 0xdd,
    0xb6, 0x09, 0xdd, 0x77, 0x17, 0x11, 0x05, 0x00,
    0x19, 0x7e, 0xdd, 0x77, 0x1c, 0x23, 0x7e, 0xdd,
    0x77, 0x1d, 0xdd, 0x46, 0x04, 0x16, 0x03, 0xc5,
    0xd5, 0xcd, 0x9c, 0x05, 0xfd, 0xe5, 0xcd, 0x3b,
    0x04, 0xcd, 0x6f, 0x04, 0xfd, 0xe1, 0x06, 0x07,
    0xdd, 0xe5, 0xe1, 0x11, 0x0e, 0x00, 0x19, 0xcd,
    0xce, 0x06, 0xed, 0xa2, 0x20, 0xf9, 0xd1, 0xc1,
    0xdd, 0x7e, 0x0e, 0xe6, 0xc0, 0x28, 0x35, 0xcd,
    0x35, 0x07, 0x3e, 0x01, 0xb9, 0x28, 0x1d, 0x3e,
    0x06, 0xb9, 0xcc, 0x22, 0x04, 0x3e, 0x02, 0xb9,
    0xcc, 0xfa, 0x03, 0x3e, 0x05, 0xb9, 0xcc, 0x09,
    0x04, 0xdd, 0x7e, 0x04, 0x0f, 0x3c, 0xb8, 0xcc,
    0x98, 0x05, 0x10, 0xb3, 0xdd, 0x7e, 0x07, 0xfe,
    0xfe, 0x28, 0x0a, 0xcd, 0x7b, 0x04, 0x38, 0x05,
    0xb7, 0xc2, 0x92, 0x03, 0x4f, 0xdd, 0x71, 0x07,
    0x79, 0xc9, 0xcd, 0x83, 0x05, 0xc8, 0xcd, 0x98,
    0x05, 0xdd, 0x7e, 0x08, 0xfe, 0x03, 0xd0, 0x18,
    0x0a, 0xdd, 0x7e, 0x08, 0xfe, 0x03, 0xd0, 0xcd,
    0x83, 0x05, 0xc8, 0xf5, 0xdd, 0x36, 0x08, 0x03,
    0xcd, 0x9c, 0x05, 0xf1, 0xdd, 0x77, 0x08, 0xc3,
    0x98, 0x05, 0xcd, 0x83, 0x05, 0xc8, 0xcd, 0x98,
    0x05, 0xdd, 0x7e, 0x08, 0xf5, 0xdd, 0x36, 0x08,
    0x0a, 0xcd, 0x9c, 0x05, 0xf1, 0xdd, 0x77, 0x08,
    0xc3, 0x9c, 0x05, 0x0e, 0xfb, 0xdd, 0xe5, 0xe1,
    0x11, 0x15, 0x00, 0x19, 0x46, 0x23, 0xcd, 0xb8,
    0x06, 0xcd, 0xc5, 0x06, 0xed, 0xa3, 0x20, 0xf9,
    0xdd, 0x46, 0x0c, 0xdd, 0x5e, 0x0d, 0xcb, 0xbb,
    0xdd, 0x56, 0x03, 0x15, 0xdd, 0x6e, 0x0a, 0xdd,
    0x66, 0x0b, 0xe5, 0xfd, 0xe1, 0xdd, 0x6e, 0x05,
    0xdd, 0x66, 0x06, 0xc9, 0xcb, 0x7f, 0xc0, 0xdb,
    0xfa, 0xcb, 0x6f, 0x28, 0xf7, 0xd5, 0x11, 0x00,
    0x00, 0xfd, 0xe9, 0x79, 0x07, 0x5f, 0x16, 0x00,
    0x21, 0x0c, 0x0f, 0x19, 0x5e, 0x23, 0x56, 0xc5,
    0xd5, 0x11, 0xb0, 0x0b, 0xcd, 0xc3, 0x01, 0x3e,
    0x41, 0xdd, 0x86, 0x00, 0xcd, 0xcc, 0x01, 0x11,
    0xae, 0x0e, 0xcd, 0xc3, 0x01, 0xd1, 0xcd, 0xc3,
    0x01, 0x3e, 0xff, 0xdd, 0xbe, 0x07, 0x37, 0xc4,
    0xbc, 0x04, 0xc1, 0xc9, 0xcd, 0xc3, 0x01, 0xcd,
    0xe2, 0x04, 0xfe, 0x0d, 0x20, 0xf9, 0x11, 0xab,
    0x0e, 0xc3, 0xc3, 0x01, 0x11, 0xc7, 0x0b, 0xcd,
    0xc3, 0x01, 0xcd, 0xe2, 0x04, 0xe6, 0xdf, 0x5f,
    0xfe, 0x41, 0x37, 0x28, 0x08, 0xfe, 0x52, 0x28,
    0x04, 0xd6, 0x49, 0x20, 0xed, 0xf5, 0x7b, 0xcd,
    0xcc, 0x01, 0x11, 0xab, 0x0e, 0xcd, 0xc3, 0x01,
    0xf1, 0xc9, 0xc5, 0xe5, 0xfd, 0x6e, 0x18, 0xfd,
    0x66, 0x19, 0xcd, 0xdd, 0x01, 0xe1, 0xc1, 0xc9,
    0xe6, 0x20, 0x28, 0x36, 0xdb, 0xfa, 0xfe, 0xf0,
    0x28, 0x0b, 0x1b, 0x7b, 0xb2, 0x28, 0x2d, 0xdb,
    0xfa, 0xfe, 0xf0, 0x20, 0xeb, 0xed, 0xa2, 0xd1,
    0xd5, 0x7b, 0xb2, 0x20, 0x01, 0x05, 0xfb, 0x76,
    0xdb, 0xfa, 0xe6, 0x20, 0x28, 0x14, 0xed, 0xa2,
    0x20, 0xf4, 0x1d, 0xf2, 0x09, 0x05, 0xd1, 0x15,
    0xd5, 0xf2, 0x09, 0x05, 0xfb, 0x76, 0xdb, 0xf7,
    0xed, 0xa2, 0xd1, 0xc9, 0xd1, 0xe1, 0xfd, 0xe1,
    0xfd, 0xe5, 0xc5, 0x0e, 0xf7, 0xfd, 0x7e, 0x12,
    0x47, 0xcb, 0xdf, 0xed, 0x79, 0xdb, 0xfa, 0xcb,
    0x6f, 0x20, 0xfa, 0xed, 0x41, 0xc1, 0xe9, 0xe6,
    0x20, 0x28, 0xdf, 0xdb, 0xfa, 0xfe, 0xb0, 0x28,
    0x0b, 0x1b, 0x7b, 0xb2, 0x28, 0xd6, 0xdb, 0xfa,
    0xfe, 0xb0, 0x20, 0xeb, 0xed, 0xa3, 0xd1, 0xd5,
    0x7b, 0xb2, 0x20, 0x01, 0x05, 0xfb, 0x76, 0xdb,
    0xfa, 0xe6, 0x20, 0x28, 0xbd, 0xed, 0xa3, 0x20,
    0xf4, 0x1d, 0xf2, 0x60, 0x05, 0xd1, 0x15, 0xd5,
    0xf2, 0x60, 0x05, 0xfb, 0x76, 0xdb, 0xf7, 0xed,
    0xa3, 0xd1, 0xc9, 0x04, 0x15, 0xc0, 0x06, 0x01,
    0xc9, 0xfd, 0xcb, 0x0b, 0x4e, 0x20, 0x04, 0xfd,
    0xcb, 0x0b, 0x86, 0xfd, 0x36, 0x01, 0x00, 0xc9,
    0x3e, 0x00, 0x18, 0x02, 0x3e, 0x08, 0xc5, 0xd5,
    0x4f, 0xdd, 0x7e, 0x00, 0xf5, 0xcd, 0x5b, 0x06,
    0x7e, 0xe6, 0x07, 0xcb, 0x57, 0x28, 0x01, 0x0f,
    0x5f, 0x23, 0xf1, 0xfd, 0xbe, 0x11, 0x28, 0x1b,
    0xfd, 0x77, 0x11, 0xcb, 0x7e, 0xc5, 0x28, 0x0f,
    0xfd, 0xbe, 0x13, 0xfd, 0x77, 0x13, 0xf5, 0xd5,
    0xe5, 0xc4, 0x48, 0x06, 0xe1, 0xd1, 0xf1, 0xcd,
    0xfa, 0x06, 0xc1, 0xcb, 0x76, 0x28, 0x0f, 0x3e,
    0x13, 0xcb, 0x6e, 0x28, 0x02, 0x3e, 0x27, 0xdd,
    0xbe, 0x08, 0x30, 0x02, 0xcb, 0xd3, 0xdb, 0xf5,
    0xe6, 0x03, 0x57, 0xa3, 0xf5, 0x7e, 0xe6, 0x03,
    0x47, 0x04, 0x3e, 0x08, 0x07, 0x10, 0xfd, 0xb2,
    0xb3, 0xd3, 0xf7, 0xfd, 0x77, 0x12, 0xcd, 0x6c,
    0x06, 0xdd, 0x7e, 0x08, 0xcb, 0x59, 0x28, 0x0a,
    0xb7, 0x20, 0x04, 0xcb, 0x99, 0x18, 0x03, 0xba,
    0x28, 0x06, 0xcd, 0xd7, 0x06, 0xf1, 0x37, 0xf5,
    0xf1, 0xc5, 0xe5, 0xcd, 0x74, 0x06, 0xe1, 0xc1,
    0xcb, 0x59, 0xcc, 0x28, 0x06, 0xd1, 0xc1, 0xc9,
    0xcd, 0xc5, 0x06, 0x3e, 0x04, 0xd3, 0xfb, 0xcd,
    0xc5, 0x06, 0x3e, 0x03, 0xa6, 0xd3, 0xfb, 0xcd,
    0xce, 0x06, 0xdb, 0xfb, 0xe6, 0x10, 0xc0, 0xcd,
    0xd7, 0x06, 0x3e, 0xff, 0xb7, 0x37, 0x18, 0x2c,
    0x3c, 0xcb, 0xf7, 0xfd, 0x5e, 0x0f, 0xfd, 0x56,
    0x10, 0x12, 0xfd, 0x5e, 0x0d, 0xfd, 0x56, 0x0e,
    0xc3, 0xac, 0x04, 0xfd, 0xe5, 0xe1, 0x11, 0x1a,
    0x00, 0x19, 0x57, 0x07, 0x07, 0x07, 0x82, 0x5f,
    0x16, 0x00, 0x19, 0xc9, 0xe5, 0x11, 0x07, 0x00,
    0x19, 0x56, 0xe1, 0xc9, 0x23, 0xfd, 0x46, 0x12,
    0x0e, 0xf7, 0x56, 0x1e, 0x00, 0x23, 0x30, 0x28,
    0xed, 0x5e, 0x3e, 0x00, 0xed, 0x47, 0x20, 0x0c,
    0xfb, 0x1b, 0x7a, 0xb3, 0x28, 0x08, 0x0e, 0xf7,
    0xed, 0x41, 0x18, 0xf5, 0x56, 0xfb, 0xed, 0x41,
    0x18, 0xfc, 0xed, 0x46, 0x33, 0x33, 0xcd, 0x28,
    0x07, 0x7a, 0xbe, 0x30, 0x04, 0x56, 0x18, 0x01,
    0xc0, 0xed, 0x41, 0xed, 0x41, 0x00, 0x1b, 0x7a,
    0xb3, 0xc2, 0xa9, 0x06, 0xc9, 0xcd, 0x11, 0x07,
    0xdb, 0xfa, 0xe6, 0x0f, 0x20, 0xf7, 0xdb, 0xfa,
    0xcb, 0x67, 0x20, 0xf4, 0xc9, 0xdb, 0xfa, 0xe6,
    0xc0, 0xfe, 0x80, 0x20, 0xf8, 0xc9, 0xdb, 0xfa,
    0xe6, 0xc0, 0xfe, 0xc0, 0x20, 0xf8, 0xc9, 0xcd,
    0xc5, 0x06, 0x3e, 0x07, 0xb1, 0xd3, 0xfb, 0xcd,
    0xc5, 0x06, 0x3e, 0x03, 0xa6, 0xcb, 0x59, 0x28,
    0x03, 0xdd, 0xb6, 0x09, 0xd3, 0xfb, 0xcb, 0x59,
    0xc8, 0xcd, 0xc5, 0x06, 0xdd, 0x7e, 0x08, 0xd3,
    0xfb, 0xc9, 0xe5, 0x23, 0x23, 0x23, 0xcd, 0xc5,
    0x06, 0x3e, 0x03, 0xd3, 0xfb, 0x01, 0xfb, 0x02,
    0xcd, 0xc5, 0x06, 0xed, 0xa3, 0x20, 0xf9, 0xe1,
    0xc9, 0xcd, 0xc5, 0x06, 0x3e, 0x08, 0xd3, 0xfb,
    0xcd, 0xce, 0x06, 0xdb, 0xfb, 0xe6, 0xc0, 0xfe,
    0x80, 0xc8, 0xcd, 0xce, 0x06, 0xdb, 0xfb, 0xc9,
    0xcd, 0x11, 0x07, 0xe5, 0xd5, 0x11, 0x05, 0x00,
    0x19, 0x77, 0xd1, 0xe1, 0xc9, 0x0e, 0x01, 0xdd,
    0xcb, 0x0f, 0x4e, 0xc0, 0x0c, 0xdd, 0xcb, 0x10,
    0x4e, 0xc0, 0xdd, 0xcb, 0x10, 0x66, 0xc0, 0x0c,
    0xdd, 0xcb, 0x10, 0x6e, 0xc0, 0x0c, 0xdd, 0xcb,
    0x0f, 0x6e, 0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x7e,
    0xc0, 0xdd, 0xcb, 0x0f, 0x56, 0xc0, 0xdd, 0xcb,
    0x0f, 0x46, 0xc0, 0xdd, 0xcb, 0x10, 0x76, 0xc0,
    0xdd, 0xcb, 0x10, 0x46, 0xc0, 0x0c, 0xdd, 0xcb,
    0x0e, 0x5e, 0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x66,
    0xc0, 0x0c, 0xdd, 0xcb, 0x0e, 0x66, 0xc0, 0x0c,
    0xdd, 0xcb, 0x0e, 0x7e, 0xc0, 0x0c, 0xc9, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x13, 0x22, 0x25, 0xff,
    0xff, 0x11, 0x41, 0x0d, 0xcd, 0xc3, 0x01, 0xcd,
    0xe2, 0x04, 0xfe, 0x31, 0xfa, 0x07, 0x08, 0xfe,
    0x3a, 0xf2, 0x07, 0x08, 0xf5, 0xcd, 0xcc, 0x01,
    0x11, 0xab, 0x0e, 0xcd, 0xc3, 0x01, 0xf1, 0xd6,
    0x30, 0xfe, 0x04, 0xf2, 0x29, 0x08, 0x32, 0xfe,
    0xff, 0x3d, 0x07, 0x4f, 0xaf, 0x47, 0x21, 0x3b,
    0x08, 0x09, 0x5e, 0x23, 0x56, 0xeb, 0xcd, 0xdd,
    0x01, 0x18, 0xc6, 0x4d, 0x08, 0x4d, 0x08, 0x4d,
    0x08, 0xa3, 0x08, 0xcd, 0x08, 0x21, 0x09, 0xc9,
    0x09, 0x1a, 0x0a, 0x3d, 0x0a, 0x3e, 0x20, 0x32,
    0xfd, 0xff, 0x3e, 0x5f, 0x47, 0x3a, 0xfd, 0xff,
    0xcd, 0x80, 0x08, 0x05, 0x28, 0x08, 0x3c, 0xf2,
    0x58, 0x08, 0x3e, 0x20, 0x18, 0xf2, 0xcd, 0x9e,
    0x08, 0xc0, 0x3e, 0x0d, 0xcd, 0x80, 0x08, 0x3e,
    0x0a, 0xcd, 0x80, 0x08, 0x3a, 0xfd, 0xff, 0x3c,
    0xfa, 0x4d, 0x08, 0x32, 0xfd, 0xff, 0x18, 0xd2,
    0xf5, 0xc5, 0x4f, 0x3a, 0xfe, 0xff, 0xfe, 0x01,
    0x20, 0x05, 0xcd, 0x75, 0x01, 0x18, 0x0c, 0xfe,
    0x02, 0x20, 0x05, 0xcd, 0x71, 0x01, 0x18, 0x03,
    0xcd, 0x98, 0x01, 0xc1, 0xf1, 0xc9, 0xdb, 0xfd,
    0xcb, 0x4f, 0xc9, 0xdb, 0xfe, 0xaf, 0x57, 0x4f,
    0xcd, 0x71, 0x01, 0xcd, 0x85, 0x01, 0xba, 0x20,
    0x0f, 0xcd, 0x9e, 0x08, 0x20, 0x05, 0x7a, 0x3c,
    0xf2, 0xa6, 0x08, 0x11, 0x49, 0x0e, 0x18, 0x03,
    0x11, 0x3f, 0x0e, 0xcd, 0xc3, 0x01, 0x11, 0x34,
    0x0e, 0xcd, 0xc3, 0x01, 0xc9, 0x11, 0x85, 0x0c,
    0xcd, 0xc3, 0x01, 0xcd, 0xee, 0x08, 0x21, 0x00,
    0x00, 0x01, 0xff, 0xfb, 0xcd, 0x2a, 0xfc, 0xb7,
    0x28, 0x03, 0xcd, 0xf6, 0x08, 0xcd, 0xfe, 0x08,
    0xcd, 0x9e, 0x08, 0x28, 0xe9, 0xc9, 0xaf, 0x32,
    0xfe, 0xff, 0x32, 0xfd, 0xff, 0xc9, 0x3a, 0xfd,
    0xff, 0x3c, 0x32, 0xfd, 0xff, 0xc9, 0x11, 0x15,
    0x0e, 0xcd, 0xc3, 0x01, 0x3a, 0xfe, 0xff, 0x3c,
    0x32, 0xfe, 0xff, 0xcd, 0xb3, 0xfc, 0x11, 0xae,
    0x0e, 0xcd, 0xc3, 0x01, 0x3a, 0xfd, 0xff, 0xcd,
    0xb3, 0xfc, 0x11, 0x24, 0x0e, 0xcd, 0xc3, 0x01,
    0xc9, 0xcd, 0x3e, 0x09, 0xdd, 0x36, 0x01, 0x27,
    0xdd, 0x36, 0x03, 0x05, 0xcd, 0x74, 0x09, 0xcd,
    0xee, 0x08, 0xcd, 0x9b, 0x09, 0xcd, 0xac, 0x09,
    0xcd, 0x9e, 0x08, 0x28, 0xf5, 0xc9, 0xcd, 0x5d,
    0x01, 0x11, 0x53, 0x0e, 0xcd, 0xc3, 0x01, 0xcd,
    0xe2, 0x04, 0xe6, 0xdf, 0xd6, 0x41, 0x38, 0xf7,
    0xfe, 0x04, 0x30, 0xf3, 0xdd, 0x77, 0x00, 0xf5,
    0x11, 0x6e, 0x0e, 0xcd, 0xc3, 0x01, 0xf1, 0xc6,
    0x41, 0xcd, 0xcc, 0x01, 0x11, 0xde, 0x0c, 0xcd,
    0xc3, 0x01, 0xcd, 0xe2, 0x04, 0x11, 0xab, 0x0e,
    0xcd, 0xc3, 0x01, 0xc9, 0x01, 0x5f, 0xaa, 0x21,
    0x00, 0x80, 0xe5, 0x11, 0x02, 0x80, 0x71, 0x23,
    0x70, 0x2b, 0x01, 0x00, 0x08, 0xed, 0xb0, 0x01,
    0xb6, 0x6d, 0x71, 0x23, 0x70, 0x2b, 0x01, 0x00,
    0x0c, 0xed, 0xb0, 0xc1, 0xdd, 0x71, 0x05, 0xdd,
    0x70, 0x06, 0xc9, 0x11, 0x96, 0x0e, 0xcd, 0xc3,
    0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0x0e, 0x03,
    0xcd, 0xc0, 0x09, 0xc9, 0x11, 0xa3, 0x0e, 0xcd,
    0xc3, 0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0x01,
    0x03, 0xcd, 0xc0, 0x09, 0xcd, 0xfe, 0x08, 0xc9,
    0xdd, 0x7e, 0x07, 0xfe, 0x00, 0xc4, 0xf6, 0x08,
    0xc9, 0xcd, 0x3e, 0x09, 0x06, 0x00, 0x0e, 0x27,
    0xdd, 0x36, 0x05, 0x00, 0xdd, 0x36, 0x06, 0x80,
    0xdd, 0x36, 0x03, 0x01, 0xcd, 0xee, 0x08, 0xc5,
    0xcd, 0xfd, 0x09, 0xc1, 0xcd, 0xed, 0x09, 0xcd,
    0x9e, 0x08, 0x28, 0xf3, 0xc9, 0x04, 0x0d, 0xfa,
    0xf3, 0x09, 0xc9, 0x78, 0x41, 0x4f, 0xc5, 0xcd,
    0xfe, 0x08, 0xc1, 0x18, 0xf0, 0xc5, 0xdd, 0x70,
    0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0x01, 0x03,
    0xcd, 0xc0, 0x09, 0xc1, 0xdd, 0x71, 0x01, 0xdd,
    0x36, 0x07, 0xff, 0xcd, 0x01, 0x03, 0xcd, 0xc0,
    0x09, 0xc9, 0xcd, 0x5d, 0x01, 0xaf, 0xdd, 0x77,
    0x00, 0xdd, 0x77, 0x01, 0xdd, 0x36, 0x03, 0x01,
    0xdd, 0x36, 0x05, 0x00, 0xdd, 0x36, 0x06, 0x80,
    0xdd, 0x36, 0x07, 0xfe, 0xcd, 0x01, 0x03, 0xcd,
    0x9e, 0x08, 0x28, 0xf4, 0xc9, 0xf1, 0xc9, 0xdb,
    0xf6, 0x3a, 0x00, 0x00, 0xfe, 0xc3, 0xd3, 0xf6,
    0x20, 0x03, 0xc3, 0x47, 0x00, 0x11, 0x85, 0x0c,
    0xcd, 0xc3, 0x01, 0x21, 0x00, 0x00, 0x01, 0xff,
    0xfb, 0xcd, 0x2a, 0xfc, 0xb7, 0xc2, 0xd4, 0x00,
    0x11, 0xa9, 0x0c, 0xcd, 0xc3, 0x01, 0xc3, 0x47,
    0x00, 0xdb, 0xf6, 0x22, 0x00, 0xfd, 0x21, 0xd2,
    0xfc, 0x56, 0x23, 0xcb, 0x7e, 0x20, 0x18, 0x5e,
    0xcb, 0x73, 0xcb, 0xb3, 0x23, 0xe5, 0x28, 0x05,
    0xcd, 0x54, 0xfc, 0x18, 0x07, 0xdd, 0x21, 0x74,
    0xfc, 0xcd, 0x58, 0xfc, 0xe1, 0x18, 0xe2, 0xaf,
    0xd3, 0xf6, 0xc9, 0xdd, 0x21, 0x76, 0xfc, 0x2a,
    0x00, 0xfd, 0x7b, 0x85, 0x6f, 0x7c, 0xce, 0x00,
    0x67, 0xcd, 0x72, 0xfc, 0x23, 0x23, 0x23, 0x78,
    0xbc, 0xd8, 0x20, 0xf5, 0x79, 0xbd, 0xd8, 0x18,
    0xf0, 0xdd, 0xe9, 0x72, 0xc9, 0x7e, 0xba, 0x20,
    0x01, 0xc9, 0xd3, 0xf6, 0xdb, 0xf5, 0xcb, 0x6f,
    0xcc, 0x8a, 0xfc, 0xf1, 0xf1, 0xf1, 0x3e, 0xff,
    0xc9, 0xd5, 0x11, 0x1a, 0x0d, 0xcd, 0xc3, 0x01,
    0xdb, 0xf6, 0x7e, 0xd3, 0xf6, 0xcd, 0xb3, 0xfc,
    0x11, 0x24, 0x0d, 0xcd, 0xc3, 0x01, 0xd1, 0x7a,
    0xcd, 0xb3, 0xfc, 0x11, 0x2f, 0x0d, 0xcd, 0xc3,
    0x01, 0x7c, 0xcd, 0xb3, 0xfc, 0x7d, 0xcd, 0xb3,
    0xfc, 0xc9, 0xf5, 0x0f, 0x0f, 0x0f, 0x0f, 0xe6,
    0x0f, 0xcd, 0xc4, 0xfc, 0xf1, 0xe6, 0x0f, 0xcd,
    0xc4, 0xfc, 0xc9, 0xfe, 0x0a, 0xfa, 0xcb, 0xfc,
    0xc6, 0x07, 0xc6, 0x30, 0x4f, 0xcd, 0xcc, 0x01,
    0xc9, 0xff, 0x00, 0x00, 0x01, 0xff, 0x02, 0x00,
    0x41, 0xff, 0x42, 0xff, 0x40, 0x00, 0x00, 0xff,
    0x01, 0x00, 0x02, 0x00, 0x42, 0x00, 0x40, 0xff,
    0x41, 0xaa, 0x00, 0x55, 0x02, 0xaa, 0x01, 0xaa,
    0x41, 0x55, 0x01, 0xaa, 0x40, 0x55, 0x00, 0x55,
    0x42, 0x55, 0x41, 0x55, 0x40, 0x00, 0x80, 0x57,
    0x72, 0x6f, 0x6e, 0x67, 0x20, 0x74, 0x72, 0x61,
    0x63, 0x6b, 0xae, 0x00, 0x57, 0x72, 0x69, 0x74,
    0x65, 0x20, 0x70, 0x72, 0x6f, 0x74, 0x65, 0x63,
    0x74, 0x65, 0x64, 0xae, 0x00, 0x44, 0x61, 0x74,
    0x61, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0xae,
    0x00, 0x4e, 0x6f, 0x74, 0x20, 0x66, 0x6f, 0x75,
    0x6e, 0x64, 0xae, 0x00, 0x4c, 0x6f, 0x73, 0x74,
    0x20, 0x64, 0x61, 0x74, 0x61, 0xae, 0x00, 0x44,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x6e, 0x6f, 0x74,
    0x20, 0x72, 0x65, 0x61, 0x64, 0x79, 0xae, 0x00,
    0x49, 0x6e, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20,
    0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e, 0x64, 0xae,
    0x00, 0x55, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e,
    0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0xae, 0x00,
    0x0d, 0x0a, 0x44, 0x69, 0x73, 0x6b, 0x20, 0x65,
    0x72, 0x72, 0x6f, 0x72, 0x20, 0x6f, 0x6e, 0x20,
    0x64, 0x72, 0x69, 0x76, 0x65, 0xa0, 0x00, 0x0d,
    0x0a, 0x54, 0x79, 0x70, 0x65, 0x20, 0x52, 0x20,
    0x74, 0x6f, 0x20, 0x74, 0x72, 0x79, 0x20, 0x61,
    0x67, 0x61, 0x69, 0x6e, 0x2c, 0x20, 0x41, 0x20,
    0x74, 0x6f, 0x20, 0x61, 0x62, 0x6f, 0x72, 0x74,
    0x2c, 0x20, 0x6f, 0x72, 0x20, 0x49, 0x20, 0x74,
    0x6f, 0x20, 0x69, 0x67, 0x6e, 0x6f, 0x72, 0x65,
    0x3a, 0xa0, 0x00, 0x0a, 0x45, 0x72, 0x72, 0x6f,
    0x72, 0x20, 0x6f, 0x6e, 0x20, 0x43, 0x50, 0x2f,
    0x4d, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
    0x20, 0x64, 0x69, 0x73, 0x6b, 0x65, 0x74, 0x74,
    0x65, 0x2e, 0x00, 0x0d, 0x0a, 0x50, 0x75, 0x73,
    0x68, 0x20, 0x72, 0x65, 0x73, 0x65, 0x74, 0x20,
    0x74, 0x6f, 0x20, 0x74, 0x72, 0x79, 0x20, 0x61,
    0x67, 0x61, 0x69, 0x6e, 0x2e, 0xa0, 0x00, 0x0d,
    0x4d, 0x69, 0x63, 0x72, 0x6f, 0x2d, 0x44, 0x65,
    0x63, 0x69, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x2d,
    0x2d, 0x20, 0x52, 0x4f, 0x4d, 0x20, 0x52, 0x65,
    0x76, 0x2e, 0x20, 0x32, 0x2e, 0x35, 0x0d, 0x0a,
    0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
    0x74, 0x20, 0x31, 0x39, 0x38, 0x32, 0x2c, 0x20,
    0x31, 0x39, 0x38, 0x33, 0x20, 0x4d, 0x6f, 0x72,
    0x72, 0x6f, 0x77, 0x20, 0x44, 0x65, 0x73, 0x69,
    0x67, 0x6e, 0x73, 0x2c, 0x20, 0x49, 0x6e, 0x63,
    0x2e, 0x0d, 0x0a, 0x0a, 0x00, 0x54, 0x65, 0x73,
    0x74, 0x69, 0x6e, 0x67, 0x20, 0x6d, 0x65, 0x6d,
    0x6f, 0x72, 0x79, 0x20, 0x2d, 0x2d, 0x20, 0x70,
    0x6c, 0x65, 0x61, 0x73, 0x65, 0x20, 0x73, 0x74,
    0x61, 0x6e, 0x64, 0x20, 0x62, 0x79, 0x3a, 0x20,
    0x00, 0x4d, 0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20,
    0x4f, 0x2e, 0x4b, 0x2e, 0x0d, 0x0a, 0x0a, 0x00,
    0x49, 0x6e, 0x73, 0x65, 0x72, 0x74, 0x20, 0x43,
    0x50, 0x2f, 0x4d, 0x20, 0x73, 0x79, 0x73, 0x74,
    0x65, 0x6d, 0x20, 0x64, 0x69, 0x73, 0x6b, 0x65,
    0x74, 0x74, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x44,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x41, 0x20, 0x61,
    0x6e, 0x64, 0x20, 0x70, 0x72, 0x65, 0x73, 0x73,
    0x20, 0x5b, 0x52, 0x45, 0x54, 0x55, 0x52, 0x4e,
    0x5d, 0x20, 0x07, 0x00, 0x07, 0x07, 0x52, 0x4f,
    0x4d, 0x20, 0x4d, 0x65, 0x6d, 0x6f, 0x72, 0x79,
    0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00, 0x07,
    0x07, 0x52, 0x41, 0x4d, 0x20, 0x4d, 0x65, 0x6d,
    0x6f, 0x72, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f,
    0x72, 0x00, 0x07, 0x07, 0x0d, 0x0a, 0x52, 0x65,
    0x61, 0x64, 0x20, 0x00, 0x20, 0x65, 0x78, 0x70,
    0x65, 0x63, 0x74, 0x65, 0x64, 0x20, 0x00, 0x20,
    0x61, 0x74, 0x20, 0x52, 0x41, 0x4d, 0x20, 0x6c,
    0x6f, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
    0x00, 0x0d, 0x0a, 0x44, 0x69, 0x61, 0x67, 0x6e,
    0x6f, 0x73, 0x74, 0x69, 0x63, 0x73, 0x20, 0x4d,
    0x65, 0x6e, 0x75, 0x0d, 0x0a, 0x0a, 0x31, 0x29,
    0x20, 0x50, 0x6f, 0x72, 0x74, 0x20, 0x31, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x5c, 0x0d, 0x0a, 0x32, 0x29, 0x20, 0x50,
    0x6f, 0x72, 0x74, 0x20, 0x32, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x3e, 0x20, 0x42, 0x61, 0x72, 0x62, 0x65, 0x72,
    0x2d, 0x50, 0x6f, 0x6c, 0x65, 0x20, 0x70, 0x61,
    0x74, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x74, 0x65,
    0x73, 0x74, 0x0d, 0x0a, 0x33, 0x29, 0x20, 0x43,
    0x65, 0x6e, 0x74, 0x72, 0x6f, 0x6e, 0x69, 0x63,
    0x73, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x20, 0x2f,
    0x0d, 0x0a, 0x34, 0x29, 0x20, 0x4c, 0x6f, 0x6f,
    0x70, 0x20, 0x62, 0x61, 0x63, 0x6b, 0x20, 0x6f,
    0x6e, 0x20, 0x70, 0x6f, 0x72, 0x74, 0x20, 0x32,
    0x0d, 0x0a, 0x35, 0x29, 0x20, 0x52, 0x41, 0x4d,
    0x20, 0x74, 0x65, 0x73, 0x74, 0x0d, 0x0a, 0x36,
    0x29, 0x20, 0x46, 0x44, 0x43, 0x20, 0x52, 0x2f,
    0x57, 0x0d, 0x0a, 0x37, 0x29, 0x20, 0x46, 0x44,
    0x43, 0x20, 0x73, 0x65, 0x65, 0x6b, 0x20, 0x74,
    0x65, 0x73, 0x74, 0x0d, 0x0a, 0x38, 0x29, 0x20,
    0x56, 0x46, 0x4f, 0x20, 0x74, 0x65, 0x73, 0x74,
    0x0d, 0x0a, 0x39, 0x29, 0x20, 0x42, 0x6f, 0x6f,
    0x74, 0x0d, 0x0a, 0x0a, 0x45, 0x6e, 0x74, 0x65,
    0x72, 0x20, 0x23, 0x3a, 0x00, 0x0d, 0x0a, 0x45,
    0x6e, 0x64, 0x20, 0x6f, 0x66, 0x20, 0x70, 0x61,
    0x73, 0x73, 0x20, 0x00, 0x20, 0x65, 0x72, 0x72,
    0x6f, 0x72, 0x73, 0x20, 0x73, 0x6f, 0x20, 0x66,
    0x61, 0x72, 0x2e, 0x00, 0x6c, 0x6f, 0x6f, 0x70,
    0x20, 0x74, 0x65, 0x73, 0x74, 0x2e, 0x00, 0x0d,
    0x0a, 0x46, 0x61, 0x69, 0x6c, 0x65, 0x64, 0x20,
    0x00, 0x0d, 0x0a, 0x50, 0x61, 0x73, 0x73, 0x65,
    0x64, 0x20, 0x00, 0x45, 0x6e, 0x74, 0x65, 0x72,
    0x20, 0x64, 0x72, 0x69, 0x76, 0x65, 0x20, 0x74,
    0x6f, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x28,
    0x41, 0x2d, 0x44, 0x29, 0x3a, 0x00, 0x0d, 0x0a,
    0x49, 0x6e, 0x73, 0x65, 0x72, 0x74, 0x20, 0x61,
    0x20, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x74,
    0x65, 0x64, 0x20, 0x64, 0x69, 0x73, 0x6b, 0x65,
    0x74, 0x74, 0x65, 0x20, 0x69, 0x6e, 0x20, 0x64,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x00, 0x0d, 0x0a,
    0x57, 0x72, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x2e,
    0x2e, 0x2e, 0x00, 0x72, 0x65, 0x61, 0x64, 0x69,
    0x6e, 0x67, 0x00, 0x0d, 0x0a, 0x00, 0x3a, 0xa0,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x14, 0x00, 0xfe,
    0x00, 0x00, 0x00, 0xff, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x46,
    0x00, 0x00, 0x00, 0x01, 0x03, 0x05, 0x1c, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x75, 0x01,
    0x89, 0x01, 0x01, 0xd8, 0x58, 0x05, 0x6f, 0x03,
    0x05, 0x1c, 0xff, 0x02, 0x59, 0x58, 0x05, 0x6f,
    0x03, 0x05, 0x1c, 0xff, 0x04, 0x5a, 0x58, 0x05,
    0x6f, 0x03, 0x05, 0x1c, 0xff, 0x04, 0x5b, 0x58,
    0x05, 0x6f, 0x03, 0x05, 0x1c, 0xff, 0x4c, 0x0b,
    0x3f, 0x0b, 0x5d, 0x0b, 0x5d, 0x0b, 0x69, 0x0b,
    0x7f, 0x0b, 0x74, 0x0b, 0x7f, 0x0b, 0x90, 0x0b,
    0xa1, 0x0b, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    };

/* MD3 3.1 ROM is 4096 bytes */
static uint8 mmd_rom_31[MMD_ROM_SIZE] = {
    0xc3, 0x2c, 0x00, 0xc3, 0x68, 0x01, 0xc3, 0x30,
    0x05, 0xc3, 0xa4, 0x02, 0xc3, 0xb1, 0x02, 0xc3,
    0x35, 0x03, 0xc3, 0xa6, 0x04, 0xc3, 0xfd, 0x04,
    0xc3, 0x79, 0x00, 0x87, 0x0f, 0xc3, 0x40, 0x01,
    0xc3, 0x82, 0x01, 0xc3, 0x94, 0x01, 0xc3, 0x2d,
    0x08, 0xc3, 0x12, 0x07, 0xed, 0x46, 0x31, 0xfe,
    0xff, 0xfd, 0x21, 0xa6, 0x0f, 0xcd, 0x85, 0x00,
    0x11, 0x03, 0x0d, 0xcd, 0x51, 0x01, 0xcd, 0xb9,
    0x00, 0xcd, 0xd7, 0x00, 0xcd, 0x00, 0xfc, 0xdb,
    0xfc, 0xdb, 0xf5, 0xcb, 0x6f, 0xcc, 0xef, 0x08,
    0xcd, 0x08, 0x01, 0xcd, 0x3f, 0x05, 0xdd, 0x36,
    0x07, 0xfe, 0xcd, 0x35, 0x03, 0xfe, 0x06, 0x20,
    0x10, 0x11, 0x8e, 0x0d, 0xcd, 0x68, 0x01, 0xcd,
    0x89, 0x04, 0x11, 0x81, 0x0f, 0xcd, 0x68, 0x01,
    0x3d, 0xb7, 0xc4, 0x35, 0x03, 0xb7, 0xca, 0x00,
    0xfe, 0x11, 0xc7, 0x0c, 0xc3, 0xc9, 0x00, 0x2a,
    0x06, 0xfc, 0x37, 0x0b, 0x0b, 0x21, 0x93, 0x00,
    0x06, 0x13, 0x4e, 0x23, 0x7e, 0x23, 0xed, 0x79,
    0x10, 0xf8, 0xc9, 0xf3, 0x3e, 0xf0, 0xff, 0xf0,
    0xff, 0xf3, 0x7e, 0xf1, 0x0d, 0xf1, 0x00, 0xf3,
    0xbe, 0xf2, 0x68, 0xf2, 0x00, 0xfd, 0x80, 0xff,
    0x80, 0xfd, 0x80, 0xff, 0x80, 0xfd, 0x40, 0xff,
    0x40, 0xfd, 0xce, 0xff, 0xce, 0xfd, 0x37, 0xff,
    0x37, 0x21, 0x00, 0x00, 0x01, 0xfd, 0x0f, 0x11,
    0xca, 0x0d, 0xcd, 0xfb, 0x00, 0xc8, 0xcd, 0x68,
    0x01, 0xdb, 0xf5, 0xcb, 0x6f, 0xca, 0x47, 0x00,
    0x11, 0xe7, 0x0c, 0xcd, 0x68, 0x01, 0x76, 0x21,
    0x0b, 0x0b, 0x11, 0x00, 0xfc, 0x01, 0x00, 0x01,
    0xed, 0xb0, 0x01, 0xf7, 0x02, 0xd5, 0xe1, 0x13,
    0x36, 0xff, 0xed, 0xb0, 0x3a, 0x82, 0x00, 0x77,
    0x21, 0x00, 0xfc, 0x01, 0xf8, 0x03, 0x11, 0xdd,
    0x0d, 0x18, 0xc7, 0xaf, 0xf5, 0xf1, 0xae, 0x23,
    0x0b, 0xf5, 0x78, 0xb1, 0x20, 0xf7, 0xf1, 0xc9,
    0x21, 0x87, 0x0f, 0x11, 0x00, 0xff, 0x01, 0x5d,
    0x00, 0xed, 0xb0, 0xdd, 0x21, 0x00, 0xff, 0xfd,
    0x21, 0x1f, 0xff, 0xc9, 0x06, 0xff, 0x18, 0x02,
    0x06, 0xfd, 0x79, 0x48, 0xed, 0x40, 0xcb, 0x40,
    0x28, 0xfa, 0x0d, 0xed, 0x79, 0xc9, 0x0e, 0xff,
    0x18, 0x02, 0x0e, 0xfd, 0xed, 0x78, 0xcb, 0x4f,
    0x28, 0xfa, 0x0d, 0xed, 0x78, 0xe6, 0x7f, 0xc9,
    0xdb, 0xf5, 0xcb, 0x67, 0x20, 0xfa, 0x79, 0xd3,
    0xf4, 0x3e, 0x80, 0xd3, 0xf5, 0xaf, 0xd3, 0xf5,
    0xc9, 0x21, 0x00, 0x00, 0x2b, 0x7c, 0xb5, 0x20,
    0xfb, 0xcd, 0x68, 0x01, 0x06, 0x32, 0x3e, 0x0a,
    0xcd, 0x71, 0x01, 0x10, 0xf9, 0x11, 0x09, 0x0d,
    0x1a, 0xb7, 0xc8, 0xcd, 0x71, 0x01, 0x13, 0x18,
    0xf7, 0x4f, 0xc5, 0xd5, 0xe5, 0xfd, 0x6e, 0x16,
    0xfd, 0x66, 0x17, 0xcd, 0x01, 0x08, 0xe1, 0xd1,
    0xc1, 0xc9, 0xfd, 0x36, 0x04, 0x00, 0xfd, 0xcb,
    0x0b, 0xde, 0xfd, 0xcb, 0x0b, 0xd6, 0xfd, 0x36,
    0x0a, 0x02, 0x18, 0x65, 0xfd, 0xcb, 0x0b, 0x9e,
    0xfd, 0x71, 0x0a, 0x79, 0xfe, 0x02, 0x20, 0x18,
    0xfd, 0x7e, 0x08, 0xfd, 0x77, 0x04, 0xfd, 0x7e,
    0x00, 0xfd, 0x77, 0x05, 0xfd, 0x7e, 0x01, 0xfd,
    0x77, 0x06, 0xfd, 0x7e, 0x02, 0xfd, 0x77, 0x07,
    0xfd, 0x7e, 0x04, 0xb7, 0x28, 0x33, 0xfd, 0x35,
    0x04, 0xfd, 0x7e, 0x00, 0xfd, 0xbe, 0x05, 0x20,
    0x28, 0xfd, 0x7e, 0x01, 0xfd, 0xbe, 0x06, 0x20,
    0x20, 0xfd, 0x7e, 0x02, 0xfd, 0xbe, 0x07, 0x20,
    0x18, 0xfd, 0x34, 0x07, 0xfd, 0x7e, 0x09, 0xfd,
    0xbe, 0x07, 0x30, 0x07, 0xfd, 0x34, 0x06, 0xfd,
    0x36, 0x07, 0x01, 0xfd, 0xcb, 0x0b, 0x96, 0x18,
    0x08, 0xfd, 0x36, 0x04, 0x00, 0xfd, 0xcb, 0x0b,
    0xd6, 0xfd, 0x7e, 0x00, 0xcd, 0xf3, 0x05, 0x23,
    0x7e, 0x2b, 0xfd, 0x5e, 0x02, 0x1d, 0x53, 0xe6,
    0x18, 0x06, 0x00, 0xfe, 0x00, 0x28, 0x14, 0x06,
    0x01, 0xfe, 0x08, 0x28, 0x0c, 0x06, 0x03, 0xfe,
    0x10, 0x28, 0x04, 0x06, 0x07, 0xcb, 0x2b, 0xcb,
    0x2b, 0xcb, 0x2b, 0xcb, 0x76, 0x20, 0x01, 0x1c,
    0xfd, 0x73, 0x03, 0x78, 0xa2, 0xfd, 0x77, 0x0c,
    0xfd, 0xcb, 0x0b, 0x46, 0xfd, 0xcb, 0x0b, 0xc6,
    0x28, 0x22, 0xfd, 0x7e, 0x00, 0xdd, 0xbe, 0x00,
    0x20, 0x10, 0xfd, 0x7e, 0x01, 0xdd, 0xbe, 0x01,
    0x20, 0x08, 0xfd, 0x7e, 0x03, 0xdd, 0xbe, 0x02,
    0x28, 0x2a, 0xaf, 0xfd, 0xcb, 0x0b, 0x4e, 0xc4,
    0xb1, 0x02, 0xb7, 0xc0, 0xfd, 0x7e, 0x00, 0xdd,
    0x77, 0x00, 0xfd, 0x7e, 0x01, 0xdd, 0x77, 0x01,
    0xfd, 0x7e, 0x03, 0xdd, 0x77, 0x02, 0xaf, 0xfd,
    0xcb, 0x0b, 0x56, 0xc4, 0xa4, 0x02, 0xb7, 0xc0,
    0xfd, 0xcb, 0x0b, 0x8e, 0xfd, 0x6e, 0x0c, 0x26,
    0x00, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29,
    0xdd, 0x5e, 0x05, 0xdd, 0x56, 0x06, 0x19, 0xfd,
    0x5e, 0x14, 0xfd, 0x56, 0x15, 0x01, 0x80, 0x00,
    0xaf, 0xfd, 0xcb, 0x0b, 0x5e, 0xc0, 0xfd, 0xcb,
    0x0b, 0xce, 0xeb, 0xc9, 0x21, 0xa6, 0x04, 0xdd,
    0x75, 0x0a, 0xdd, 0x74, 0x0b, 0x0e, 0x06, 0x18,
    0x0b, 0x21, 0xfd, 0x04, 0xdd, 0x75, 0x0a, 0xdd,
    0x74, 0x0b, 0x0e, 0x05, 0xdd, 0x7e, 0x00, 0xcd,
    0xf3, 0x05, 0x23, 0x7e, 0xe6, 0x40, 0xb1, 0xdd,
    0x77, 0x16, 0xdd, 0x36, 0x15, 0x09, 0x3e, 0x18,
    0xa6, 0x0f, 0x0f, 0x0f, 0xdd, 0x77, 0x1b, 0x06,
    0x80, 0x11, 0x80, 0x00, 0xfe, 0x00, 0x28, 0x11,
    0x06, 0xff, 0x11, 0x00, 0x00, 0xfe, 0x01, 0x28,
    0x08, 0x16, 0x01, 0xfe, 0x02, 0x28, 0x02, 0x16,
    0x03, 0xdd, 0x70, 0x1e, 0xdd, 0xcb, 0x0d, 0x7e,
    0x20, 0x06, 0xdd, 0x73, 0x0c, 0xdd, 0x72, 0x0d,
    0xaf, 0xcb, 0x56, 0xdd, 0x46, 0x01, 0xdd, 0x4e,
    0x02, 0x28, 0x03, 0xcb, 0x18, 0x17, 0xdd, 0x77,
    0x19, 0x07, 0x07, 0xdd, 0x77, 0x09, 0xdd, 0x70,
    0x08, 0xdd, 0x70, 0x18, 0xdd, 0x71, 0x1a, 0x3e,
    0x03, 0xa6, 0xdd, 0xb6, 0x09, 0xdd, 0x77, 0x17,
    0x11, 0x05, 0x00, 0x19, 0x7e, 0xdd, 0x77, 0x1c,
    0x23, 0x7e, 0xdd, 0x77, 0x1d, 0xdd, 0x46, 0x04,
    0x16, 0x03, 0xc5, 0xd5, 0xcd, 0x43, 0x05, 0xfd,
    0xe5, 0xcd, 0xd2, 0x03, 0xcd, 0x06, 0x04, 0xfd,
    0xe1, 0x06, 0x07, 0xdd, 0xe5, 0xe1, 0x11, 0x0e,
    0x00, 0x19, 0xcd, 0x5e, 0x06, 0xed, 0xa2, 0x20,
    0xf9, 0xd1, 0xc1, 0xdd, 0x7e, 0x0e, 0xe6, 0xc0,
    0x28, 0x35, 0xcd, 0xc5, 0x06, 0x3e, 0x01, 0xb9,
    0x28, 0x1d, 0x3e, 0x06, 0xb9, 0xcc, 0xb9, 0x03,
    0x3e, 0x02, 0xb9, 0xcc, 0x9d, 0x03, 0x3e, 0x05,
    0xb9, 0xcc, 0xa0, 0x03, 0xdd, 0x7e, 0x04, 0x0f,
    0x3c, 0xb8, 0xcc, 0x3f, 0x05, 0x10, 0xb3, 0xdd,
    0x7e, 0x07, 0xfe, 0xfe, 0x28, 0x0a, 0xcd, 0x12,
    0x04, 0x38, 0x05, 0xb7, 0xc2, 0x35, 0x03, 0x4f,
    0xdd, 0x71, 0x07, 0x79, 0xc9, 0xcd, 0x3f, 0x05,
    0xcd, 0x2a, 0x05, 0xc8, 0xdd, 0x7e, 0x08, 0xfe,
    0x03, 0xd0, 0xf5, 0xdd, 0x36, 0x08, 0x03, 0xcd,
    0x43, 0x05, 0xf1, 0xdd, 0x77, 0x08, 0xc3, 0x3f,
    0x05, 0xcd, 0x2a, 0x05, 0xc8, 0xcd, 0x3f, 0x05,
    0xdd, 0x7e, 0x08, 0xf5, 0xdd, 0x36, 0x08, 0x0a,
    0xcd, 0x43, 0x05, 0xf1, 0xdd, 0x77, 0x08, 0xc3,
    0x43, 0x05, 0x0e, 0xfb, 0xdd, 0xe5, 0xe1, 0x11,
    0x15, 0x00, 0x19, 0x46, 0x23, 0xcd, 0x48, 0x06,
    0xcd, 0x55, 0x06, 0xed, 0xa3, 0x20, 0xf9, 0xdd,
    0x46, 0x0c, 0xdd, 0x5e, 0x0d, 0xcb, 0xbb, 0xdd,
    0x56, 0x03, 0x15, 0xdd, 0x6e, 0x0a, 0xdd, 0x66,
    0x0b, 0xe5, 0xfd, 0xe1, 0xdd, 0x6e, 0x05, 0xdd,
    0x66, 0x06, 0xc9, 0xcb, 0x7f, 0xc0, 0xdb, 0xfa,
    0xcb, 0x6f, 0x28, 0xf7, 0xd5, 0x11, 0x00, 0x80,
    0xfd, 0xe9, 0x79, 0x07, 0x5f, 0x16, 0x00, 0x21,
    0xe2, 0x0f, 0x19, 0x5e, 0x23, 0x56, 0xc5, 0xd5,
    0x11, 0x7c, 0x0c, 0xcd, 0x68, 0x01, 0x3e, 0x41,
    0xdd, 0x86, 0x00, 0xcd, 0x71, 0x01, 0x11, 0x84,
    0x0f, 0xcd, 0x68, 0x01, 0xd1, 0xcd, 0x68, 0x01,
    0x3e, 0xff, 0xdd, 0xbe, 0x07, 0x37, 0xc4, 0x63,
    0x04, 0xc1, 0xc9, 0x3c, 0xcb, 0xf7, 0xfd, 0x5e,
    0x0f, 0xfd, 0x56, 0x10, 0x12, 0xfd, 0x5e, 0x0d,
    0xfd, 0x56, 0x0e, 0xcd, 0x68, 0x01, 0xcd, 0x89,
    0x04, 0xfe, 0x0d, 0x20, 0xf9, 0x11, 0x81, 0x0f,
    0xc3, 0x68, 0x01, 0x11, 0x93, 0x0c, 0xcd, 0x68,
    0x01, 0xcd, 0x89, 0x04, 0xe6, 0xdf, 0x5f, 0xfe,
    0x41, 0x37, 0x28, 0x08, 0xfe, 0x52, 0x28, 0x04,
    0xd6, 0x49, 0x20, 0xed, 0xf5, 0x7b, 0xcd, 0x71,
    0x01, 0x11, 0x81, 0x0f, 0xcd, 0x68, 0x01, 0xf1,
    0xc9, 0xc5, 0xe5, 0xfd, 0x6e, 0x18, 0xfd, 0x66,
    0x19, 0xcd, 0x01, 0x08, 0xe1, 0xc1, 0xc9, 0xe6,
    0x20, 0x28, 0x36, 0xdb, 0xfa, 0xfe, 0xf0, 0x28,
    0x0b, 0x1b, 0x7b, 0xb2, 0x28, 0x2d, 0xdb, 0xfa,
    0xfe, 0xf0, 0x20, 0xeb, 0xed, 0xa2, 0xd1, 0xd5,
    0x7b, 0xb2, 0x20, 0x01, 0x05, 0xfb, 0x76, 0xdb,
    0xfa, 0xe6, 0x20, 0x28, 0x14, 0xed, 0xa2, 0x20,
    0xf4, 0x1d, 0xf2, 0xb0, 0x04, 0xd1, 0x15, 0xd5,
    0xf2, 0xb0, 0x04, 0xfb, 0x76, 0xdb, 0xf7, 0xed,
    0xa2, 0xd1, 0xc9, 0xd1, 0xe1, 0xfd, 0xe1, 0xfd,
    0xe5, 0xc5, 0x0e, 0xf7, 0xfd, 0x7e, 0x12, 0x47,
    0xcb, 0xdf, 0xed, 0x79, 0xdb, 0xfa, 0xcb, 0x6f,
    0x20, 0xfa, 0xed, 0x41, 0xc1, 0xe9, 0xe6, 0x20,
    0x28, 0xdf, 0xdb, 0xfa, 0xfe, 0xb0, 0x28, 0x0b,
    0x1b, 0x7b, 0xb2, 0x28, 0xd6, 0xdb, 0xfa, 0xfe,
    0xb0, 0x20, 0xeb, 0xed, 0xa3, 0xd1, 0xd5, 0x7b,
    0xb2, 0x20, 0x01, 0x05, 0xfb, 0x76, 0xdb, 0xfa,
    0xe6, 0x20, 0x28, 0xbd, 0xed, 0xa3, 0x20, 0xf4,
    0x1d, 0xf2, 0x07, 0x05, 0xd1, 0x15, 0xd5, 0xf2,
    0x07, 0x05, 0xfb, 0x76, 0xdb, 0xf7, 0xed, 0xa3,
    0xd1, 0xc9, 0x04, 0x15, 0xc0, 0x06, 0x01, 0xc9,
    0xfd, 0xcb, 0x0b, 0x4e, 0x20, 0x04, 0xfd, 0xcb,
    0x0b, 0x86, 0xfd, 0x36, 0x01, 0x00, 0xc9, 0x3e,
    0x00, 0x18, 0x02, 0x3e, 0x08, 0xc5, 0xd5, 0x4f,
    0xdd, 0x7e, 0x00, 0xf5, 0xcd, 0xf3, 0x05, 0x7e,
    0xe6, 0x07, 0xcb, 0x57, 0x28, 0x01, 0x0f, 0x5f,
    0x23, 0xf1, 0xfd, 0xbe, 0x11, 0x28, 0x1b, 0xfd,
    0x77, 0x11, 0xcb, 0x7e, 0xc5, 0x28, 0x0f, 0xfd,
    0xbe, 0x13, 0xfd, 0x77, 0x13, 0xf5, 0xd5, 0xe5,
    0xc4, 0x43, 0x04, 0xe1, 0xd1, 0xf1, 0xcd, 0x8a,
    0x06, 0xc1, 0xcb, 0x76, 0x28, 0x0f, 0x3e, 0x13,
    0xcb, 0x6e, 0x28, 0x02, 0x3e, 0x27, 0xdd, 0xbe,
    0x08, 0x30, 0x02, 0xcb, 0xd3, 0xdb, 0xf5, 0xe6,
    0x03, 0x57, 0xa3, 0xf5, 0x7e, 0xe6, 0x03, 0x47,
    0x04, 0x3e, 0x08, 0x07, 0x10, 0xfd, 0xb2, 0xb3,
    0xd3, 0xf7, 0xfd, 0x77, 0x12, 0xe5, 0x11, 0x07,
    0x00, 0x19, 0x56, 0xe1, 0xdd, 0x7e, 0x08, 0xcb,
    0x59, 0x28, 0x0a, 0xb7, 0x20, 0x04, 0xcb, 0x99,
    0x18, 0x03, 0xba, 0x28, 0x06, 0xcd, 0x67, 0x06,
    0xf1, 0x37, 0xf5, 0xf1, 0xc5, 0xe5, 0xcd, 0x04,
    0x06, 0xe1, 0xc1, 0xcb, 0x59, 0xcc, 0xd3, 0x05,
    0xd1, 0xc1, 0xc9, 0xcd, 0x55, 0x06, 0x3e, 0x04,
    0xd3, 0xfb, 0xcd, 0x55, 0x06, 0x3e, 0x03, 0xa6,
    0xd3, 0xfb, 0xcd, 0x5e, 0x06, 0xdb, 0xfb, 0xe6,
    0x10, 0xc0, 0xcd, 0x67, 0x06, 0x3e, 0xff, 0xb7,
    0x37, 0x18, 0x11, 0xfd, 0xe5, 0xe1, 0x11, 0x1a,
    0x00, 0x19, 0x57, 0x07, 0x07, 0x07, 0x82, 0x5f,
    0x16, 0x00, 0x19, 0xc9, 0x23, 0xfd, 0x46, 0x12,
    0x0e, 0xf7, 0x56, 0x1e, 0x00, 0x23, 0x30, 0x28,
    0xed, 0x5e, 0x3e, 0x00, 0xed, 0x47, 0x20, 0x0c,
    0xfb, 0x1b, 0x7a, 0xb3, 0x28, 0x08, 0x0e, 0xf7,
    0xed, 0x41, 0x18, 0xf5, 0x56, 0xfb, 0xed, 0x41,
    0x18, 0xfc, 0xed, 0x46, 0x33, 0x33, 0xcd, 0xb8,
    0x06, 0x7a, 0xbe, 0x30, 0x04, 0x56, 0x18, 0x01,
    0xc0, 0xed, 0x41, 0xed, 0x41, 0x00, 0x1b, 0x7a,
    0xb3, 0xc2, 0x39, 0x06, 0xc9, 0xcd, 0xa1, 0x06,
    0xdb, 0xfa, 0xe6, 0x0f, 0x20, 0xf7, 0xdb, 0xfa,
    0xcb, 0x67, 0x20, 0xf4, 0xc9, 0xdb, 0xfa, 0xe6,
    0xc0, 0xfe, 0x80, 0x20, 0xf8, 0xc9, 0xdb, 0xfa,
    0xe6, 0xc0, 0xfe, 0xc0, 0x20, 0xf8, 0xc9, 0xcd,
    0x55, 0x06, 0x3e, 0x07, 0xb1, 0xd3, 0xfb, 0xcd,
    0x55, 0x06, 0x3e, 0x03, 0xa6, 0xcb, 0x59, 0x28,
    0x03, 0xdd, 0xb6, 0x09, 0xd3, 0xfb, 0xcb, 0x59,
    0xc8, 0xcd, 0x55, 0x06, 0xdd, 0x7e, 0x08, 0xd3,
    0xfb, 0xc9, 0xe5, 0x23, 0x23, 0x23, 0xcd, 0x55,
    0x06, 0x3e, 0x03, 0xd3, 0xfb, 0x01, 0xfb, 0x02,
    0xcd, 0x55, 0x06, 0xed, 0xa3, 0x20, 0xf9, 0xe1,
    0xc9, 0xcd, 0x55, 0x06, 0x3e, 0x08, 0xd3, 0xfb,
    0xcd, 0x5e, 0x06, 0xdb, 0xfb, 0xe6, 0xc0, 0xfe,
    0x80, 0xc8, 0xcd, 0x5e, 0x06, 0xdb, 0xfb, 0xc9,
    0xcd, 0xa1, 0x06, 0xe5, 0xd5, 0x11, 0x05, 0x00,
    0x19, 0x77, 0xd1, 0xe1, 0xc9, 0x0e, 0x01, 0xdd,
    0xcb, 0x0f, 0x4e, 0xc0, 0x0c, 0xdd, 0xcb, 0x10,
    0x4e, 0xc0, 0xdd, 0xcb, 0x10, 0x66, 0xc0, 0x0c,
    0xdd, 0xcb, 0x10, 0x6e, 0xc0, 0x0c, 0xdd, 0xcb,
    0x0f, 0x6e, 0xc0, 0x0c, 0xdd, 0xcb, 0x0f, 0x7e,
    0xc0, 0xdd, 0xcb, 0x0f, 0x56, 0xc0, 0xdd, 0xcb,
    0x0f, 0x46, 0xc0, 0xdd, 0xcb, 0x10, 0x46, 0xc0,
    0x0c, 0xdd, 0xcb, 0x0e, 0x5e, 0xc0, 0x0c, 0xdd,
    0xcb, 0x0f, 0x66, 0xc0, 0x0c, 0xdd, 0xcb, 0x0e,
    0x66, 0xc0, 0x0c, 0xdd, 0xcb, 0x0e, 0x7e, 0xc0,
    0x0c, 0xc9, 0xfd, 0x6e, 0x08, 0xfd, 0x66, 0x09,
    0xfd, 0xe5, 0xe5, 0xfd, 0xe1, 0xfd, 0x7e, 0x12,
    0xb7, 0x20, 0x2a, 0x3e, 0x1f, 0xb9, 0xda, 0xee,
    0x07, 0x3e, 0x1b, 0xb9, 0x20, 0x07, 0xfd, 0x36,
    0x12, 0x01, 0xfd, 0xe1, 0xc9, 0xfd, 0xe1, 0xfd,
    0x6e, 0x02, 0xfd, 0x66, 0x03, 0xcd, 0x02, 0x08,
    0xb7, 0xca, 0xc3, 0x07, 0xfd, 0x6e, 0x06, 0xfd,
    0x66, 0x07, 0xc3, 0xde, 0x07, 0xfe, 0x01, 0x28,
    0x54, 0xfe, 0x02, 0x28, 0x71, 0xfd, 0x7e, 0x14,
    0x81, 0xfd, 0x77, 0x17, 0xfd, 0x36, 0x12, 0x00,
    0xfd, 0xe5, 0xe1, 0x11, 0x16, 0x00, 0x19, 0xeb,
    0x1a, 0xfd, 0x86, 0x13, 0x12, 0xfd, 0x7e, 0x15,
    0xf5, 0x1f, 0x9f, 0x47, 0x7b, 0x90, 0x5f, 0xfd,
    0xe5, 0xe1, 0xcd, 0xe3, 0x07, 0x1a, 0x4f, 0xf1,
    0xcb, 0x4f, 0xf5, 0xc4, 0x15, 0x08, 0xcd, 0xf0,
    0x07, 0xd5, 0x11, 0x06, 0x00, 0xcd, 0xf2, 0x07,
    0xd1, 0x7b, 0x80, 0x04, 0x80, 0x5f, 0x1a, 0x4f,
    0xf1, 0xc4, 0x15, 0x08, 0xcd, 0xf0, 0x07, 0x11,
    0x0c, 0x00, 0xf1, 0x18, 0x4d, 0x3e, 0x3d, 0xb9,
    0x28, 0x2d, 0xfd, 0x36, 0x12, 0x00, 0xfd, 0xe1,
    0xfd, 0x6e, 0x00, 0xfd, 0x66, 0x01, 0xcd, 0x02,
    0x08, 0xb7, 0x20, 0x13, 0xc5, 0x0e, 0x1b, 0xcd,
    0xf0, 0x07, 0xc1, 0x48, 0x18, 0x2a, 0xfd, 0x71,
    0x16, 0xfd, 0x36, 0x12, 0x03, 0x18, 0x0c, 0xfd,
    0x6e, 0x04, 0xfd, 0x66, 0x05, 0x18, 0x07, 0xfd,
    0x36, 0x12, 0x02, 0xfd, 0xe1, 0xc9, 0x09, 0x5e,
    0x23, 0x56, 0xeb, 0x7e, 0xfe, 0xff, 0xc8, 0x4f,
    0xcd, 0xf0, 0x07, 0x23, 0x18, 0xf5, 0xfd, 0xe1,
    0xdd, 0xe9, 0xfd, 0xe5, 0xe1, 0x19, 0x18, 0xeb,
    0x3e, 0x05, 0xd1, 0xc9, 0x13, 0x22, 0x31, 0xff,
    0xff, 0xe9, 0x41, 0x0e, 0x00, 0x7e, 0xb8, 0x20,
    0x04, 0xaf, 0x47, 0x3d, 0xc9, 0xd6, 0xff, 0xc8,
    0x23, 0x0c, 0x0c, 0x18, 0xf0, 0xc5, 0x79, 0x0e,
    0x30, 0xd6, 0x0a, 0x38, 0x03, 0x0c, 0x18, 0xf9,
    0x47, 0x3e, 0x30, 0xb9, 0xc4, 0xf0, 0x07, 0x3e,
    0x3a, 0x80, 0xc1, 0x4f, 0xc9, 0xe5, 0x3e, 0x04,
    0xb9, 0x38, 0xc5, 0xfd, 0x71, 0x00, 0xcb, 0x43,
    0x20, 0x7d, 0x79, 0xcd, 0xf3, 0x05, 0xcb, 0x7e,
    0x20, 0x75, 0x23, 0xe5, 0xc5, 0xcd, 0xbb, 0x08,
    0xc1, 0xe1, 0xb7, 0x20, 0xab, 0xe5, 0xdd, 0x6e,
    0x05, 0xdd, 0x66, 0x06, 0x11, 0x99, 0x00, 0x19,
    0x06, 0x19, 0x1e, 0x00, 0x3e, 0x00, 0x2b, 0xae,
    0x57, 0xb3, 0x5f, 0x7a, 0x10, 0xf8, 0x23, 0xb7,
    0x7b, 0x11, 0xe0, 0x08, 0x20, 0x19, 0xb7, 0x28,
    0x16, 0xe5, 0xdd, 0x6e, 0x05, 0xdd, 0x66, 0x06,
    0x11, 0x89, 0x00, 0x19, 0xeb, 0xe1, 0xcb, 0x56,
    0x28, 0x05, 0xe1, 0xcb, 0xd6, 0x18, 0x03, 0xe1,
    0xcb, 0x96, 0xaf, 0x26, 0x00, 0x69, 0x29, 0x29,
    0x29, 0x29, 0xc1, 0x09, 0xe5, 0xd5, 0x11, 0x0a,
    0x00, 0x19, 0x5e, 0x23, 0x56, 0xe1, 0xd5, 0x20,
    0x05, 0x01, 0x0f, 0x00, 0xed, 0xb0, 0xe1, 0x7e,
    0xfd, 0x77, 0x09, 0x23, 0x23, 0x23, 0x7e, 0x3c,
    0xfd, 0x77, 0x08, 0xd1, 0x3e, 0x00, 0xc9, 0xf6,
    0xff, 0x18, 0xd0, 0xc5, 0xaf, 0xfd, 0xcb, 0x0b,
    0x4e, 0xc4, 0xb1, 0x02, 0xfd, 0xcb, 0x0b, 0x8e,
    0xfd, 0xcb, 0x0b, 0x86, 0xc1, 0xb7, 0xc0, 0xdd,
    0x71, 0x00, 0xdd, 0x36, 0x01, 0x00, 0xdd, 0x36,
    0x02, 0x01, 0xcd, 0x30, 0x05, 0xc3, 0xa4, 0x02,
    0x28, 0x00, 0x04, 0x0f, 0x01, 0x5e, 0x00, 0x7f,
    0x00, 0xc0, 0x00, 0x20, 0x00, 0x02, 0x00, 0x11,
    0x17, 0x0e, 0xcd, 0x68, 0x01, 0xcd, 0x89, 0x04,
    0xfe, 0x31, 0x38, 0xf9, 0xfe, 0x3a, 0x30, 0xf5,
    0xf5, 0xcd, 0x71, 0x01, 0x11, 0x81, 0x0f, 0xcd,
    0x68, 0x01, 0xf1, 0xd6, 0x30, 0x32, 0xff, 0xff,
    0x3d, 0x07, 0x4f, 0xaf, 0x47, 0x21, 0x22, 0x09,
    0x09, 0x5e, 0x23, 0x56, 0xeb, 0xcd, 0x01, 0x08,
    0x18, 0xcd, 0x34, 0x09, 0x34, 0x09, 0x34, 0x09,
    0x7e, 0x09, 0xa8, 0x09, 0xf8, 0x09, 0x97, 0x0a,
    0xe8, 0x0a, 0x06, 0x01, 0x21, 0xfe, 0xff, 0x36,
    0x20, 0x06, 0x4f, 0x7e, 0xcd, 0x5b, 0x09, 0x3c,
    0xf2, 0x45, 0x09, 0x3e, 0x20, 0x10, 0xf5, 0xcd,
    0x79, 0x09, 0xc0, 0x3e, 0x0d, 0xcd, 0x5b, 0x09,
    0x3e, 0x0a, 0xcd, 0x5b, 0x09, 0x34, 0xfa, 0x34,
    0x09, 0x18, 0xde, 0xf5, 0xc5, 0x4f, 0x3a, 0xff,
    0xff, 0xfe, 0x01, 0x20, 0x05, 0xcd, 0x20, 0x01,
    0x18, 0x0c, 0xfe, 0x02, 0x20, 0x05, 0xcd, 0x1c,
    0x01, 0x18, 0x03, 0xcd, 0x40, 0x01, 0xc1, 0xf1,
    0xc9, 0xdb, 0xfd, 0xcb, 0x4f, 0xc9, 0xdb, 0xfe,
    0xaf, 0x57, 0x4f, 0xcd, 0x1c, 0x01, 0xcd, 0x2e,
    0x01, 0xba, 0x20, 0x0f, 0xcd, 0x79, 0x09, 0x20,
    0x05, 0x7a, 0x3c, 0xf2, 0x81, 0x09, 0x11, 0x1f,
    0x0f, 0x18, 0x03, 0x11, 0x15, 0x0f, 0xcd, 0x68,
    0x01, 0x11, 0x0a, 0x0f, 0xcd, 0x68, 0x01, 0xc9,
    0x11, 0x5b, 0x0d, 0xcd, 0x68, 0x01, 0xcd, 0xc6,
    0x09, 0x21, 0x00, 0x00, 0x01, 0xff, 0xfb, 0xcd,
    0x27, 0xfc, 0xcd, 0xce, 0x09, 0xcd, 0xd5, 0x09,
    0xcd, 0x79, 0x09, 0x28, 0xec, 0xc9, 0xaf, 0x32,
    0xff, 0xff, 0x32, 0xfe, 0xff, 0xc9, 0xb7, 0xc8,
    0x21, 0xfe, 0xff, 0x34, 0xc9, 0x11, 0xeb, 0x0e,
    0xcd, 0x68, 0x01, 0x3a, 0xff, 0xff, 0x3c, 0x32,
    0xff, 0xff, 0xcd, 0xb8, 0xfc, 0x11, 0x84, 0x0f,
    0xcd, 0x68, 0x01, 0x3a, 0xfe, 0xff, 0xcd, 0xb8,
    0xfc, 0x11, 0xfa, 0x0e, 0xcd, 0x68, 0x01, 0xc9,
    0xcd, 0x15, 0x0a, 0xdd, 0x36, 0x01, 0x27, 0xdd,
    0x36, 0x03, 0x05, 0xcd, 0x4b, 0x0a, 0xcd, 0xc6,
    0x09, 0xcd, 0x72, 0x0a, 0xcd, 0x83, 0x0a, 0xcd,
    0x79, 0x09, 0x28, 0xf5, 0xc9, 0xcd, 0x08, 0x01,
    0x11, 0x29, 0x0f, 0xcd, 0x68, 0x01, 0xcd, 0x89,
    0x04, 0xe6, 0xdf, 0xd6, 0x41, 0x38, 0xf7, 0xfe,
    0x04, 0x30, 0xf3, 0xdd, 0x77, 0x00, 0xf5, 0x11,
    0x44, 0x0f, 0xcd, 0x68, 0x01, 0xf1, 0xc6, 0x41,
    0xcd, 0x71, 0x01, 0x11, 0xb4, 0x0d, 0xcd, 0x68,
    0x01, 0xcd, 0x89, 0x04, 0x11, 0x81, 0x0f, 0xcd,
    0x68, 0x01, 0xc9, 0x01, 0x5f, 0xaa, 0x21, 0x00,
    0x80, 0xe5, 0x11, 0x02, 0x80, 0x71, 0x23, 0x70,
    0x2b, 0x01, 0x00, 0x08, 0xed, 0xb0, 0x01, 0xb6,
    0x6d, 0x71, 0x23, 0x70, 0x2b, 0x01, 0x00, 0x0c,
    0xed, 0xb0, 0xc1, 0xdd, 0x71, 0x05, 0xdd, 0x70,
    0x06, 0xc9, 0x11, 0x6c, 0x0f, 0xcd, 0x68, 0x01,
    0xdd, 0x36, 0x07, 0xff, 0xcd, 0xb1, 0x02, 0xcd,
    0xce, 0x09, 0xc9, 0x11, 0x79, 0x0f, 0xcd, 0x68,
    0x01, 0xdd, 0x36, 0x07, 0xff, 0xcd, 0xa4, 0x02,
    0xcd, 0xce, 0x09, 0xcd, 0xd5, 0x09, 0xc9, 0xcd,
    0x15, 0x0a, 0x06, 0x00, 0x0e, 0x27, 0xdd, 0x36,
    0x05, 0x00, 0xdd, 0x36, 0x06, 0x80, 0xdd, 0x36,
    0x03, 0x01, 0xcd, 0xc6, 0x09, 0xc5, 0xcd, 0xcb,
    0x0a, 0xc1, 0xcd, 0xbb, 0x0a, 0xcd, 0x79, 0x09,
    0x28, 0xf3, 0xc9, 0x04, 0x0d, 0xfa, 0xc1, 0x0a,
    0xc9, 0x78, 0x41, 0x4f, 0xc5, 0xcd, 0xd5, 0x09,
    0xc1, 0x18, 0xf0, 0xc5, 0xdd, 0x70, 0x01, 0xdd,
    0x36, 0x07, 0xff, 0xcd, 0xa4, 0x02, 0xcd, 0xce,
    0x09, 0xc1, 0xdd, 0x71, 0x01, 0xdd, 0x36, 0x07,
    0xff, 0xcd, 0xa4, 0x02, 0xcd, 0xce, 0x09, 0xc9,
    0xcd, 0x08, 0x01, 0xaf, 0xdd, 0x77, 0x00, 0xdd,
    0x77, 0x01, 0xdd, 0x36, 0x03, 0x01, 0xdd, 0x36,
    0x05, 0x00, 0xdd, 0x36, 0x06, 0x80, 0xdd, 0x36,
    0x07, 0xfe, 0xcd, 0xa4, 0x02, 0xcd, 0x79, 0x09,
    0x28, 0xf4, 0xc9, 0xdb, 0xf6, 0x3a, 0x00, 0x00,
    0xfe, 0xc3, 0xd3, 0xf6, 0xc8, 0x11, 0x5b, 0x0d,
    0xcd, 0x68, 0x01, 0x21, 0x00, 0x00, 0x01, 0xff,
    0xfb, 0xcd, 0x27, 0xfc, 0xb7, 0x11, 0xdd, 0x0d,
    0xc2, 0xc6, 0x00, 0x11, 0x7f, 0x0d, 0xcd, 0x68,
    0x01, 0xc9, 0xdb, 0xf6, 0x22, 0x00, 0xfd, 0x21,
    0xd2, 0xfc, 0x56, 0x23, 0xcb, 0x7e, 0x20, 0x18,
    0x5e, 0xcb, 0x73, 0xcb, 0xb3, 0x23, 0xe5, 0x28,
    0x05, 0xcd, 0x56, 0xfc, 0x18, 0x07, 0xdd, 0x21,
    0x76, 0xfc, 0xcd, 0x5a, 0xfc, 0xe1, 0x18, 0xe2,
    0x3e, 0xc3, 0x32, 0x00, 0x00, 0xaf, 0xd3, 0xf6,
    0xc9, 0xdd, 0x21, 0x78, 0xfc, 0x2a, 0x00, 0xfd,
    0x7b, 0x85, 0x6f, 0x7c, 0xce, 0x00, 0x67, 0xcd,
    0x74, 0xfc, 0x23, 0x23, 0x23, 0x78, 0xbc, 0xd8,
    0x20, 0xf5, 0x79, 0xbd, 0xd8, 0x18, 0xf0, 0xdd,
    0xe9, 0x72, 0xc9, 0x7e, 0xba, 0x20, 0x01, 0xc9,
    0x5f, 0xd3, 0xf6, 0xdb, 0xf5, 0xcb, 0x6f, 0xcc,
    0x8d, 0xfc, 0xf1, 0xf1, 0xf1, 0x3e, 0xff, 0xc9,
    0xd5, 0x11, 0xf0, 0x0d, 0xcd, 0x68, 0x01, 0xdb,
    0xf6, 0xd1, 0xd5, 0x7b, 0xd3, 0xf6, 0xcd, 0xb8,
    0xfc, 0x11, 0xfa, 0x0d, 0xcd, 0x68, 0x01, 0xd1,
    0x7a, 0xcd, 0xb8, 0xfc, 0x11, 0x05, 0x0e, 0xcd,
    0x68, 0x01, 0x7c, 0xcd, 0xb8, 0xfc, 0x7d, 0xcd,
    0xb8, 0xfc, 0xc9, 0xf5, 0x0f, 0x0f, 0x0f, 0x0f,
    0xe6, 0x0f, 0xcd, 0xc5, 0xfc, 0xf1, 0xe6, 0x0f,
    0xfe, 0x0a, 0x38, 0x02, 0xc6, 0x07, 0xc6, 0x30,
    0x4f, 0xcd, 0x71, 0x01, 0xc9, 0xff, 0x00, 0x00,
    0x01, 0xff, 0x02, 0x00, 0x41, 0xff, 0x42, 0xff,
    0x40, 0x00, 0x00, 0xff, 0x01, 0x00, 0x02, 0x00,
    0x42, 0x00, 0x40, 0xff, 0x41, 0xaa, 0x00, 0x55,
    0x02, 0xaa, 0x01, 0xaa, 0x41, 0x55, 0x01, 0xaa,
    0x40, 0x55, 0x00, 0x55, 0x42, 0x55, 0x41, 0x55,
    0x40, 0x00, 0x80, 0x57, 0x72, 0x6f, 0x6e, 0x67,
    0x20, 0x74, 0x72, 0x61, 0x63, 0x6b, 0xae, 0x00,
    0x57, 0x72, 0x69, 0x74, 0x65, 0x20, 0x70, 0x72,
    0x6f, 0x74, 0x65, 0x63, 0x74, 0x65, 0x64, 0xae,
    0x00, 0x44, 0x61, 0x74, 0x61, 0x20, 0x65, 0x72,
    0x72, 0x6f, 0x72, 0xae, 0x00, 0x4e, 0x6f, 0x74,
    0x20, 0x66, 0x6f, 0x75, 0x6e, 0x64, 0xae, 0x00,
    0x4c, 0x6f, 0x73, 0x74, 0x20, 0x64, 0x61, 0x74,
    0x61, 0xae, 0x00, 0x44, 0x72, 0x69, 0x76, 0x65,
    0x20, 0x6e, 0x6f, 0x74, 0x20, 0x72, 0x65, 0x61,
    0x64, 0x79, 0xae, 0x00, 0x49, 0x6e, 0x76, 0x61,
    0x6c, 0x69, 0x64, 0x20, 0x63, 0x6f, 0x6d, 0x6d,
    0x61, 0x6e, 0x64, 0xae, 0x00, 0x55, 0x6e, 0x6b,
    0x6e, 0x6f, 0x77, 0x6e, 0x20, 0x65, 0x72, 0x72,
    0x6f, 0x72, 0xae, 0x00, 0x0d, 0x0a, 0x44, 0x69,
    0x73, 0x6b, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72,
    0x20, 0x6f, 0x6e, 0x20, 0x64, 0x72, 0x69, 0x76,
    0x65, 0xa0, 0x00, 0x0d, 0x0a, 0x54, 0x79, 0x70,
    0x65, 0x20, 0x52, 0x20, 0x74, 0x6f, 0x20, 0x74,
    0x72, 0x79, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e,
    0x2c, 0x20, 0x41, 0x20, 0x74, 0x6f, 0x20, 0x61,
    0x62, 0x6f, 0x72, 0x74, 0x2c, 0x20, 0x6f, 0x72,
    0x20, 0x49, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x67,
    0x6e, 0x6f, 0x72, 0x65, 0x3a, 0xa0, 0x00, 0x0a,
    0x45, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x6f, 0x6e,
    0x20, 0x43, 0x50, 0x2f, 0x4d, 0x20, 0x73, 0x79,
    0x73, 0x74, 0x65, 0x6d, 0x20, 0x64, 0x69, 0x73,
    0x6b, 0x65, 0x74, 0x74, 0x65, 0x2e, 0x00, 0x0d,
    0x0a, 0x50, 0x75, 0x73, 0x68, 0x20, 0x72, 0x65,
    0x73, 0x65, 0x74, 0x20, 0x74, 0x6f, 0x20, 0x74,
    0x72, 0x79, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6e,
    0x2e, 0xa0, 0x00, 0x1b, 0x25, 0x1b, 0x47, 0x30,
    0x00, 0x0d, 0x4d, 0x69, 0x63, 0x72, 0x6f, 0x2d,
    0x44, 0x65, 0x63, 0x69, 0x73, 0x69, 0x6f, 0x6e,
    0x20, 0x2d, 0x2d, 0x20, 0x52, 0x4f, 0x4d, 0x20,
    0x52, 0x65, 0x76, 0x2e, 0x20, 0x33, 0x2e, 0x31,
    0x0d, 0x0a, 0x43, 0x6f, 0x70, 0x79, 0x72, 0x69,
    0x67, 0x68, 0x74, 0x20, 0x31, 0x39, 0x38, 0x32,
    0x2c, 0x31, 0x39, 0x38, 0x33, 0x2c, 0x31, 0x39,
    0x38, 0x34, 0x20, 0x4d, 0x6f, 0x72, 0x72, 0x6f,
    0x77, 0x20, 0x44, 0x65, 0x73, 0x69, 0x67, 0x6e,
    0x73, 0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x0d,
    0x0a, 0x0a, 0x00, 0x54, 0x65, 0x73, 0x74, 0x69,
    0x6e, 0x67, 0x20, 0x6d, 0x65, 0x6d, 0x6f, 0x72,
    0x79, 0x20, 0x2d, 0x2d, 0x20, 0x70, 0x6c, 0x65,
    0x61, 0x73, 0x65, 0x20, 0x73, 0x74, 0x61, 0x6e,
    0x64, 0x20, 0x62, 0x79, 0x3a, 0x20, 0x00, 0x4d,
    0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20, 0x4f, 0x2e,
    0x4b, 0x2e, 0x0d, 0x0a, 0x0a, 0x00, 0x49, 0x6e,
    0x73, 0x65, 0x72, 0x74, 0x20, 0x43, 0x50, 0x2f,
    0x4d, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
    0x20, 0x64, 0x69, 0x73, 0x6b, 0x65, 0x74, 0x74,
    0x65, 0x20, 0x69, 0x6e, 0x20, 0x44, 0x72, 0x69,
    0x76, 0x65, 0x20, 0x41, 0x20, 0x61, 0x6e, 0x64,
    0x20, 0x70, 0x72, 0x65, 0x73, 0x73, 0x20, 0x5b,
    0x52, 0x45, 0x54, 0x55, 0x52, 0x4e, 0x5d, 0x20,
    0x07, 0x00, 0x07, 0x07, 0x52, 0x4f, 0x4d, 0x20,
    0x4d, 0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20, 0x65,
    0x72, 0x72, 0x6f, 0x72, 0x00, 0x07, 0x07, 0x52,
    0x41, 0x4d, 0x20, 0x4d, 0x65, 0x6d, 0x6f, 0x72,
    0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00,
    0x07, 0x07, 0x0d, 0x0a, 0x52, 0x65, 0x61, 0x64,
    0x20, 0x00, 0x20, 0x65, 0x78, 0x70, 0x65, 0x63,
    0x74, 0x65, 0x64, 0x20, 0x00, 0x20, 0x61, 0x74,
    0x20, 0x52, 0x41, 0x4d, 0x20, 0x6c, 0x6f, 0x63,
    0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x00, 0x0d,
    0x0a, 0x44, 0x69, 0x61, 0x67, 0x6e, 0x6f, 0x73,
    0x74, 0x69, 0x63, 0x73, 0x20, 0x4d, 0x65, 0x6e,
    0x75, 0x0d, 0x0a, 0x0a, 0x31, 0x29, 0x20, 0x50,
    0x6f, 0x72, 0x74, 0x20, 0x31, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x5c,
    0x0d, 0x0a, 0x32, 0x29, 0x20, 0x50, 0x6f, 0x72,
    0x74, 0x20, 0x32, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3e, 0x20,
    0x42, 0x61, 0x72, 0x62, 0x65, 0x72, 0x2d, 0x50,
    0x6f, 0x6c, 0x65, 0x20, 0x70, 0x61, 0x74, 0x74,
    0x65, 0x72, 0x6e, 0x20, 0x74, 0x65, 0x73, 0x74,
    0x0d, 0x0a, 0x33, 0x29, 0x20, 0x43, 0x65, 0x6e,
    0x74, 0x72, 0x6f, 0x6e, 0x69, 0x63, 0x73, 0x20,
    0x70, 0x6f, 0x72, 0x74, 0x20, 0x2f, 0x0d, 0x0a,
    0x34, 0x29, 0x20, 0x4c, 0x6f, 0x6f, 0x70, 0x20,
    0x62, 0x61, 0x63, 0x6b, 0x20, 0x6f, 0x6e, 0x20,
    0x70, 0x6f, 0x72, 0x74, 0x20, 0x32, 0x0d, 0x0a,
    0x35, 0x29, 0x20, 0x52, 0x41, 0x4d, 0x20, 0x74,
    0x65, 0x73, 0x74, 0x0d, 0x0a, 0x36, 0x29, 0x20,
    0x46, 0x44, 0x43, 0x20, 0x52, 0x2f, 0x57, 0x0d,
    0x0a, 0x37, 0x29, 0x20, 0x46, 0x44, 0x43, 0x20,
    0x73, 0x65, 0x65, 0x6b, 0x20, 0x74, 0x65, 0x73,
    0x74, 0x0d, 0x0a, 0x38, 0x29, 0x20, 0x56, 0x46,
    0x4f, 0x20, 0x74, 0x65, 0x73, 0x74, 0x0d, 0x0a,
    0x39, 0x29, 0x20, 0x42, 0x6f, 0x6f, 0x74, 0x0d,
    0x0a, 0x0a, 0x45, 0x6e, 0x74, 0x65, 0x72, 0x20,
    0x23, 0x3a, 0x00, 0x0d, 0x0a, 0x45, 0x6e, 0x64,
    0x20, 0x6f, 0x66, 0x20, 0x70, 0x61, 0x73, 0x73,
    0x20, 0x00, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72,
    0x73, 0x20, 0x73, 0x6f, 0x20, 0x66, 0x61, 0x72,
    0x2e, 0x00, 0x6c, 0x6f, 0x6f, 0x70, 0x20, 0x74,
    0x65, 0x73, 0x74, 0x2e, 0x00, 0x0d, 0x0a, 0x46,
    0x61, 0x69, 0x6c, 0x65, 0x64, 0x20, 0x00, 0x0d,
    0x0a, 0x50, 0x61, 0x73, 0x73, 0x65, 0x64, 0x20,
    0x00, 0x45, 0x6e, 0x74, 0x65, 0x72, 0x20, 0x64,
    0x72, 0x69, 0x76, 0x65, 0x20, 0x74, 0x6f, 0x20,
    0x74, 0x65, 0x73, 0x74, 0x20, 0x28, 0x41, 0x2d,
    0x44, 0x29, 0x3a, 0x00, 0x0d, 0x0a, 0x49, 0x6e,
    0x73, 0x65, 0x72, 0x74, 0x20, 0x61, 0x20, 0x66,
    0x6f, 0x72, 0x6d, 0x61, 0x74, 0x74, 0x65, 0x64,
    0x20, 0x64, 0x69, 0x73, 0x6b, 0x65, 0x74, 0x74,
    0x65, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x72, 0x69,
    0x76, 0x65, 0x20, 0x00, 0x0d, 0x0a, 0x57, 0x72,
    0x69, 0x74, 0x69, 0x6e, 0x67, 0x2e, 0x2e, 0x2e,
    0x00, 0x72, 0x65, 0x61, 0x64, 0x69, 0x6e, 0x67,
    0x00, 0x0d, 0x0a, 0x00, 0x3a, 0xa0, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x0a, 0x00, 0xfe, 0x00, 0x00,
    0x00, 0xa6, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x09, 0x46, 0x00, 0x00,
    0x00, 0x01, 0x03, 0x05, 0x1c, 0xff, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x01, 0x32, 0x01,
    0x01, 0xd8, 0x58, 0x05, 0x6f, 0x03, 0x05, 0x1c,
    0xff, 0x02, 0x59, 0x58, 0x05, 0x6f, 0x03, 0x05,
    0x1c, 0xff, 0x04, 0x5a, 0x58, 0x05, 0x6f, 0x03,
    0x05, 0x1c, 0xff, 0x04, 0x5b, 0x58, 0x05, 0x6f,
    0x03, 0x05, 0x1c, 0xff, 0x18, 0x0c, 0x0b, 0x0c,
    0x29, 0x0c, 0x29, 0x0c, 0x35, 0x0c, 0x4b, 0x0c,
    0x40, 0x0c, 0x4b, 0x0c, 0x5c, 0x0c, 0x6d, 0x0c,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    };

static uint8 *mmd_rom = mmd_rom_31;    /* Default 3.1 ROM */

/*******************/
/* MMD Definitions */
/*******************/

#define MMD_MAX_DRIVES  4
#define MMD_UNITS       MMD_MAX_DRIVES + 1
#define MMD_SIO1_UNIT   MMD_UNITS - 1
#define MMD_SECTOR_LEN  1024
#define MMD_SPT         5           /* Sectors per Track */
#define MMD_BPT         (MMD_SPT * MMD_SECTOR_LEN)
#define MMD_TRACKS      40
#define MMD_SIDES       2
#define MMD_CAPACITY    (MMD_BPT * MMD_TRACKS * MMD_SIDES)  /* Default Disk Capacity */

#define MMD_IO_BASE     0xf0
#define MMD_IO_SIZE     14        /* FE is used by SIMH */

#define MMD_REG_CH0DIV   0xf0
#define MMD_REG_CH1DIV   0xf1
#define MMD_REG_CH2DIV   0xf2
#define MMD_REG_CTCSEL   0xf3
#define MMD_REG_CENDATA  0xf4
#define MMD_REG_MTRCHK   0xf5
#define MMD_REG_ROMCTL   0xf6
#define MMD_REG_VFO      0xf7
#define MMD_REG_TC       0xf7
#define MMD_REG_MOTOR    0xf8
#define MMD_REG_FDCSTAT  0xfa
#define MMD_REG_FDCDATA  0xfb
#define MMD_REG_CDATA    0xfc
#define MMD_REG_CSTAT    0xfd


#define MMD_CH_1         0x7e    /* Select CTC Channel 1 */
#define MMD_CH_2         0xbe    /* Select CTC Channel 2 */

#define MMD_STAT_MTR     0x03
#define MMD_STAT_MTRON   0x01
#define MMD_STAT_IDX     0x04
#define MMD_STAT_ACK     0x08
#define MMD_STAT_BUSY    0x10
#define MMD_STAT_DIAG    0x20
#define MMD_STAT_STROBE  0x80

#define MMD_STAT_D0B     0x01
#define MMD_STAT_D1B     0x02
#define MMD_STAT_D2B     0x04
#define MMD_STAT_D3B     0x08
#define MMD_STAT_BUSY    0x10
#define MMD_STAT_EXM     0x20
#define MMD_STAT_DIO     0x40
#define MMD_STAT_RQM     0x80

#define MMD_STAT0_US0     0x01    /* Unit Select 0 */
#define MMD_STAT0_US1     0x02    /* Unit Select 1 */
#define MMD_STAT0_HD      0x04    /* Head Address */
#define MMD_STAT0_NR      0x08    /* Not Ready */
#define MMD_STAT0_EC      0x10    /* Equipment Check */
#define MMD_STAT0_SE      0x20    /* Seek End */
#define MMD_STAT0_ICMASK  0xc0    /* Interrupt Code Mask */
#define MMD_STAT0_NT      0x00    /* Normal Termination */
#define MMD_STAT0_AT1     0x40    /* Abnormal Termination */
#define MMD_STAT0_IC      0x80    /* Invalid Command Issue */
#define MMD_STAT0_AT2     0xc0    /* Abnormal Termination */

#define MMD_STAT1_MA      0x10    /* Missing Address Mark */
#define MMD_STAT1_NW      0x20    /* Not Writable */
#define MMD_STAT1_ND      0x40    /* No Data */
#define MMD_STAT1_OR      0x10    /* Overrun Error */
#define MMD_STAT1_CRC     0x20    /* CRC Error */
#define MMD_STAT1_EN      0x80    /* End of Cylinder */

#define MMD_STAT2_MD      0x01    /* Missing Address Mark */
#define MMD_STAT2_BC      0x02    /* Bad Cylinder */
#define MMD_STAT2_SN      0x04    /* Scan Not Satisfied */
#define MMD_STAT2_SH      0x08    /* Scan Equal Hit */
#define MMD_STAT2_WC      0x10    /* Wrong Cylinder */
#define MMD_STAT2_DD      0x20    /* Data Error in Data Field */
#define MMD_STAT2_CM      0x40    /* Control Mark */

#define MMD_STAT3_US0     0x01    /* Unit Select 0 */
#define MMD_STAT3_US1     0x02    /* Unit Select 1 */
#define MMD_STAT3_HD      0x04    /* Head Address */
#define MMD_STAT3_TS      0x08    /* Two Side */
#define MMD_STAT3_TO      0x10    /* Track 0 */
#define MMD_STAT3_RY      0x20    /* Ready */
#define MMD_STAT3_WP      0x40    /* Write Protected */
#define MMD_STAT3_FT      0x80    /* Fault */

#define MMD_CMD_MASK     0x1f
#define MMD_CMD_SPECIFY  0x03
#define MMD_CMD_SDS      0x04
#define MMD_CMD_WRITE    0x05
#define MMD_CMD_READ     0x06
#define MMD_CMD_RECALIB  0x07
#define MMD_CMD_SIS      0x08
#define MMD_CMD_FORMAT   0x0d
#define MMD_CMD_SEEK     0x0f

#define MMD_FLAG_SK      0x20
#define MMD_FLAG_MF      0x40
#define MMD_FLAG_MT      0x80

static uint8 secbuf[MMD_SECTOR_LEN];   /* Sector Buffer */

/***************/
/* MMD Device */
/***************/

#define MMD_NAME  "Morrow Micro-Decision"
#define MMD_SNAME "MMD"

/*
** MMD Registers and Interface Controls
*/
typedef struct {
    uint8   phase;                          /* Phase */
    uint8   status;                         /* Main Status Register */
    uint8   motor[MMD_MAX_DRIVES];          /* Motor Status */
    uint8   stat0[MMD_MAX_DRIVES];          /* Status 0 Register */
    uint8   stat1[MMD_MAX_DRIVES];          /* Status 1 Register */
    uint8   stat2[MMD_MAX_DRIVES];          /* Status 2 Register */
    uint8   stat3[MMD_MAX_DRIVES];          /* Status 3 Register */
    uint8   c[MMD_MAX_DRIVES];
    uint8   hd[MMD_MAX_DRIVES];             /* Head Register */
    uint8   h;                              /* Head Register */
    uint8   r;                              /* Sector Register */
    uint8   n;                              /* Number Register */
    uint8   eot;                            /* EOT Register */
    uint8   gpl;                            /* GPL Register */
    uint8   dtl;                            /* DTL Register */
    uint8   sto;                            /* STO Register */
    uint8   pcn;                            /* PCN Register */
    uint8   sc;                             /* SC Register */
    uint8   d;                              /* D Register */
    uint8   cmdflags;                       /* Command Flags */
    uint8   cmd;                            /* Command Register */
    uint8   stack[10];                      /* Stack Buffer */
    uint8   stkidx;                         /* Stack Index */
    uint8   stkcnt;                         /* Stack Count */
    uint32  secidx;                         /* Sector index */
    uint32  seccnt;                         /* Sector count */
    uint8   rxd;                            /* Console rx data register */
    uint8   txd;                            /* Console tx data register */
    uint8   txp;                            /* Console tx data pending */
    uint8   cstatus;                        /* Console status register */
    uint16  baud;                           /* Console baud rate */
    uint8   intenable;                      /* Interrupt enable */
    uint8   intvec;                         /* Interrupt vector */
    uint8   databus;                        /* Mode 2 interrupt data bus */
    uint8   tc;                             /* Terminal Count */
} MMD_REG;

enum { MMD_PHASE_CMDNEW, MMD_PHASE_CMDSTK, MMD_PHASE_EXECRD, MMD_PHASE_EXECWR, MMD_PHASE_RSLTSTK };

typedef struct {
    uint32    rom_base;     /* Memory Base Address */
    uint32    rom_size;     /* Memory Address space requirement */
    uint32    io_base;      /* I/O Base Address */
    uint32    io_size;      /* I/O Address Space requirement */
    int32     conn[2];      /* Connected Status */
    uint8     diagEnabled;  /* DIAG is enabled */
    uint8     romEnabled;   /* ROM is enabled */
    uint32    sio1ticks;    /* SIO Timer ticks */
    uint8     drive;        /* Currently selected drive */
    MMD_REG   MMD;          /* MMD Registers and Data */
    UNIT *uptr[MMD_UNITS];
} MMD_CTX;

static MMD_CTX mmd_ctx_data = {
    MMD_ROM_BASE, MMD_ROM_SIZE, MMD_IO_BASE, MMD_IO_SIZE,
    {0,0}, FALSE, TRUE, 6, 10
};

static MMD_CTX *mmd_ctx = &mmd_ctx_data;

static UNIT mmd_unit[MMD_UNITS] = {
    { UDATA (mmd_svc, UNIT_FIX + UNIT_ATTABLE + UNIT_DISABLE + UNIT_ROABLE, MMD_CAPACITY), 10000 },
    { UDATA (mmd_svc, UNIT_FIX + UNIT_ATTABLE + UNIT_DISABLE + UNIT_ROABLE, MMD_CAPACITY), 10000 },
    { UDATA (mmd_svc, UNIT_FIX + UNIT_ATTABLE + UNIT_DISABLE + UNIT_ROABLE, MMD_CAPACITY), 10000 },
    { UDATA (mmd_svc, UNIT_FIX + UNIT_ATTABLE + UNIT_DISABLE + UNIT_ROABLE, MMD_CAPACITY), 10000 },
    { UDATA (mmd_sio1_svc, UNIT_ATTABLE + UNIT_DISABLE, 0), 10000 },
};

static REG mmd_reg[] = {
    { DRDATAD (DRIVE, mmd_ctx_data.drive, 8, "Current drive register"), },
    { HRDATAD (STATUS, mmd_ctx_data.MMD.status, 8, "Main Status register"), },
    { HRDATAD (CMD, mmd_ctx_data.MMD.cmd, 8, "Command register"), },
    { DRDATAD (SECTOR, mmd_ctx_data.MMD.r, 8, "Sector register"), },
    { DRDATAD (NUMBER, mmd_ctx_data.MMD.n, 8, "Number register"), },
    { DRDATAD (HEAD, mmd_ctx_data.MMD.h, 8, "Head register"), },
    { FLDATAD (ROM, mmd_ctx_data.romEnabled, 0, "ROM enabled bit"), },
    { FLDATAD (DIAG, mmd_ctx_data.diagEnabled, 0, "DIAG enabled bit"), },
    { HRDATAD (BAUD, mmd_ctx_data.MMD.baud, 16, "Console port baud register"), },
    { HRDATAD (TXP, mmd_ctx_data.MMD.txp, 8, "Console port TX Pending register"), },
    { HRDATAD (TXD, mmd_ctx_data.MMD.txd, 8, "Console port TX register"), },
    { HRDATAD (RXD, mmd_ctx_data.MMD.rxd, 8, "Console port RX register"), },
    { FLDATAD (INTENA, mmd_ctx_data.MMD.intenable, 1, "Interrupt enable"), },
    { DRDATAD (INTVEC, mmd_ctx_data.MMD.intvec, 8, "Interrupt vector (0-7)"), },
    { DRDATAD (DATABUS, mmd_ctx_data.MMD.databus, 8, "Mode 2 Interrupt Data Bus"), },
    { NULL }
};

static MTAB mmd_mod[] = {
    { MTAB_XTD|MTAB_VDV|MTAB_VALR, 0, "DIAG", "DIAG={ENABLE|DISABLE}",
        &mmd_set_diag, &mmd_show_diag, NULL, "Set/Show ROM DIAG enabled/disabled status"},
    { MTAB_XTD|MTAB_VDV|MTAB_VALR, 0, "ROM", "ROM={25|31}",
        &mmd_set_rom, &mmd_show_rom, NULL, "Set/Show ROM version"},
    { 0 }
};

DEVICE mmd_dev = {
    MMD_SNAME,                            /* name */
    mmd_unit,                             /* unit */
    mmd_reg,                              /* registers */
    mmd_mod,                              /* modifiers */
    MMD_UNITS,                            /* # units */
    10,                                   /* address radix */
    31,                                   /* address width */
    1,                                    /* addr increment */
    8,                                    /* data radix */
    8,                                    /* data width */
    NULL,                                 /* examine routine */
    NULL,                                 /* deposit routine */
    &mmd_reset,                           /* reset routine */
    &mmd_boot,                            /* boot routine */
    &mmd_attach,                          /* attach routine */
    &mmd_detach,                          /* detach routine */
    &mmd_ctx_data,                        /* context */
    (DEV_DISABLE | DEV_DIS | DEV_DEBUG),  /* flags */
    ERROR_MSG,                            /* debug control */
    mmd_dt,                               /* debug flags */
    NULL,                                 /* mem size routine */
    NULL,                                 /* logical name */
    NULL,                                 /* help */
    NULL,                                 /* attach help */
    NULL,                                 /* context for help */
    &mmd_description                      /* description */
};

/********/
/* UART */
/********/

static TMLN mmd_tmln[1] = { /* line descriptors */
    { 0 }
};

static TMXR mmd_tmxr = {    /* multiplexer descriptor */
    1,                      /* number of terminal lines */
    0,                      /* listening port (reserved) */
    0,                      /* master socket  (reserved) */
    mmd_tmln,               /* line descriptor array */
    NULL,                   /* line connection order */
    NULL                    /* multiplexer device (derived internally) */
};

typedef struct {
    PNP_INFO  pnp;          /* Must be first */
    int32     conn;         /* Connected Status */
    uint32    ticks;        /* SIO Timer ticks */
    uint16    baud;         /* Baud rate */
    uint8     rxd;          /* Receive Buffer */
    uint8     txd;          /* Transmit Buffer */
    uint8     txp;          /* Transmit Pending */
    uint8     status;       /* Status Buffer */
    TMLN     *tmln;         /* TMLN pointer */
    TMXR     *tmxr;         /* TMXR pointer */
} MMD_UART_CTX;

#define MMD_STAT_TXRDY   0x01
#define MMD_STAT_RXRDY   0x02

/*********************/
/* MMDM Modem Device */
/*********************/

#define MMDM_NAME  "Morrow Micro Decision Modem Port"
#define MMDM_SNAME "MMDM"

#define MMD_REG_MDATA   0xfe
#define MMD_REG_MSTAT   0xff

#define MMD_BAUD_110    1136
#define MMD_BAUD_300     417
#define MMD_BAUD_600     208
#define MMD_BAUD_1200    104
#define MMD_BAUD_2400     52
#define MMD_BAUD_4800     26
#define MMD_BAUD_9600     13

static TMLN mmdm_tmln[1] = { /* line descriptors */
    { 0 }
};

static TMXR mmdm_tmxr = {    /* multiplexer descriptor */
    1,                       /* number of terminal lines */
    0,                       /* listening port (reserved) */
    0,                       /* master socket  (reserved) */
    mmdm_tmln,               /* line descriptor array */
    NULL,                    /* line connection order */
    NULL                     /* multiplexer device (derived internally) */
};

static MMD_UART_CTX mmdm_ctx_data = {
    {0, 0, MMD_REG_MDATA, 2}, 0, 0, 9600,
    0, 0, 0, 0, mmdm_tmln, &mmdm_tmxr
};

static MMD_UART_CTX *mmdm_ctx = &mmdm_ctx_data;

static REG mmdm_reg[] = {
    { HRDATAD (BAUD, mmdm_ctx_data.baud, 16, "Modem port baud register"), },
    { HRDATAD (STAT, mmdm_ctx_data.status, 8, "Modem port status register"), },
    { HRDATAD (TXP, mmdm_ctx_data.txp, 8, "Modem port TX pending register"), },
    { HRDATAD (TXD, mmdm_ctx_data.txd, 8, "Modem port TX data register"), },
    { HRDATAD (RXD, mmdm_ctx_data.rxd, 8, "Modem port RX register"), },
    { DRDATAD (TICKS, mmdm_ctx_data.ticks, 32, "Modem timer ticks"), },
    { NULL }
};

static UNIT mmdm_unit[] = {
    { UDATA (mmdm_svc, UNIT_ATTABLE + UNIT_DISABLE, 0), 10000 }
};

static MTAB mmdm_mod[] = {
    { 0 }
};

DEVICE mmdm_dev = {
    MMDM_SNAME,                                     /* name */
    mmdm_unit,                                      /* unit */
    mmdm_reg,                                       /* registers */
    mmdm_mod,                                       /* modifiers */
    1,                                              /* # units */
    10,                                             /* address radix */
    31,                                             /* address width */
    1,                                              /* addr increment */
    8,                                              /* data radix */
    8,                                              /* data width */
    NULL,                                           /* examine routine */
    NULL,                                           /* deposit routine */
    &mmdm_reset,                                    /* reset routine */
    NULL,                                           /* boot routine */
    &mmdm_attach,                                   /* attach routine */
    &mmdm_detach,                                   /* detach routine */
    &mmdm_ctx_data,                                 /* context */
    (DEV_DISABLE | DEV_DIS | DEV_DEBUG | DEV_MUX),  /* flags */
    ERROR_MSG,                                      /* debug control */
    mmd_dt,                                         /* debug flags */
    NULL,                                           /* mem size routine */
    NULL,                                           /* logical name */
    NULL,                                           /* help */
    NULL,                                           /* attach help */
    NULL,                                           /* context for help */
    &mmd_description                                /* description */
};

/*************/
/* Functions */
/*************/


static const char* mmd_description(DEVICE *dptr) {
    return MMD_NAME;
}

/* Reset routines */
static t_stat mmd_reset(DEVICE *dptr)
{
    uint8 i;

    if (dptr->flags & DEV_DIS) { /* Disconnect I/O Ports */
        sim_map_resource(mmd_ctx->rom_base, mmd_ctx->rom_size, RESOURCE_TYPE_MEMORY, &mmdrom, "mmdrom", TRUE);
        sim_map_resource(mmd_ctx->io_base, mmd_ctx->io_size, RESOURCE_TYPE_IO, &mmddev, "mmddev", TRUE);
    } else {
        if (sim_map_resource(mmd_ctx->rom_base, mmd_ctx->rom_size, RESOURCE_TYPE_MEMORY, &mmdrom, "mmdrom", FALSE) != 0) {
            sim_debug(ERROR_MSG, &mmd_dev, "Error mapping MEM resource at 0x%04x\n", mmd_ctx->rom_base);
            return SCPE_ARG;
        }
        /* Connect I/O Ports at base address */
        if (sim_map_resource(mmd_ctx->io_base, mmd_ctx->io_size, RESOURCE_TYPE_IO, &mmddev, "mmddev", FALSE) != 0) {
            sim_debug(ERROR_MSG, &mmd_dev, "Error mapping I/O resource at 0x%02x\n", mmd_ctx->io_base);
            return SCPE_ARG;
        }

        /* Start timers for console unit */
        sim_activate_after(mmd_ctx->uptr[MMD_SIO1_UNIT], 500);  /* activate 500 us timer */
    }

    mmd_ctx->drive = 0;

    mmd_ctx->MMD.phase = MMD_PHASE_CMDNEW;
    mmd_ctx->MMD.status = MMD_STAT_RQM;
    mmd_ctx->MMD.r = 1;
    mmd_ctx->MMD.h = 1;
    mmd_ctx->MMD.cmd = 0;
    mmd_ctx->MMD.sto = 1;
    mmd_ctx->MMD.pcn = 2;
    mmd_ctx->MMD.cstatus = MMD_STAT_TXRDY;
    mmd_ctx->MMD.intenable = TRUE;
    mmd_ctx->MMD.intvec = 0;
    mmd_ctx->MMD.databus = 0x7f;
    mmd_ctx->MMD.tc = FALSE;

    for (i=0; i < MMD_UNITS; i++) {
        if (mmd_ctx->uptr[i] == NULL) {
            mmd_ctx->uptr[i] = &mmd_dev.units[i];
        }
    }

    /* Reset Registers */
    for (i=0; i < MMD_MAX_DRIVES; i++) {
        mmd_ctx->MMD.motor[i] |= MMD_STAT_DIAG;
        if (mmd_ctx->diagEnabled) {
            mmd_ctx->MMD.motor[i] &= ~MMD_STAT_DIAG;
        }
        mmd_ctx->MMD.stat0[i] = 0;
        mmd_ctx->MMD.stat1[i] = 0;
        mmd_ctx->MMD.stat2[i] = 0;
        mmd_ctx->MMD.stat3[i] = i;    /* Set US1/US0 */
        mmd_ctx->MMD.c[i] = 0;
        mmd_ctx->MMD.hd[i] = 0;

        /* If drive is attached, set RY flag */
        if (mmd_dev.units[i].flags & UNIT_ATT) {
            mmd_ctx->MMD.stat3[i] |= MMD_STAT3_RY;

            if (mmd_ctx->uptr[i]->capac == MMD_CAPACITY) {
                mmd_ctx->MMD.stat3[i] |= MMD_STAT3_TS;
            }
        }
    }

    sim_debug(STATUS_MSG, &mmd_dev, "reset controller.\n");

    return SCPE_OK;
}

static t_stat mmdm_reset(DEVICE *dptr) {
    MMD_UART_CTX *xptr;

    xptr = (MMD_UART_CTX *) dptr->ctxt;

    dptr->units[0].dptr = dptr;

    if (dptr->flags & DEV_DIS) { /* Disconnect Modem I/O Ports */
        sim_map_resource(xptr->pnp.io_base, xptr->pnp.io_size, RESOURCE_TYPE_IO, &mmddev, "mmdmdev", TRUE);

        mmdm_ctx->status = 0;
    } else {
        /* Connect I/O Ports at base address */
        if (sim_map_resource(xptr->pnp.io_base, xptr->pnp.io_size, RESOURCE_TYPE_IO, &mmddev, "mmdmdev", FALSE) != 0) {
            sim_debug(ERROR_MSG, &mmd_dev, "Error mapping I/O resource at 0x%02x\n", xptr->pnp.io_base);
            return SCPE_ARG;
        }

        mmdm_ctx->status = MMD_STAT_TXRDY;

        /* Start timer */
        sim_activate_after(dptr->units, 500);  /* reactivate 500us timer */
    }

    return SCPE_OK;
}

/* Service routines */
static t_stat mmd_svc(UNIT *uptr)
{
    mmd_ctx->MMD.status &= ~MMD_STAT_BUSY;

    sim_activate_after(uptr, 4000);  /* reactivate 4ms timer */

    return SCPE_OK;
}

static t_stat mmdm_svc(UNIT *uptr)
{
    int32 c;
    t_stat r;

    mmdm_ctx->ticks++;

    /* Check for new incoming connection */
    if (uptr->flags & UNIT_ATT) {
        if (tmxr_poll_conn(&mmdm_tmxr) >= 0) {      /* poll connection */

            mmdm_ctx->conn = 1;          /* set connected   */

            sim_debug(STATUS_MSG, uptr->dptr, "new modem connection.\n");
        }

        /* Modem TX byte pending? */
        if (mmdm_ctx->txp) {
            r = tmxr_putc_ln(&mmdm_tmln[0], mmdm_ctx->txd);

            mmdm_ctx->txp = FALSE;

            if (r == SCPE_LOST) {
                mmdm_ctx->conn = 0;          /* Connection was lost */
                sim_debug(STATUS_MSG, uptr->dptr, "modem lost connection.\n");
            }
        }

        /* Update TX Ready */
        if (!(mmdm_ctx->status & MMD_STAT_TXRDY)) {
            tmxr_poll_tx(&mmdm_tmxr);
            mmdm_ctx->status |= (tmxr_txdone_ln(&mmdm_tmln[0]) && mmdm_ctx->conn) ? MMD_STAT_TXRDY : 0;
        }

        /* Check for Data if modem RX buffer empty */
        if (!(mmdm_ctx->status & MMD_STAT_RXRDY)) {
            tmxr_poll_rx(&mmdm_tmxr);

            c = tmxr_getc_ln(&mmdm_tmln[0]);

            if (c & (TMXR_VALID | SCPE_KFLAG)) {
                mmdm_ctx->rxd = c & 0xff;
                mmdm_ctx->status |= MMD_STAT_RXRDY;
            }
        }

        /* Restart timer */
        sim_activate_after(uptr, 500);  /* reactivate 500 us timer */
    }

    return SCPE_OK;
}

static t_stat mmd_sio1_svc(UNIT *uptr)
{
    int32 c;
    t_stat r;

    mmd_ctx->sio1ticks++;

    /* Check for new incoming connection */
    if (uptr->flags & UNIT_ATT) {
        if (tmxr_poll_conn(&mmd_tmxr) >= 0) {      /* poll connection */

            mmd_ctx->conn[0] = 1;          /* set connected   */

            sim_debug(STATUS_MSG, uptr->dptr, "new connection.\n");
        }
    }

    /* Serial 1 TX byte pending? */
    if (mmd_ctx->MMD.txp) {
        if (uptr->flags & UNIT_ATT) {
            r = tmxr_putc_ln(&mmd_tmln[0], mmd_ctx->MMD.txd);
        } else {
            r = sim_putchar(mmd_ctx->MMD.txd);
        }

        mmd_ctx->MMD.txp = FALSE;

        if (r == SCPE_LOST) {
            mmd_ctx->conn[0] = 0;          /* Connection was lost */
            sim_debug(STATUS_MSG, uptr->dptr, "lost connection.\n");
        }
    }

    /* Update TX Ready */
    if (!(mmd_ctx->MMD.cstatus & MMD_STAT_TXRDY)) {
        if (uptr->flags & UNIT_ATT) {
            tmxr_poll_tx(&mmd_tmxr);
            mmd_ctx->MMD.cstatus |= (tmxr_txdone_ln(&mmd_tmln[0]) && mmd_ctx->conn[0]) ? MMD_STAT_TXRDY : 0;
        } else {
            mmd_ctx->MMD.cstatus |= MMD_STAT_TXRDY;
        }
    }

    /* Check for Data if Serial 1 RX buffer empty */
    if (!(mmd_ctx->MMD.cstatus & MMD_STAT_RXRDY)) {
        if (uptr->flags & UNIT_ATT) {
            tmxr_poll_rx(&mmd_tmxr);

            c = tmxr_getc_ln(&mmd_tmln[0]);
        } else {
            c = sim_poll_kbd();
        }

        if (c & (TMXR_VALID | SCPE_KFLAG)) {
            mmd_ctx->MMD.rxd = c & 0xff;
            mmd_ctx->MMD.cstatus |= MMD_STAT_RXRDY;
        }
    }

    /* Restart timer */
    sim_activate_after(uptr, 500);  /* reactivate 500 us timer */

    return SCPE_OK;
}

/* Attach routines */
static t_stat mmd_attach(UNIT *uptr, CONST char *cptr)
{
    t_stat r;
    unsigned int i = 0;

    /* Attaching to console interface? */
    if (uptr == &mmd_dev.units[MMD_SIO1_UNIT]) {
        if ((r = tmxr_attach(&mmd_tmxr, uptr, cptr)) == SCPE_OK) {
            mmd_tmln[0].rcve = 1;

            sim_debug(VERBOSE_MSG, uptr->dptr, "attached '%s' to console interface.\n", cptr);
        }

        return r;
    }

    r = attach_unit(uptr, cptr);    /* attach unit  */
    if (r != SCPE_OK) {             /* error?       */
        sim_debug(ERROR_MSG, &mmd_dev, "ATTACH error=%d\n", r);
        return r;
    }

    /* Determine length of this disk */
    if (sim_fsize(uptr->fileref) != 0) {
        uptr->capac = sim_fsize(uptr->fileref);
    } else {
        uptr->capac = MMD_CAPACITY;
    }

    for (i = 0; i < MMD_UNITS; i++) {
        if (mmd_dev.units[i].fileref == uptr->fileref) {
            break;
        }
    }

    if (i >= MMD_UNITS) {
        mmd_detach(uptr);

        return SCPE_ARG;
    }

    /* Set Status 3 */
    mmd_ctx->MMD.stat3[i] |= MMD_STAT3_RY;
    if (uptr->flags | UNIT_RO) {
        mmd_ctx->MMD.stat3[i] |= MMD_STAT3_WP;
    }

    if (uptr->capac == MMD_CAPACITY) {
        mmd_ctx->MMD.stat3[i] |= MMD_STAT3_TS;
    }

    /* Default for new file is DSK */
    uptr->u3 = IMAGE_TYPE_DSK;

    sim_debug(VERBOSE_MSG, uptr->dptr, "unit %d, attached to '%s' size=%d\n",
        i, cptr, uptr->capac);

    return SCPE_OK;
}

static t_stat mmdm_attach(UNIT *uptr, CONST char *cptr)
{
    t_stat r;

    if ((r = tmxr_attach(&mmdm_tmxr, uptr, cptr)) == SCPE_OK) {
        mmdm_tmln[0].rcve = 1;

        sim_debug(VERBOSE_MSG, uptr->dptr, "attached '%s' to modem interface.\n", cptr);
    }

    return r;
}

/* Detach routines */
static t_stat mmd_detach(UNIT *uptr)
{
    t_stat r;
    int8 i;

    /* detaching console interface? */
    if (uptr == &mmd_dev.units[MMD_SIO1_UNIT]) {
        if ((r = tmxr_detach(&mmd_tmxr, uptr)) == SCPE_OK) {
            sim_debug(VERBOSE_MSG, uptr->dptr, "detached from console interface.\n");
        }

        return r;
    }

    for (i = 0; i < MMD_UNITS; i++) {
        if (mmd_dev.units[i].fileref == uptr->fileref) {
            break;
        }
    }

    if (i >= MMD_UNITS) {
        return SCPE_ARG;
    }

    mmd_ctx->MMD.stat3[i] &= (MMD_STAT3_US0 | MMD_STAT3_US1);

    DBG_PRINT(("Detach MMD%d\n", i));

    r = detach_unit(uptr);  /* detach unit */

    if (r != SCPE_OK) {
        return r;
    }

    mmdm_dev.units[i].fileref = NULL;

    sim_debug(VERBOSE_MSG, uptr->dptr, "unit %d detached.\n", i);

    return SCPE_OK;
}

static t_stat mmdm_detach(UNIT *uptr)
{
    t_stat r;

    if ((r = tmxr_detach(&mmdm_tmxr, uptr)) == SCPE_OK) {
        sim_debug(VERBOSE_MSG, uptr->dptr, "detached from modem interface.\n");
    }

    return r;
}

static t_stat mmd_config_line(DEVICE *dev, TMLN *tmln, int baud)
{
    char config[20];
    const char *fmt;

    t_stat r = SCPE_IERR;

    if (tmln->serport) {
        fmt = "8N1";

        switch (baud) {
            case MMD_BAUD_110:
                baud = 110;
                break;

            case MMD_BAUD_300:
                baud = 300;
                break;

            case MMD_BAUD_600:
                baud = 600;
                break;

            case MMD_BAUD_1200:
                baud = 1200;
                break;

            case MMD_BAUD_2400:
                baud = 2400;
                break;

            case MMD_BAUD_4800:
                baud = 4800;
                break;

            case MMD_BAUD_9600:
                baud = 9600;
                break;

            default:
                return SCPE_OK;
        }

        sprintf(config, "%d-%s", baud, fmt);

        r = tmxr_set_config_line(tmln, config);

        sim_debug(STATUS_MSG, dev, "port configuration set to '%s'.\n", config);
    }

    return r;
}

static t_stat mmd_boot(int32 unitno, DEVICE *dptr)
{
    sim_debug(VERBOSE_MSG, dptr, "Booting using ROM at 0x%04x\n", mmd_ctx->rom_base);

    /* Enable ROM */
    mmd_ctx->romEnabled = TRUE;

    *((int32 *) sim_PC->loc) = mmd_ctx->rom_base;

    return SCPE_OK;
}

static t_stat mmd_set_diag(UNIT *uptr, int32 val, CONST char *cptr, void *desc)
{
    int i;

    if (!cptr) return SCPE_IERR;
    if (!strlen(cptr)) return SCPE_ARG;

    /* this assumes that the parameter has already been upcased */
    if (!strncmp(cptr, "ENABLE", strlen(cptr))) {
        mmd_ctx->diagEnabled = TRUE;
    } else if (!strncmp(cptr, "DISABLE", strlen(cptr))) {
        mmd_ctx->diagEnabled = FALSE;
    } else {
        return SCPE_ARG;
    }

    for (i = 0; i < MMD_MAX_DRIVES; i++) {
        mmd_ctx->MMD.motor[i] |= MMD_STAT_DIAG;
        if (!mmd_ctx->diagEnabled) {
            mmd_ctx->MMD.motor[i] &= ~MMD_STAT_DIAG;
        }
    }

    return SCPE_OK;
}

static t_stat mmd_show_diag(FILE *st, UNIT *uptr, int32 val, CONST void *desc)
{
    if (mmd_ctx->diagEnabled) {
        fprintf(st, "DIAG");
    } else {
        fprintf(st, "NODIAG");
    }

    return SCPE_OK;
}

static t_stat mmd_set_rom(UNIT *uptr, int32 val, CONST char *cptr, void *desc)
{
    if (!cptr) return SCPE_IERR;
    if (!strlen(cptr)) return SCPE_ARG;

    /* this assumes that the parameter has already been upcased */
    if (!strncmp(cptr, "13", strlen(cptr))) {
        mmd_rom = mmd_rom_13;
    } else if (!strncmp(cptr, "23", strlen(cptr))) {
        mmd_rom = mmd_rom_23;
    } else if (!strncmp(cptr, "24", strlen(cptr))) {
        mmd_rom = mmd_rom_24;
    } else if (!strncmp(cptr, "25", strlen(cptr))) {
        mmd_rom = mmd_rom_25;
    } else if (!strncmp(cptr, "31", strlen(cptr))) {
        mmd_rom = mmd_rom_31;
    } else {
        return SCPE_ARG;
    }

    return SCPE_OK;
}

static t_stat mmd_show_rom(FILE *st, UNIT *uptr, int32 val, CONST void *desc)
{
    if (mmd_rom == mmd_rom_13) {
        fprintf(st, "ROM=v1.3");
    } else if (mmd_rom == mmd_rom_23) {
        fprintf(st, "ROM=v2.3");
    } else if (mmd_rom == mmd_rom_24) {
        fprintf(st, "ROM=v2.4");
    } else if (mmd_rom == mmd_rom_25) {
        fprintf(st, "ROM=v2.5");
    } else if (mmd_rom == mmd_rom_31) {
        fprintf(st, "ROM=v3.1");
    } else {
        fprintf(st, "ROM=v?.?");
    }

    return SCPE_OK;
}

static int32 mmdrom(int32 Addr, int32 rw, int32 Data)
{
    /*
    ** The MD3 ROM occupies 4096 bytes (4K) of RAM at
    ** location 0000H. If the ROM is enabled, return
    ** data in the ROM. If the ROM is disabled,
    ** use RAM.
    */
    if (mmd_ctx->romEnabled == TRUE && !rw) {
        return(mmd_rom[Addr & MMD_ROM_MASK]);
    }

    return mmdmem(Addr, rw, Data);
}

static int32 mmdmem(int32 Addr, int32 rw, int32 Data)
{
    if (rw) {
       mmd_mem[Addr & MMD_ROM_MASK] = Data & 0xff;
    }

    return(mmd_mem[Addr & MMD_ROM_MASK]);
}

static int32 mmddev(int32 Addr, int32 rw, int32 data)
{
    if (rw == 0) {
        return(MMD_Dev_Read(Addr));
    } else {
        return(MMD_Dev_Write(Addr, data));
    }
}

static void intZ80(void)
{
    if (mmd_ctx->MMD.intenable) {
        vectorInterrupt |= (1 << mmd_ctx->MMD.intvec);
        dataBus[mmd_ctx->MMD.intvec] = mmd_ctx->MMD.databus;
    }
}

static uint8 MMD_Dev_Read(uint32 Addr)
{
    uint8 cData = 0xff;

    switch(Addr & 0xff) {
        case MMD_REG_ROMCTL:
            mmd_ctx->romEnabled = FALSE;
            break;

        case MMD_REG_CSTAT:
            cData = mmd_ctx->MMD.cstatus;
            break;

        case MMD_REG_MSTAT:
            cData = mmdm_ctx->status;
            break;

        case MMD_REG_CDATA:
            cData = mmd_ctx->MMD.rxd;
            mmd_ctx->MMD.cstatus &= ~MMD_STAT_RXRDY;   /* Clear RX Ready Flag */
            break;

        case MMD_REG_MDATA:
            cData = mmdm_ctx->rxd;
            mmdm_ctx->status &= ~MMD_STAT_RXRDY;   /* Clear RX Ready Flag */
            break;

        case MMD_REG_FDCSTAT:
            cData = mmd_ctx->MMD.status;
            break;

        case MMD_REG_FDCDATA:
            cData = MMD_FDC_Read_Data();
            break;

        case MMD_REG_MTRCHK:
            /* Toggle index if drive mounted to simulate rotation */
            if (mmd_ctx->MMD.stat3[mmd_ctx->drive] & MMD_STAT3_RY) {
                mmd_ctx->MMD.motor[mmd_ctx->drive] ^= MMD_STAT_IDX;
            }

            cData = mmd_ctx->MMD.motor[mmd_ctx->drive];
            break;

        case MMD_REG_MOTOR:
        case MMD_REG_TC:
            MMD_Term_Count();
            break;

        default:
            sim_debug(ERROR_MSG, &mmd_dev, "READ Invalid I/O Address %02x (%02x)\n",
                Addr & 0xFF, Addr & 0x01);
            break;
    }

    return (cData);
}

static uint8 MMD_FDC_Read_Data(void)
{
    uint8 cData = 0xff;

    switch (mmd_ctx->MMD.phase) {
        case MMD_PHASE_RSLTSTK:
            cData = MMD_Result_Stack();
            break;

        case MMD_PHASE_EXECRD:
            cData = MMD_Exec_Read();
            break;

        default:
            sim_debug(ERROR_MSG, &mmd_dev, "Invalid Phase (%02x)\n", mmd_ctx->MMD.phase);
            break;
    }

    return cData;
}

static uint8 MMD_Dev_Write(uint32 Addr, int32 Data)
{
    switch(Addr & 0xff) {
        case MMD_REG_CTCSEL:
            if (Data==MMD_CH_1) {
                mmd_ctx->MMD.baud = 0;
            }
            else if (Data==MMD_CH_2) {
                mmdm_ctx->baud = 0;
            }
            break;

        case MMD_REG_CH0DIV:
            break;

        case MMD_REG_CH1DIV:
            mmd_ctx->MMD.baud = (mmd_ctx->MMD.baud >> 8) | (Data << 8);
            mmd_config_line(&mmd_dev, &mmd_tmln[0], mmd_ctx->MMD.baud);
            break;

        case MMD_REG_CH2DIV:
            mmdm_ctx->baud = (mmdm_ctx->baud >> 8) | (Data << 8);
            mmd_config_line(&mmdm_dev, &mmdm_tmln[0], mmdm_ctx->baud);
            break;

        case MMD_REG_CENDATA:
            break;

        case MMD_REG_ROMCTL:
            mmd_ctx->romEnabled = TRUE;
            break;

        case MMD_REG_CDATA:
            mmd_ctx->MMD.txd = Data;
            mmd_ctx->MMD.txp = TRUE;    /* Set TX pending flag */
            mmd_ctx->MMD.cstatus &= ~MMD_STAT_TXRDY;
            break;

        case MMD_REG_MDATA:
            mmdm_ctx->txd = Data;
            mmdm_ctx->txp = TRUE;    /* Set TX pending flag */
            mmdm_ctx->status &= ~MMD_STAT_TXRDY;
            break;

        case MMD_REG_VFO:
        case MMD_REG_MOTOR:
            /* Set motor on/off */
            mmd_ctx->MMD.motor[mmd_ctx->drive] &= ~MMD_STAT_MTR;
            mmd_ctx->MMD.motor[mmd_ctx->drive] |= Data;
            break;

        case MMD_REG_FDCDATA:
            MMD_FDC_Write_Data(Data);
            break;

        default:
            sim_debug(ERROR_MSG, &mmd_dev, "WRITE Invalid I/O Address %02x (%02x)\n",
                Addr & 0xFF, Addr & 0x01);
            break;
    }

    return(0xff);
}

static void MMD_FDC_Write_Data(int32 Data)
{
    switch (mmd_ctx->MMD.phase) {
        case MMD_PHASE_CMDNEW:        /* New Command */
            MMD_Command_New(Data);
            break;

        case MMD_PHASE_CMDSTK:
            MMD_Command_Stack(Data);
            break;

        case MMD_PHASE_EXECWR:
            MMD_Exec_Write(Data);
            break;

        default:
            sim_debug(ERROR_MSG, &mmd_dev, "Invalid Phase (%02x)\n", mmd_ctx->MMD.phase);
            break;
    }
}

static void MMD_Command_New(int32 Data)
{
    mmd_ctx->MMD.cmd = Data & MMD_CMD_MASK;
    mmd_ctx->MMD.cmdflags = Data & ~MMD_CMD_MASK;
    mmd_ctx->MMD.stkidx = 0;
    mmd_ctx->MMD.phase = MMD_PHASE_CMDSTK;

    switch (mmd_ctx->MMD.cmd) {
        case MMD_CMD_SIS:
            MMD_Command_Proc();       /* 0 additional bytes */
            break;

        case MMD_CMD_SDS:
        case MMD_CMD_RECALIB:
            mmd_ctx->MMD.stkcnt = 1;  /* 1 additional byte */
            break;

        case MMD_CMD_SEEK:
        case MMD_CMD_SPECIFY:
            mmd_ctx->MMD.stkcnt = 2;  /* 2 additional bytes */
            break;

        case MMD_CMD_FORMAT:
            mmd_ctx->MMD.stkcnt = 5;  /* 5 additional bytes */
            break;

        case MMD_CMD_READ:
        case MMD_CMD_WRITE:
            mmd_ctx->MMD.stkcnt = 8;  /* 8 additional bytes */
            break;

        default:
            MMD_Command_Proc();       /* 0 additional bytes */
            sim_debug(ERROR_MSG, &mmd_dev, "Invalid FDC Command (%02x)\n", mmd_ctx->MMD.cmd);
            break;
    }

    sim_debug(FDC_MSG, &mmd_dev, "cmd=%02X (%s) status=%02X stkcnt=%d stkidx=%d phase=%d\n",
        Data, MMD_Command_String(mmd_ctx->MMD.cmd), mmd_ctx->MMD.status,
        mmd_ctx->MMD.stkcnt, mmd_ctx->MMD.stkidx, mmd_ctx->MMD.phase);
}

static void MMD_Command_Stack(int32 Data)
{
    mmd_ctx->MMD.stack[mmd_ctx->MMD.stkidx++] = Data;

    if (--mmd_ctx->MMD.stkcnt == 0) {
        MMD_Command_Proc();   /* Process Command */
    }
}

static void MMD_Command_Proc(void)
{

    switch (mmd_ctx->MMD.cmd) {
        case MMD_CMD_SPECIFY:
            mmd_ctx->MMD.stat0[mmd_ctx->drive] &= ~MMD_STAT0_ICMASK;
            mmd_ctx->MMD.phase = MMD_PHASE_CMDNEW;
            break;

        case MMD_CMD_RECALIB:
            mmd_ctx->drive = mmd_ctx->MMD.stack[0] & 0x03;
            mmd_ctx->MMD.c[mmd_ctx->drive] = 0;
            mmd_ctx->MMD.pcn = 0;
            mmd_ctx->MMD.stat0[mmd_ctx->drive] = mmd_ctx->MMD.stack[0] & 0x07;
            mmd_ctx->MMD.stat0[mmd_ctx->drive] |= MMD_STAT0_SE;
            mmd_ctx->MMD.stat3[mmd_ctx->drive] |= MMD_STAT3_TO;
            mmd_ctx->MMD.phase = MMD_PHASE_CMDNEW;
            intZ80();   /* Interrupt Z80 */
            break;

        case MMD_CMD_SEEK:
            mmd_ctx->drive = mmd_ctx->MMD.stack[0] & 0x03;
            mmd_ctx->MMD.hd[mmd_ctx->drive] = (mmd_ctx->MMD.stack[0] & 0x04) >> 2;
            mmd_ctx->MMD.c[mmd_ctx->drive] = mmd_ctx->MMD.stack[1];
            mmd_ctx->MMD.pcn = mmd_ctx->MMD.stack[1];
            mmd_ctx->MMD.stat3[mmd_ctx->drive] &= ~MMD_STAT3_TO;
            if (!mmd_ctx->MMD.c[mmd_ctx->drive]) {
                mmd_ctx->MMD.stat3[mmd_ctx->drive] |= MMD_STAT3_TO;
            }
            mmd_ctx->MMD.stat0[mmd_ctx->drive] = mmd_ctx->MMD.stack[0] & 0x07;
            mmd_ctx->MMD.stat0[mmd_ctx->drive] |= MMD_STAT0_SE;
            mmd_ctx->MMD.phase = MMD_PHASE_CMDNEW;
            break;

        case MMD_CMD_SDS:
            mmd_ctx->drive = mmd_ctx->MMD.stack[0] & 0x03;
            mmd_ctx->MMD.hd[mmd_ctx->drive] = (mmd_ctx->MMD.stack[0] & 0x04) >> 2;
            mmd_ctx->MMD.stack[0] = mmd_ctx->MMD.stat3[mmd_ctx->drive];
            mmd_ctx->MMD.stkcnt = 1;
            mmd_ctx->MMD.stkidx = 0;
            mmd_ctx->MMD.status |= MMD_STAT_DIO;
            mmd_ctx->MMD.phase = MMD_PHASE_RSLTSTK;
            mmd_ctx->MMD.stat0[mmd_ctx->drive] = mmd_ctx->MMD.stack[0] & 0x07;
            break;

        case MMD_CMD_SIS:
            mmd_ctx->MMD.stack[0] = mmd_ctx->MMD.sto;
            mmd_ctx->MMD.stack[1] = mmd_ctx->MMD.pcn;
            mmd_ctx->MMD.stkcnt = 2;
            mmd_ctx->MMD.stkidx = 0;
            mmd_ctx->MMD.status |= MMD_STAT_DIO;
            mmd_ctx->MMD.phase = MMD_PHASE_RSLTSTK;
            mmd_ctx->MMD.stat0[mmd_ctx->drive] &= ~MMD_STAT0_ICMASK;
            break;

        case MMD_CMD_FORMAT:
            mmd_ctx->drive = mmd_ctx->MMD.stack[0] & 0x03;
            mmd_ctx->MMD.hd[mmd_ctx->drive] = (mmd_ctx->MMD.stack[0] & 0x04) >> 2;
            mmd_ctx->MMD.n = mmd_ctx->MMD.stack[1];
            mmd_ctx->MMD.sc = mmd_ctx->MMD.stack[2];
            mmd_ctx->MMD.gpl = mmd_ctx->MMD.stack[3];
            mmd_ctx->MMD.d = mmd_ctx->MMD.stack[4];
            MMD_Exec_Format();
            break;

        case MMD_CMD_READ:
        case MMD_CMD_WRITE:
            mmd_ctx->drive = mmd_ctx->MMD.stack[0] & 0x03;
            mmd_ctx->MMD.hd[mmd_ctx->drive] = (mmd_ctx->MMD.stack[0] & 0x04) >> 2;
            mmd_ctx->MMD.c[mmd_ctx->drive] = mmd_ctx->MMD.stack[1];
            mmd_ctx->MMD.h = mmd_ctx->MMD.stack[2];
            mmd_ctx->MMD.r = mmd_ctx->MMD.stack[3];
            mmd_ctx->MMD.n = mmd_ctx->MMD.stack[4];
            mmd_ctx->MMD.eot = mmd_ctx->MMD.stack[5];
            mmd_ctx->MMD.gpl = mmd_ctx->MMD.stack[6];
            mmd_ctx->MMD.dtl = mmd_ctx->MMD.stack[7];
            mmd_ctx->MMD.stat0[mmd_ctx->drive] = mmd_ctx->MMD.stack[0] & 0x07;

            if (mmd_ctx->MMD.cmd == MMD_CMD_READ) {
                mmd_ctx->MMD.stat0[mmd_ctx->drive] |= MMD_Read_Sector(mmd_ctx->drive);
                mmd_ctx->MMD.status |= (MMD_STAT_BUSY | MMD_STAT_EXM | MMD_STAT_DIO);
                mmd_ctx->MMD.phase = MMD_PHASE_EXECRD;
            }
            else {
                mmd_ctx->MMD.seccnt = MMD_SECTOR_LEN;
                mmd_ctx->MMD.secidx = 0;
                mmd_ctx->MMD.status |= (MMD_STAT_BUSY | MMD_STAT_EXM);
                mmd_ctx->MMD.phase = MMD_PHASE_EXECWR;
            }

            if (mmd_ctx->MMD.stat0[mmd_ctx->drive] & MMD_STAT0_AT1) {
                mmd_ctx->MMD.status &= ~MMD_STAT_RQM;   /* Abnormal Termination */
            }
            break;

        default:
            mmd_ctx->MMD.stat0[mmd_ctx->drive] &= ~MMD_STAT0_ICMASK;
            mmd_ctx->MMD.stat0[mmd_ctx->drive] = MMD_STAT0_IC;
            mmd_ctx->MMD.stack[0] = mmd_ctx->MMD.stat0[mmd_ctx->drive];
            mmd_ctx->MMD.stkcnt = 1;
            mmd_ctx->MMD.stkidx = 0;
            mmd_ctx->MMD.status |= MMD_STAT_DIO;
            mmd_ctx->MMD.phase = MMD_PHASE_EXECRD;
            break;
    }
}

static const char * MMD_Command_String(int32 command)
{
    char *string;

    switch (command & MMD_CMD_MASK) {
        case MMD_CMD_SPECIFY:
            string = "SPECIFY";
            break;

        case MMD_CMD_RECALIB:
            string = "RECALIB";
            break;

        case MMD_CMD_SEEK:
            string = "SEEK";
            break;

        case MMD_CMD_FORMAT:
            string = "FORMAT";
            break;

        case MMD_CMD_READ:
            string = "READ";
            break;

        case MMD_CMD_WRITE:
            string = "WRITE";
            break;

        case MMD_CMD_SDS:
            string = "SDS";
            break;

        case MMD_CMD_SIS:
            string = "SIS";
            break;

        default:
            string = "UNRECOGNIZED COMMAND";
            break;
    }

    return string;
}

static uint8 MMD_Result_Stack(void)
{
    uint8 cData;

    cData = mmd_ctx->MMD.stack[mmd_ctx->MMD.stkidx++];

    if (--mmd_ctx->MMD.stkcnt == 0) {
        mmd_ctx->MMD.phase = MMD_PHASE_CMDNEW;   /* Command Done */
        mmd_ctx->MMD.status &= ~(MMD_STAT_DIO | MMD_STAT_BUSY);
    }

    return cData;
}

static uint8 MMD_Exec_Format(void)
{
    uint8 drive = mmd_ctx->drive;

    /* Set filler byte */
    memset(secbuf, mmd_ctx->MMD.d, sizeof(secbuf));

    for (mmd_ctx->MMD.r = 1; mmd_ctx->MMD.r <= mmd_ctx->MMD.sc; mmd_ctx->MMD.r++) {
        mmd_ctx->MMD.status |= MMD_Write_Sector(drive);
    }

    mmd_ctx->MMD.stack[0] = mmd_ctx->MMD.stat0[drive];
    mmd_ctx->MMD.stack[1] = mmd_ctx->MMD.stat1[drive];
    mmd_ctx->MMD.stack[2] = mmd_ctx->MMD.stat2[drive];
    mmd_ctx->MMD.stack[3] = mmd_ctx->MMD.c[drive];
    mmd_ctx->MMD.stack[4] = mmd_ctx->MMD.hd[drive];
    mmd_ctx->MMD.stack[5] = mmd_ctx->MMD.r;
    mmd_ctx->MMD.stack[6] = mmd_ctx->MMD.n;
    mmd_ctx->MMD.stkidx = 0;
    mmd_ctx->MMD.stkcnt = 7;
    mmd_ctx->MMD.status |= MMD_STAT_DIO;
    mmd_ctx->MMD.phase = MMD_PHASE_RSLTSTK;   /* Exec Done */

    intZ80();

    return 0;
}

static uint8 MMD_Exec_Read(void)
{
    uint8 cData;
    uint8 drive = mmd_ctx->drive;

    cData = secbuf[mmd_ctx->MMD.secidx++];

    if (mmd_ctx->MMD.tc || (--mmd_ctx->MMD.seccnt == 0)) {
        if (mmd_ctx->MMD.tc || (mmd_ctx->MMD.r == mmd_ctx->MMD.eot)) {
            mmd_ctx->MMD.status &= ~MMD_STAT_EXM;
            mmd_ctx->MMD.stack[0] = mmd_ctx->MMD.stat0[drive];
            mmd_ctx->MMD.stack[1] = mmd_ctx->MMD.stat1[drive];
            mmd_ctx->MMD.stack[2] = mmd_ctx->MMD.stat2[drive];
            mmd_ctx->MMD.stack[3] = mmd_ctx->MMD.c[drive];
            mmd_ctx->MMD.stack[4] = mmd_ctx->MMD.hd[drive];
            mmd_ctx->MMD.stack[5] = mmd_ctx->MMD.r;
            mmd_ctx->MMD.stack[6] = mmd_ctx->MMD.n;
            mmd_ctx->MMD.stkidx = 0;
            mmd_ctx->MMD.stkcnt = 7;
            mmd_ctx->MMD.phase = MMD_PHASE_RSLTSTK;   /* Exec Done */
            mmd_ctx->MMD.tc = FALSE;
        }
        else {
            mmd_ctx->MMD.r++;
            mmd_ctx->MMD.stat0[drive] |= MMD_Read_Sector(drive);
        }
    }

    intZ80();   /* Interrupt Z80 */

    return cData;
}

static uint8 MMD_Exec_Write(int32 Data)
{
    uint8 drive = mmd_ctx->drive;

    secbuf[mmd_ctx->MMD.secidx++] = Data;

    if (mmd_ctx->MMD.tc || (--mmd_ctx->MMD.seccnt == 0)) {
        mmd_ctx->MMD.stat0[drive] |= MMD_Write_Sector(drive);

        if (mmd_ctx->MMD.tc || (mmd_ctx->MMD.r == mmd_ctx->MMD.eot)) {
            mmd_ctx->MMD.status &= ~MMD_STAT_EXM;
            mmd_ctx->MMD.stack[0] = mmd_ctx->MMD.stat0[drive];
            mmd_ctx->MMD.stack[1] = mmd_ctx->MMD.stat1[drive];
            mmd_ctx->MMD.stack[2] = mmd_ctx->MMD.stat2[drive];
            mmd_ctx->MMD.stack[3] = mmd_ctx->MMD.c[drive];
            mmd_ctx->MMD.stack[4] = mmd_ctx->MMD.hd[drive];
            mmd_ctx->MMD.stack[5] = mmd_ctx->MMD.r;
            mmd_ctx->MMD.stack[6] = mmd_ctx->MMD.n;
            mmd_ctx->MMD.stkidx = 0;
            mmd_ctx->MMD.stkcnt = 7;
            mmd_ctx->MMD.status |= MMD_STAT_DIO;
            mmd_ctx->MMD.phase = MMD_PHASE_RSLTSTK;   /* Exec Done */
            mmd_ctx->MMD.tc = FALSE;
        }
        else {
            mmd_ctx->MMD.r++;
        }
    }

    intZ80();   /* Interrupt Z80 */

    return Data;
}

static uint8 MMD_Term_Count(void)
{
    if (mmd_ctx->MMD.status & MMD_STAT_EXM) {
        mmd_ctx->MMD.tc = TRUE;
    }

    return 0;
}

/* Seek to sector on disk */
static uint8 MMD_Seek_Sector(uint8 drive)
{
    uint32 offset;
    uint8 track;

    /* If no disk mounted, return error */
    if (mmd_ctx->uptr[drive]->fileref == NULL) {
        return MMD_STAT0_AT1 | MMD_STAT0_NR;
    }

    track = mmd_ctx->MMD.c[drive];
    if (mmd_ctx->MMD.stat3[drive] & MMD_STAT3_TS) {   /* 2 Sided? */
        track = mmd_ctx->MMD.c[drive] * 2 + mmd_ctx->MMD.hd[drive];
    }
    offset = track * MMD_BPT + (mmd_ctx->MMD.r - 1) * MMD_SECTOR_LEN;

    if (sim_fseek(mmd_ctx->uptr[drive]->fileref, offset, SEEK_SET) != 0) {
        sim_debug(ERROR_MSG, &mmd_dev, "sim_fseek error.\n");
        return MMD_STAT0_AT1 | MMD_STAT0_EC;
    }

    sim_debug(FDC_MSG, &mmd_dev, "Seek_Sector: drive=%d head=%d track=%d sector=%d offset=%d\n",
        drive, mmd_ctx->MMD.hd[drive], track, mmd_ctx->MMD.r, offset);

    return 0;
}

/* Read sector from disk */
static uint8 MMD_Read_Sector(uint8 drive)
{
    uint8 r;

    if ((r = MMD_Seek_Sector(drive)) != 0) {
        return r;
    }

    if (sim_fread(secbuf, 1, MMD_SECTOR_LEN, mmd_ctx->uptr[drive]->fileref) != MMD_SECTOR_LEN) {
        sim_debug(ERROR_MSG, &mmd_dev, "sim_fread error.\n");
        return MMD_STAT0_AT1 | MMD_STAT0_EC;
    }

    if(mmd_dev.dctrl & RBUF_MSG) {
        MMD_Dump_Sector();
    }

    mmd_ctx->MMD.seccnt = MMD_SECTOR_LEN;
    mmd_ctx->MMD.secidx = 0;

    return 0;
}

/* Write sector to disk */
static uint8 MMD_Write_Sector(uint8 drive)
{
    uint8 r;

    if ((r = MMD_Seek_Sector(drive)) != 0) {
        return r;
    }

    if(mmd_dev.dctrl & WBUF_MSG) {
        MMD_Dump_Sector();
    }

    if (sim_fwrite(secbuf, 1, MMD_SECTOR_LEN, mmd_ctx->uptr[drive]->fileref) != MMD_SECTOR_LEN) {
        sim_debug(ERROR_MSG, &mmd_dev, "sim_fwrite error.\n");
        return MMD_STAT0_AT1 | MMD_STAT0_EC;
    }

    return r;
}

static void MMD_Dump_Sector(void) {
    int32 i;

    for (i = 0; i < 32; i++) {
        sim_printf("%02X ", secbuf[i]);
        if (((i+1) & 0xf) == 0) {
            sim_printf("\n\r");
        }
    }
    sim_printf("\n\r");
}

