use super::*;

fn new_vdp() -> Vdp {
    Vdp::new(
        TimingMode::Ntsc,
        VdpConfig {
            enforce_sprite_limits: true,
            non_linear_color_scale: false,
            deinterlace: true,
            render_vertical_border: false,
            render_horizontal_border: false,
            plane_a_enabled: true,
            plane_b_enabled: true,
            window_enabled: true,
            sprites_enabled: true,
            backdrop_enabled: true,
        },
    )
}

fn v_counter_test(expected: &[u8], vdp: &mut Vdp) {
    for (line, expected) in expected.iter().copied().enumerate() {
        vdp.state.scanline = line as u16;
        let VCounter { counter, .. } = vdp.v_counter(0);
        assert_eq!(
            expected, counter,
            "V counter for line {line}: expected 0x{expected:02X}, actual 0x{counter:02X}"
        )
    }
}

#[test]
fn v_counter_ntsc_progressive() {
    const EXPECTED: &[u8; NTSC_SCANLINES_PER_FRAME as usize] = &[
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
        0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D,
        0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C,
        0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B,
        0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A,
        0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
        0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
        0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
        0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86,
        0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95,
        0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4,
        0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB0, 0xB1, 0xB2, 0xB3,
        0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1, 0xC2,
        0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 0xD1,
        0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0,
        0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9,
        0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8,
        0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Ntsc;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v28_progressive() {
    const EXPECTED: &[u8; PAL_SCANLINES_PER_FRAME as usize] = &[
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
        0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D,
        0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C,
        0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B,
        0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A,
        0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
        0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
        0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
        0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86,
        0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95,
        0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4,
        0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB0, 0xB1, 0xB2, 0xB3,
        0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1, 0xC2,
        0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 0xD1,
        0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0,
        0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF,
        0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE,
        0xFF, 0x00, 0x01, 0x02, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 0xD1, 0xD2, 0xD3, 0xD4,
        0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0, 0xE1, 0xE2, 0xE3,
        0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2,
        0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::TwentyEightCell;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v30_progressive() {
    const EXPECTED: &[u8; PAL_SCANLINES_PER_FRAME as usize] = &[
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
        0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D,
        0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C,
        0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B,
        0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A,
        0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
        0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
        0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
        0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86,
        0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95,
        0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4,
        0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB0, 0xB1, 0xB2, 0xB3,
        0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1, 0xC2,
        0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 0xD1,
        0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0,
        0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF,
        0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE,
        0xFF, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0xD2, 0xD3, 0xD4,
        0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0, 0xE1, 0xE2, 0xE3,
        0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2,
        0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::ThirtyCell;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_ntsc_interlaced_single_even_frame() {
    const EXPECTED: &[u8; NTSC_SCANLINES_PER_FRAME as usize] = &[
        0x00, 0x00, 0x02, 0x02, 0x04, 0x04, 0x06, 0x06, 0x08, 0x08, 0x0A, 0x0A, 0x0C, 0x0C, 0x0E,
        0x0E, 0x10, 0x10, 0x12, 0x12, 0x14, 0x14, 0x16, 0x16, 0x18, 0x18, 0x1A, 0x1A, 0x1C, 0x1C,
        0x1E, 0x1E, 0x20, 0x20, 0x22, 0x22, 0x24, 0x24, 0x26, 0x26, 0x28, 0x28, 0x2A, 0x2A, 0x2C,
        0x2C, 0x2E, 0x2E, 0x30, 0x30, 0x32, 0x32, 0x34, 0x34, 0x36, 0x36, 0x38, 0x38, 0x3A, 0x3A,
        0x3C, 0x3C, 0x3E, 0x3E, 0x40, 0x40, 0x42, 0x42, 0x44, 0x44, 0x46, 0x46, 0x48, 0x48, 0x4A,
        0x4A, 0x4C, 0x4C, 0x4E, 0x4E, 0x50, 0x50, 0x52, 0x52, 0x54, 0x54, 0x56, 0x56, 0x58, 0x58,
        0x5A, 0x5A, 0x5C, 0x5C, 0x5E, 0x5E, 0x60, 0x60, 0x62, 0x62, 0x64, 0x64, 0x66, 0x66, 0x68,
        0x68, 0x6A, 0x6A, 0x6C, 0x6C, 0x6E, 0x6E, 0x70, 0x70, 0x72, 0x72, 0x74, 0x74, 0x76, 0x76,
        0x78, 0x78, 0x7A, 0x7A, 0x7C, 0x7C, 0x7E, 0x7E, 0x80, 0x80, 0x82, 0x82, 0x84, 0x84, 0x86,
        0x86, 0x88, 0x88, 0x8A, 0x8A, 0x8C, 0x8C, 0x8E, 0x8E, 0x90, 0x90, 0x92, 0x92, 0x94, 0x94,
        0x96, 0x96, 0x98, 0x98, 0x9A, 0x9A, 0x9C, 0x9C, 0x9E, 0x9E, 0xA0, 0xA0, 0xA2, 0xA2, 0xA4,
        0xA4, 0xA6, 0xA6, 0xA8, 0xA8, 0xAA, 0xAA, 0xAC, 0xAC, 0xAE, 0xAE, 0xB0, 0xB0, 0xB2, 0xB2,
        0xB4, 0xB4, 0xB6, 0xB6, 0xB8, 0xB8, 0xBA, 0xBA, 0xBC, 0xBC, 0xBE, 0xBE, 0xC0, 0xC0, 0xC2,
        0xC2, 0xC4, 0xC4, 0xC6, 0xC6, 0xC8, 0xC8, 0xCA, 0xCA, 0xCC, 0xCC, 0xCE, 0xCE, 0xD0, 0xD0,
        0xD2, 0xD2, 0xD4, 0xD4, 0xD6, 0xD6, 0xD8, 0xD8, 0xDA, 0xDA, 0xDC, 0xDC, 0xDE, 0xDE, 0xE0,
        0xE0, 0xE2, 0xE2, 0xE4, 0xE4, 0xE6, 0xE6, 0xE8, 0xE8, 0xEA, 0xE5, 0xE7, 0xE7, 0xE9, 0xE9,
        0xEB, 0xEB, 0xED, 0xED, 0xEF, 0xEF, 0xF1, 0xF1, 0xF3, 0xF3, 0xF5, 0xF5, 0xF7, 0xF7, 0xF9,
        0xF9, 0xFB, 0xFB, 0xFD, 0xFD, 0xFF, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Ntsc;
    vdp.registers.interlacing_mode = InterlacingMode::Interlaced;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = false;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_ntsc_interlaced_single_odd_frame() {
    const EXPECTED: &[u8; (NTSC_SCANLINES_PER_FRAME + 1) as usize] = &[
        0x00, 0x00, 0x02, 0x02, 0x04, 0x04, 0x06, 0x06, 0x08, 0x08, 0x0A, 0x0A, 0x0C, 0x0C, 0x0E,
        0x0E, 0x10, 0x10, 0x12, 0x12, 0x14, 0x14, 0x16, 0x16, 0x18, 0x18, 0x1A, 0x1A, 0x1C, 0x1C,
        0x1E, 0x1E, 0x20, 0x20, 0x22, 0x22, 0x24, 0x24, 0x26, 0x26, 0x28, 0x28, 0x2A, 0x2A, 0x2C,
        0x2C, 0x2E, 0x2E, 0x30, 0x30, 0x32, 0x32, 0x34, 0x34, 0x36, 0x36, 0x38, 0x38, 0x3A, 0x3A,
        0x3C, 0x3C, 0x3E, 0x3E, 0x40, 0x40, 0x42, 0x42, 0x44, 0x44, 0x46, 0x46, 0x48, 0x48, 0x4A,
        0x4A, 0x4C, 0x4C, 0x4E, 0x4E, 0x50, 0x50, 0x52, 0x52, 0x54, 0x54, 0x56, 0x56, 0x58, 0x58,
        0x5A, 0x5A, 0x5C, 0x5C, 0x5E, 0x5E, 0x60, 0x60, 0x62, 0x62, 0x64, 0x64, 0x66, 0x66, 0x68,
        0x68, 0x6A, 0x6A, 0x6C, 0x6C, 0x6E, 0x6E, 0x70, 0x70, 0x72, 0x72, 0x74, 0x74, 0x76, 0x76,
        0x78, 0x78, 0x7A, 0x7A, 0x7C, 0x7C, 0x7E, 0x7E, 0x80, 0x80, 0x82, 0x82, 0x84, 0x84, 0x86,
        0x86, 0x88, 0x88, 0x8A, 0x8A, 0x8C, 0x8C, 0x8E, 0x8E, 0x90, 0x90, 0x92, 0x92, 0x94, 0x94,
        0x96, 0x96, 0x98, 0x98, 0x9A, 0x9A, 0x9C, 0x9C, 0x9E, 0x9E, 0xA0, 0xA0, 0xA2, 0xA2, 0xA4,
        0xA4, 0xA6, 0xA6, 0xA8, 0xA8, 0xAA, 0xAA, 0xAC, 0xAC, 0xAE, 0xAE, 0xB0, 0xB0, 0xB2, 0xB2,
        0xB4, 0xB4, 0xB6, 0xB6, 0xB8, 0xB8, 0xBA, 0xBA, 0xBC, 0xBC, 0xBE, 0xBE, 0xC0, 0xC0, 0xC2,
        0xC2, 0xC4, 0xC4, 0xC6, 0xC6, 0xC8, 0xC8, 0xCA, 0xCA, 0xCC, 0xCC, 0xCE, 0xCE, 0xD0, 0xD0,
        0xD2, 0xD2, 0xD4, 0xD4, 0xD6, 0xD6, 0xD8, 0xD8, 0xDA, 0xDA, 0xDC, 0xDC, 0xDE, 0xDE, 0xE0,
        0xE0, 0xE2, 0xE2, 0xE4, 0xE4, 0xE6, 0xE6, 0xE8, 0xE8, 0xEA, 0xE5, 0xE5, 0xE7, 0xE7, 0xE9,
        0xE9, 0xEB, 0xEB, 0xED, 0xED, 0xEF, 0xEF, 0xF1, 0xF1, 0xF3, 0xF3, 0xF5, 0xF5, 0xF7, 0xF7,
        0xF9, 0xF9, 0xFB, 0xFB, 0xFD, 0xFD, 0xFF, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Ntsc;
    vdp.registers.interlacing_mode = InterlacingMode::Interlaced;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = true;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v28_interlaced_single_even_frame() {
    const EXPECTED: &[u8; (PAL_SCANLINES_PER_FRAME - 1) as usize] = &[
        0x00, 0x00, 0x02, 0x02, 0x04, 0x04, 0x06, 0x06, 0x08, 0x08, 0x0A, 0x0A, 0x0C, 0x0C, 0x0E,
        0x0E, 0x10, 0x10, 0x12, 0x12, 0x14, 0x14, 0x16, 0x16, 0x18, 0x18, 0x1A, 0x1A, 0x1C, 0x1C,
        0x1E, 0x1E, 0x20, 0x20, 0x22, 0x22, 0x24, 0x24, 0x26, 0x26, 0x28, 0x28, 0x2A, 0x2A, 0x2C,
        0x2C, 0x2E, 0x2E, 0x30, 0x30, 0x32, 0x32, 0x34, 0x34, 0x36, 0x36, 0x38, 0x38, 0x3A, 0x3A,
        0x3C, 0x3C, 0x3E, 0x3E, 0x40, 0x40, 0x42, 0x42, 0x44, 0x44, 0x46, 0x46, 0x48, 0x48, 0x4A,
        0x4A, 0x4C, 0x4C, 0x4E, 0x4E, 0x50, 0x50, 0x52, 0x52, 0x54, 0x54, 0x56, 0x56, 0x58, 0x58,
        0x5A, 0x5A, 0x5C, 0x5C, 0x5E, 0x5E, 0x60, 0x60, 0x62, 0x62, 0x64, 0x64, 0x66, 0x66, 0x68,
        0x68, 0x6A, 0x6A, 0x6C, 0x6C, 0x6E, 0x6E, 0x70, 0x70, 0x72, 0x72, 0x74, 0x74, 0x76, 0x76,
        0x78, 0x78, 0x7A, 0x7A, 0x7C, 0x7C, 0x7E, 0x7E, 0x80, 0x80, 0x82, 0x82, 0x84, 0x84, 0x86,
        0x86, 0x88, 0x88, 0x8A, 0x8A, 0x8C, 0x8C, 0x8E, 0x8E, 0x90, 0x90, 0x92, 0x92, 0x94, 0x94,
        0x96, 0x96, 0x98, 0x98, 0x9A, 0x9A, 0x9C, 0x9C, 0x9E, 0x9E, 0xA0, 0xA0, 0xA2, 0xA2, 0xA4,
        0xA4, 0xA6, 0xA6, 0xA8, 0xA8, 0xAA, 0xAA, 0xAC, 0xAC, 0xAE, 0xAE, 0xB0, 0xB0, 0xB2, 0xB2,
        0xB4, 0xB4, 0xB6, 0xB6, 0xB8, 0xB8, 0xBA, 0xBA, 0xBC, 0xBC, 0xBE, 0xBE, 0xC0, 0xC0, 0xC2,
        0xC2, 0xC4, 0xC4, 0xC6, 0xC6, 0xC8, 0xC8, 0xCA, 0xCA, 0xCC, 0xCC, 0xCE, 0xCE, 0xD0, 0xD0,
        0xD2, 0xD2, 0xD4, 0xD4, 0xD6, 0xD6, 0xD8, 0xD8, 0xDA, 0xDA, 0xDC, 0xDC, 0xDE, 0xDE, 0xE0,
        0xE0, 0xE2, 0xE2, 0xE4, 0xE4, 0xE6, 0xE6, 0xE8, 0xE8, 0xEA, 0xEA, 0xEC, 0xEC, 0xEE, 0xEE,
        0xF0, 0xF0, 0xF2, 0xF2, 0xF4, 0xF4, 0xF6, 0xF6, 0xF8, 0xF8, 0xFA, 0xFA, 0xFC, 0xFC, 0xFE,
        0xFE, 0x01, 0x01, 0xCB, 0xCB, 0xCD, 0xCD, 0xCF, 0xCF, 0xD1, 0xD1, 0xD3, 0xD3, 0xD5, 0xD5,
        0xD7, 0xD7, 0xD9, 0xD9, 0xDB, 0xDB, 0xDD, 0xDD, 0xDF, 0xDF, 0xE1, 0xE1, 0xE3, 0xE3, 0xE5,
        0xE5, 0xE7, 0xE7, 0xE9, 0xE9, 0xEB, 0xEB, 0xED, 0xED, 0xEF, 0xEF, 0xF1, 0xF1, 0xF3, 0xF3,
        0xF5, 0xF5, 0xF7, 0xF7, 0xF9, 0xF9, 0xFB, 0xFB, 0xFD, 0xFD, 0xFF, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::TwentyEightCell;
    vdp.registers.interlacing_mode = InterlacingMode::Interlaced;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = false;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v28_interlaced_single_odd_frame() {
    const EXPECTED: &[u8; PAL_SCANLINES_PER_FRAME as usize] = &[
        0x00, 0x00, 0x02, 0x02, 0x04, 0x04, 0x06, 0x06, 0x08, 0x08, 0x0A, 0x0A, 0x0C, 0x0C, 0x0E,
        0x0E, 0x10, 0x10, 0x12, 0x12, 0x14, 0x14, 0x16, 0x16, 0x18, 0x18, 0x1A, 0x1A, 0x1C, 0x1C,
        0x1E, 0x1E, 0x20, 0x20, 0x22, 0x22, 0x24, 0x24, 0x26, 0x26, 0x28, 0x28, 0x2A, 0x2A, 0x2C,
        0x2C, 0x2E, 0x2E, 0x30, 0x30, 0x32, 0x32, 0x34, 0x34, 0x36, 0x36, 0x38, 0x38, 0x3A, 0x3A,
        0x3C, 0x3C, 0x3E, 0x3E, 0x40, 0x40, 0x42, 0x42, 0x44, 0x44, 0x46, 0x46, 0x48, 0x48, 0x4A,
        0x4A, 0x4C, 0x4C, 0x4E, 0x4E, 0x50, 0x50, 0x52, 0x52, 0x54, 0x54, 0x56, 0x56, 0x58, 0x58,
        0x5A, 0x5A, 0x5C, 0x5C, 0x5E, 0x5E, 0x60, 0x60, 0x62, 0x62, 0x64, 0x64, 0x66, 0x66, 0x68,
        0x68, 0x6A, 0x6A, 0x6C, 0x6C, 0x6E, 0x6E, 0x70, 0x70, 0x72, 0x72, 0x74, 0x74, 0x76, 0x76,
        0x78, 0x78, 0x7A, 0x7A, 0x7C, 0x7C, 0x7E, 0x7E, 0x80, 0x80, 0x82, 0x82, 0x84, 0x84, 0x86,
        0x86, 0x88, 0x88, 0x8A, 0x8A, 0x8C, 0x8C, 0x8E, 0x8E, 0x90, 0x90, 0x92, 0x92, 0x94, 0x94,
        0x96, 0x96, 0x98, 0x98, 0x9A, 0x9A, 0x9C, 0x9C, 0x9E, 0x9E, 0xA0, 0xA0, 0xA2, 0xA2, 0xA4,
        0xA4, 0xA6, 0xA6, 0xA8, 0xA8, 0xAA, 0xAA, 0xAC, 0xAC, 0xAE, 0xAE, 0xB0, 0xB0, 0xB2, 0xB2,
        0xB4, 0xB4, 0xB6, 0xB6, 0xB8, 0xB8, 0xBA, 0xBA, 0xBC, 0xBC, 0xBE, 0xBE, 0xC0, 0xC0, 0xC2,
        0xC2, 0xC4, 0xC4, 0xC6, 0xC6, 0xC8, 0xC8, 0xCA, 0xCA, 0xCC, 0xCC, 0xCE, 0xCE, 0xD0, 0xD0,
        0xD2, 0xD2, 0xD4, 0xD4, 0xD6, 0xD6, 0xD8, 0xD8, 0xDA, 0xDA, 0xDC, 0xDC, 0xDE, 0xDE, 0xE0,
        0xE0, 0xE2, 0xE2, 0xE4, 0xE4, 0xE6, 0xE6, 0xE8, 0xE8, 0xEA, 0xEA, 0xEC, 0xEC, 0xEE, 0xEE,
        0xF0, 0xF0, 0xF2, 0xF2, 0xF4, 0xF4, 0xF6, 0xF6, 0xF8, 0xF8, 0xFA, 0xFA, 0xFC, 0xFC, 0xFE,
        0xFE, 0x01, 0x01, 0xC9, 0xCB, 0xCB, 0xCD, 0xCD, 0xCF, 0xCF, 0xD1, 0xD1, 0xD3, 0xD3, 0xD5,
        0xD5, 0xD7, 0xD7, 0xD9, 0xD9, 0xDB, 0xDB, 0xDD, 0xDD, 0xDF, 0xDF, 0xE1, 0xE1, 0xE3, 0xE3,
        0xE5, 0xE5, 0xE7, 0xE7, 0xE9, 0xE9, 0xEB, 0xEB, 0xED, 0xED, 0xEF, 0xEF, 0xF1, 0xF1, 0xF3,
        0xF3, 0xF5, 0xF5, 0xF7, 0xF7, 0xF9, 0xF9, 0xFB, 0xFB, 0xFD, 0xFD, 0xFF, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::TwentyEightCell;
    vdp.registers.interlacing_mode = InterlacingMode::Interlaced;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = true;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v30_interlaced_single_even_frame() {
    const EXPECTED: &[u8; (PAL_SCANLINES_PER_FRAME - 1) as usize] = &[
        0x00, 0x00, 0x02, 0x02, 0x04, 0x04, 0x06, 0x06, 0x08, 0x08, 0x0A, 0x0A, 0x0C, 0x0C, 0x0E,
        0x0E, 0x10, 0x10, 0x12, 0x12, 0x14, 0x14, 0x16, 0x16, 0x18, 0x18, 0x1A, 0x1A, 0x1C, 0x1C,
        0x1E, 0x1E, 0x20, 0x20, 0x22, 0x22, 0x24, 0x24, 0x26, 0x26, 0x28, 0x28, 0x2A, 0x2A, 0x2C,
        0x2C, 0x2E, 0x2E, 0x30, 0x30, 0x32, 0x32, 0x34, 0x34, 0x36, 0x36, 0x38, 0x38, 0x3A, 0x3A,
        0x3C, 0x3C, 0x3E, 0x3E, 0x40, 0x40, 0x42, 0x42, 0x44, 0x44, 0x46, 0x46, 0x48, 0x48, 0x4A,
        0x4A, 0x4C, 0x4C, 0x4E, 0x4E, 0x50, 0x50, 0x52, 0x52, 0x54, 0x54, 0x56, 0x56, 0x58, 0x58,
        0x5A, 0x5A, 0x5C, 0x5C, 0x5E, 0x5E, 0x60, 0x60, 0x62, 0x62, 0x64, 0x64, 0x66, 0x66, 0x68,
        0x68, 0x6A, 0x6A, 0x6C, 0x6C, 0x6E, 0x6E, 0x70, 0x70, 0x72, 0x72, 0x74, 0x74, 0x76, 0x76,
        0x78, 0x78, 0x7A, 0x7A, 0x7C, 0x7C, 0x7E, 0x7E, 0x80, 0x80, 0x82, 0x82, 0x84, 0x84, 0x86,
        0x86, 0x88, 0x88, 0x8A, 0x8A, 0x8C, 0x8C, 0x8E, 0x8E, 0x90, 0x90, 0x92, 0x92, 0x94, 0x94,
        0x96, 0x96, 0x98, 0x98, 0x9A, 0x9A, 0x9C, 0x9C, 0x9E, 0x9E, 0xA0, 0xA0, 0xA2, 0xA2, 0xA4,
        0xA4, 0xA6, 0xA6, 0xA8, 0xA8, 0xAA, 0xAA, 0xAC, 0xAC, 0xAE, 0xAE, 0xB0, 0xB0, 0xB2, 0xB2,
        0xB4, 0xB4, 0xB6, 0xB6, 0xB8, 0xB8, 0xBA, 0xBA, 0xBC, 0xBC, 0xBE, 0xBE, 0xC0, 0xC0, 0xC2,
        0xC2, 0xC4, 0xC4, 0xC6, 0xC6, 0xC8, 0xC8, 0xCA, 0xCA, 0xCC, 0xCC, 0xCE, 0xCE, 0xD0, 0xD0,
        0xD2, 0xD2, 0xD4, 0xD4, 0xD6, 0xD6, 0xD8, 0xD8, 0xDA, 0xDA, 0xDC, 0xDC, 0xDE, 0xDE, 0xE0,
        0xE0, 0xE2, 0xE2, 0xE4, 0xE4, 0xE6, 0xE6, 0xE8, 0xE8, 0xEA, 0xEA, 0xEC, 0xEC, 0xEE, 0xEE,
        0xF0, 0xF0, 0xF2, 0xF2, 0xF4, 0xF4, 0xF6, 0xF6, 0xF8, 0xF8, 0xFA, 0xFA, 0xFC, 0xFC, 0xFE,
        0xFE, 0x01, 0x01, 0x03, 0x03, 0x05, 0x05, 0x07, 0x07, 0x09, 0x09, 0xD3, 0xD3, 0xD5, 0xD5,
        0xD7, 0xD7, 0xD9, 0xD9, 0xDB, 0xDB, 0xDD, 0xDD, 0xDF, 0xDF, 0xE1, 0xE1, 0xE3, 0xE3, 0xE5,
        0xE5, 0xE7, 0xE7, 0xE9, 0xE9, 0xEB, 0xEB, 0xED, 0xED, 0xEF, 0xEF, 0xF1, 0xF1, 0xF3, 0xF3,
        0xF5, 0xF5, 0xF7, 0xF7, 0xF9, 0xF9, 0xFB, 0xFB, 0xFD, 0xFD, 0xFF, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::ThirtyCell;
    vdp.registers.interlacing_mode = InterlacingMode::Interlaced;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = false;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v30_interlaced_single_odd_frame() {
    const EXPECTED: &[u8; PAL_SCANLINES_PER_FRAME as usize] = &[
        0x00, 0x00, 0x02, 0x02, 0x04, 0x04, 0x06, 0x06, 0x08, 0x08, 0x0A, 0x0A, 0x0C, 0x0C, 0x0E,
        0x0E, 0x10, 0x10, 0x12, 0x12, 0x14, 0x14, 0x16, 0x16, 0x18, 0x18, 0x1A, 0x1A, 0x1C, 0x1C,
        0x1E, 0x1E, 0x20, 0x20, 0x22, 0x22, 0x24, 0x24, 0x26, 0x26, 0x28, 0x28, 0x2A, 0x2A, 0x2C,
        0x2C, 0x2E, 0x2E, 0x30, 0x30, 0x32, 0x32, 0x34, 0x34, 0x36, 0x36, 0x38, 0x38, 0x3A, 0x3A,
        0x3C, 0x3C, 0x3E, 0x3E, 0x40, 0x40, 0x42, 0x42, 0x44, 0x44, 0x46, 0x46, 0x48, 0x48, 0x4A,
        0x4A, 0x4C, 0x4C, 0x4E, 0x4E, 0x50, 0x50, 0x52, 0x52, 0x54, 0x54, 0x56, 0x56, 0x58, 0x58,
        0x5A, 0x5A, 0x5C, 0x5C, 0x5E, 0x5E, 0x60, 0x60, 0x62, 0x62, 0x64, 0x64, 0x66, 0x66, 0x68,
        0x68, 0x6A, 0x6A, 0x6C, 0x6C, 0x6E, 0x6E, 0x70, 0x70, 0x72, 0x72, 0x74, 0x74, 0x76, 0x76,
        0x78, 0x78, 0x7A, 0x7A, 0x7C, 0x7C, 0x7E, 0x7E, 0x80, 0x80, 0x82, 0x82, 0x84, 0x84, 0x86,
        0x86, 0x88, 0x88, 0x8A, 0x8A, 0x8C, 0x8C, 0x8E, 0x8E, 0x90, 0x90, 0x92, 0x92, 0x94, 0x94,
        0x96, 0x96, 0x98, 0x98, 0x9A, 0x9A, 0x9C, 0x9C, 0x9E, 0x9E, 0xA0, 0xA0, 0xA2, 0xA2, 0xA4,
        0xA4, 0xA6, 0xA6, 0xA8, 0xA8, 0xAA, 0xAA, 0xAC, 0xAC, 0xAE, 0xAE, 0xB0, 0xB0, 0xB2, 0xB2,
        0xB4, 0xB4, 0xB6, 0xB6, 0xB8, 0xB8, 0xBA, 0xBA, 0xBC, 0xBC, 0xBE, 0xBE, 0xC0, 0xC0, 0xC2,
        0xC2, 0xC4, 0xC4, 0xC6, 0xC6, 0xC8, 0xC8, 0xCA, 0xCA, 0xCC, 0xCC, 0xCE, 0xCE, 0xD0, 0xD0,
        0xD2, 0xD2, 0xD4, 0xD4, 0xD6, 0xD6, 0xD8, 0xD8, 0xDA, 0xDA, 0xDC, 0xDC, 0xDE, 0xDE, 0xE0,
        0xE0, 0xE2, 0xE2, 0xE4, 0xE4, 0xE6, 0xE6, 0xE8, 0xE8, 0xEA, 0xEA, 0xEC, 0xEC, 0xEE, 0xEE,
        0xF0, 0xF0, 0xF2, 0xF2, 0xF4, 0xF4, 0xF6, 0xF6, 0xF8, 0xF8, 0xFA, 0xFA, 0xFC, 0xFC, 0xFE,
        0xFE, 0x01, 0x01, 0x03, 0x03, 0x05, 0x05, 0x07, 0x07, 0x09, 0x09, 0xD1, 0xD3, 0xD3, 0xD5,
        0xD5, 0xD7, 0xD7, 0xD9, 0xD9, 0xDB, 0xDB, 0xDD, 0xDD, 0xDF, 0xDF, 0xE1, 0xE1, 0xE3, 0xE3,
        0xE5, 0xE5, 0xE7, 0xE7, 0xE9, 0xE9, 0xEB, 0xEB, 0xED, 0xED, 0xEF, 0xEF, 0xF1, 0xF1, 0xF3,
        0xF3, 0xF5, 0xF5, 0xF7, 0xF7, 0xF9, 0xF9, 0xFB, 0xFB, 0xFD, 0xFD, 0xFF, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::ThirtyCell;
    vdp.registers.interlacing_mode = InterlacingMode::Interlaced;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = true;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_ntsc_interlaced_double_even_frame() {
    const EXPECTED: &[u8; NTSC_SCANLINES_PER_FRAME as usize] = &[
        0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1A, 0x1C,
        0x1E, 0x20, 0x22, 0x24, 0x26, 0x28, 0x2A, 0x2C, 0x2E, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3A,
        0x3C, 0x3E, 0x40, 0x42, 0x44, 0x46, 0x48, 0x4A, 0x4C, 0x4E, 0x50, 0x52, 0x54, 0x56, 0x58,
        0x5A, 0x5C, 0x5E, 0x60, 0x62, 0x64, 0x66, 0x68, 0x6A, 0x6C, 0x6E, 0x70, 0x72, 0x74, 0x76,
        0x78, 0x7A, 0x7C, 0x7E, 0x80, 0x82, 0x84, 0x86, 0x88, 0x8A, 0x8C, 0x8E, 0x90, 0x92, 0x94,
        0x96, 0x98, 0x9A, 0x9C, 0x9E, 0xA0, 0xA2, 0xA4, 0xA6, 0xA8, 0xAA, 0xAC, 0xAE, 0xB0, 0xB2,
        0xB4, 0xB6, 0xB8, 0xBA, 0xBC, 0xBE, 0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE, 0xD0,
        0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE, 0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE,
        0xF0, 0xF2, 0xF4, 0xF6, 0xF8, 0xFA, 0xFC, 0xFE, 0x01, 0x03, 0x05, 0x07, 0x09, 0x0B, 0x0D,
        0x0F, 0x11, 0x13, 0x15, 0x17, 0x19, 0x1B, 0x1D, 0x1F, 0x21, 0x23, 0x25, 0x27, 0x29, 0x2B,
        0x2D, 0x2F, 0x31, 0x33, 0x35, 0x37, 0x39, 0x3B, 0x3D, 0x3F, 0x41, 0x43, 0x45, 0x47, 0x49,
        0x4B, 0x4D, 0x4F, 0x51, 0x53, 0x55, 0x57, 0x59, 0x5B, 0x5D, 0x5F, 0x61, 0x63, 0x65, 0x67,
        0x69, 0x6B, 0x6D, 0x6F, 0x71, 0x73, 0x75, 0x77, 0x79, 0x7B, 0x7D, 0x7F, 0x81, 0x83, 0x85,
        0x87, 0x89, 0x8B, 0x8D, 0x8F, 0x91, 0x93, 0x95, 0x97, 0x99, 0x9B, 0x9D, 0x9F, 0xA1, 0xA3,
        0xA5, 0xA7, 0xA9, 0xAB, 0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1,
        0xC3, 0xC5, 0xC7, 0xC9, 0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xCB, 0xCD, 0xCF, 0xD1, 0xD3,
        0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF, 0xE1, 0xE3, 0xE5, 0xE7, 0xE9, 0xEB, 0xED, 0xEF, 0xF1,
        0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Ntsc;
    vdp.registers.interlacing_mode = InterlacingMode::InterlacedDouble;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = false;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_ntsc_interlaced_double_odd_frame() {
    const EXPECTED: &[u8; (NTSC_SCANLINES_PER_FRAME + 1) as usize] = &[
        0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1A, 0x1C,
        0x1E, 0x20, 0x22, 0x24, 0x26, 0x28, 0x2A, 0x2C, 0x2E, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3A,
        0x3C, 0x3E, 0x40, 0x42, 0x44, 0x46, 0x48, 0x4A, 0x4C, 0x4E, 0x50, 0x52, 0x54, 0x56, 0x58,
        0x5A, 0x5C, 0x5E, 0x60, 0x62, 0x64, 0x66, 0x68, 0x6A, 0x6C, 0x6E, 0x70, 0x72, 0x74, 0x76,
        0x78, 0x7A, 0x7C, 0x7E, 0x80, 0x82, 0x84, 0x86, 0x88, 0x8A, 0x8C, 0x8E, 0x90, 0x92, 0x94,
        0x96, 0x98, 0x9A, 0x9C, 0x9E, 0xA0, 0xA2, 0xA4, 0xA6, 0xA8, 0xAA, 0xAC, 0xAE, 0xB0, 0xB2,
        0xB4, 0xB6, 0xB8, 0xBA, 0xBC, 0xBE, 0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE, 0xD0,
        0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE, 0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE,
        0xF0, 0xF2, 0xF4, 0xF6, 0xF8, 0xFA, 0xFC, 0xFE, 0x01, 0x03, 0x05, 0x07, 0x09, 0x0B, 0x0D,
        0x0F, 0x11, 0x13, 0x15, 0x17, 0x19, 0x1B, 0x1D, 0x1F, 0x21, 0x23, 0x25, 0x27, 0x29, 0x2B,
        0x2D, 0x2F, 0x31, 0x33, 0x35, 0x37, 0x39, 0x3B, 0x3D, 0x3F, 0x41, 0x43, 0x45, 0x47, 0x49,
        0x4B, 0x4D, 0x4F, 0x51, 0x53, 0x55, 0x57, 0x59, 0x5B, 0x5D, 0x5F, 0x61, 0x63, 0x65, 0x67,
        0x69, 0x6B, 0x6D, 0x6F, 0x71, 0x73, 0x75, 0x77, 0x79, 0x7B, 0x7D, 0x7F, 0x81, 0x83, 0x85,
        0x87, 0x89, 0x8B, 0x8D, 0x8F, 0x91, 0x93, 0x95, 0x97, 0x99, 0x9B, 0x9D, 0x9F, 0xA1, 0xA3,
        0xA5, 0xA7, 0xA9, 0xAB, 0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1,
        0xC3, 0xC5, 0xC7, 0xC9, 0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xC9, 0xCB, 0xCD, 0xCF, 0xD1,
        0xD3, 0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF, 0xE1, 0xE3, 0xE5, 0xE7, 0xE9, 0xEB, 0xED, 0xEF,
        0xF1, 0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Ntsc;
    vdp.registers.interlacing_mode = InterlacingMode::InterlacedDouble;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = true;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v28_interlaced_double_even_frame() {
    const EXPECTED: &[u8; (PAL_SCANLINES_PER_FRAME - 1) as usize] = &[
        0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1A, 0x1C,
        0x1E, 0x20, 0x22, 0x24, 0x26, 0x28, 0x2A, 0x2C, 0x2E, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3A,
        0x3C, 0x3E, 0x40, 0x42, 0x44, 0x46, 0x48, 0x4A, 0x4C, 0x4E, 0x50, 0x52, 0x54, 0x56, 0x58,
        0x5A, 0x5C, 0x5E, 0x60, 0x62, 0x64, 0x66, 0x68, 0x6A, 0x6C, 0x6E, 0x70, 0x72, 0x74, 0x76,
        0x78, 0x7A, 0x7C, 0x7E, 0x80, 0x82, 0x84, 0x86, 0x88, 0x8A, 0x8C, 0x8E, 0x90, 0x92, 0x94,
        0x96, 0x98, 0x9A, 0x9C, 0x9E, 0xA0, 0xA2, 0xA4, 0xA6, 0xA8, 0xAA, 0xAC, 0xAE, 0xB0, 0xB2,
        0xB4, 0xB6, 0xB8, 0xBA, 0xBC, 0xBE, 0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE, 0xD0,
        0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE, 0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE,
        0xF0, 0xF2, 0xF4, 0xF6, 0xF8, 0xFA, 0xFC, 0xFE, 0x01, 0x03, 0x05, 0x07, 0x09, 0x0B, 0x0D,
        0x0F, 0x11, 0x13, 0x15, 0x17, 0x19, 0x1B, 0x1D, 0x1F, 0x21, 0x23, 0x25, 0x27, 0x29, 0x2B,
        0x2D, 0x2F, 0x31, 0x33, 0x35, 0x37, 0x39, 0x3B, 0x3D, 0x3F, 0x41, 0x43, 0x45, 0x47, 0x49,
        0x4B, 0x4D, 0x4F, 0x51, 0x53, 0x55, 0x57, 0x59, 0x5B, 0x5D, 0x5F, 0x61, 0x63, 0x65, 0x67,
        0x69, 0x6B, 0x6D, 0x6F, 0x71, 0x73, 0x75, 0x77, 0x79, 0x7B, 0x7D, 0x7F, 0x81, 0x83, 0x85,
        0x87, 0x89, 0x8B, 0x8D, 0x8F, 0x91, 0x93, 0x95, 0x97, 0x99, 0x9B, 0x9D, 0x9F, 0xA1, 0xA3,
        0xA5, 0xA7, 0xA9, 0xAB, 0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1,
        0xC3, 0xC5, 0xC7, 0xC9, 0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF,
        0xE1, 0xE3, 0xE5, 0xE7, 0xE9, 0xEB, 0xED, 0xEF, 0xF1, 0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD,
        0xFF, 0x00, 0x02, 0x95, 0x97, 0x99, 0x9B, 0x9D, 0x9F, 0xA1, 0xA3, 0xA5, 0xA7, 0xA9, 0xAB,
        0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1, 0xC3, 0xC5, 0xC7, 0xC9,
        0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF, 0xE1, 0xE3, 0xE5, 0xE7,
        0xE9, 0xEB, 0xED, 0xEF, 0xF1, 0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::TwentyEightCell;
    vdp.registers.interlacing_mode = InterlacingMode::InterlacedDouble;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = false;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v28_interlaced_double_odd_frame() {
    const EXPECTED: &[u8; PAL_SCANLINES_PER_FRAME as usize] = &[
        0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1A, 0x1C,
        0x1E, 0x20, 0x22, 0x24, 0x26, 0x28, 0x2A, 0x2C, 0x2E, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3A,
        0x3C, 0x3E, 0x40, 0x42, 0x44, 0x46, 0x48, 0x4A, 0x4C, 0x4E, 0x50, 0x52, 0x54, 0x56, 0x58,
        0x5A, 0x5C, 0x5E, 0x60, 0x62, 0x64, 0x66, 0x68, 0x6A, 0x6C, 0x6E, 0x70, 0x72, 0x74, 0x76,
        0x78, 0x7A, 0x7C, 0x7E, 0x80, 0x82, 0x84, 0x86, 0x88, 0x8A, 0x8C, 0x8E, 0x90, 0x92, 0x94,
        0x96, 0x98, 0x9A, 0x9C, 0x9E, 0xA0, 0xA2, 0xA4, 0xA6, 0xA8, 0xAA, 0xAC, 0xAE, 0xB0, 0xB2,
        0xB4, 0xB6, 0xB8, 0xBA, 0xBC, 0xBE, 0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE, 0xD0,
        0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE, 0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE,
        0xF0, 0xF2, 0xF4, 0xF6, 0xF8, 0xFA, 0xFC, 0xFE, 0x01, 0x03, 0x05, 0x07, 0x09, 0x0B, 0x0D,
        0x0F, 0x11, 0x13, 0x15, 0x17, 0x19, 0x1B, 0x1D, 0x1F, 0x21, 0x23, 0x25, 0x27, 0x29, 0x2B,
        0x2D, 0x2F, 0x31, 0x33, 0x35, 0x37, 0x39, 0x3B, 0x3D, 0x3F, 0x41, 0x43, 0x45, 0x47, 0x49,
        0x4B, 0x4D, 0x4F, 0x51, 0x53, 0x55, 0x57, 0x59, 0x5B, 0x5D, 0x5F, 0x61, 0x63, 0x65, 0x67,
        0x69, 0x6B, 0x6D, 0x6F, 0x71, 0x73, 0x75, 0x77, 0x79, 0x7B, 0x7D, 0x7F, 0x81, 0x83, 0x85,
        0x87, 0x89, 0x8B, 0x8D, 0x8F, 0x91, 0x93, 0x95, 0x97, 0x99, 0x9B, 0x9D, 0x9F, 0xA1, 0xA3,
        0xA5, 0xA7, 0xA9, 0xAB, 0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1,
        0xC3, 0xC5, 0xC7, 0xC9, 0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF,
        0xE1, 0xE3, 0xE5, 0xE7, 0xE9, 0xEB, 0xED, 0xEF, 0xF1, 0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD,
        0xFF, 0x00, 0x02, 0x93, 0x95, 0x97, 0x99, 0x9B, 0x9D, 0x9F, 0xA1, 0xA3, 0xA5, 0xA7, 0xA9,
        0xAB, 0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1, 0xC3, 0xC5, 0xC7,
        0xC9, 0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF, 0xE1, 0xE3, 0xE5,
        0xE7, 0xE9, 0xEB, 0xED, 0xEF, 0xF1, 0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::TwentyEightCell;
    vdp.registers.interlacing_mode = InterlacingMode::InterlacedDouble;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = true;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v30_interlaced_double_even_frame() {
    const EXPECTED: &[u8; (PAL_SCANLINES_PER_FRAME - 1) as usize] = &[
        0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1A, 0x1C,
        0x1E, 0x20, 0x22, 0x24, 0x26, 0x28, 0x2A, 0x2C, 0x2E, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3A,
        0x3C, 0x3E, 0x40, 0x42, 0x44, 0x46, 0x48, 0x4A, 0x4C, 0x4E, 0x50, 0x52, 0x54, 0x56, 0x58,
        0x5A, 0x5C, 0x5E, 0x60, 0x62, 0x64, 0x66, 0x68, 0x6A, 0x6C, 0x6E, 0x70, 0x72, 0x74, 0x76,
        0x78, 0x7A, 0x7C, 0x7E, 0x80, 0x82, 0x84, 0x86, 0x88, 0x8A, 0x8C, 0x8E, 0x90, 0x92, 0x94,
        0x96, 0x98, 0x9A, 0x9C, 0x9E, 0xA0, 0xA2, 0xA4, 0xA6, 0xA8, 0xAA, 0xAC, 0xAE, 0xB0, 0xB2,
        0xB4, 0xB6, 0xB8, 0xBA, 0xBC, 0xBE, 0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE, 0xD0,
        0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE, 0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE,
        0xF0, 0xF2, 0xF4, 0xF6, 0xF8, 0xFA, 0xFC, 0xFE, 0x01, 0x03, 0x05, 0x07, 0x09, 0x0B, 0x0D,
        0x0F, 0x11, 0x13, 0x15, 0x17, 0x19, 0x1B, 0x1D, 0x1F, 0x21, 0x23, 0x25, 0x27, 0x29, 0x2B,
        0x2D, 0x2F, 0x31, 0x33, 0x35, 0x37, 0x39, 0x3B, 0x3D, 0x3F, 0x41, 0x43, 0x45, 0x47, 0x49,
        0x4B, 0x4D, 0x4F, 0x51, 0x53, 0x55, 0x57, 0x59, 0x5B, 0x5D, 0x5F, 0x61, 0x63, 0x65, 0x67,
        0x69, 0x6B, 0x6D, 0x6F, 0x71, 0x73, 0x75, 0x77, 0x79, 0x7B, 0x7D, 0x7F, 0x81, 0x83, 0x85,
        0x87, 0x89, 0x8B, 0x8D, 0x8F, 0x91, 0x93, 0x95, 0x97, 0x99, 0x9B, 0x9D, 0x9F, 0xA1, 0xA3,
        0xA5, 0xA7, 0xA9, 0xAB, 0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1,
        0xC3, 0xC5, 0xC7, 0xC9, 0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF,
        0xE1, 0xE3, 0xE5, 0xE7, 0xE9, 0xEB, 0xED, 0xEF, 0xF1, 0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD,
        0xFF, 0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0xA5, 0xA7, 0xA9, 0xAB,
        0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1, 0xC3, 0xC5, 0xC7, 0xC9,
        0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF, 0xE1, 0xE3, 0xE5, 0xE7,
        0xE9, 0xEB, 0xED, 0xEF, 0xF1, 0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::ThirtyCell;
    vdp.registers.interlacing_mode = InterlacingMode::InterlacedDouble;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = false;
    v_counter_test(EXPECTED, &mut vdp);
}

#[test]
fn v_counter_pal_v30_interlaced_double_odd_frame() {
    const EXPECTED: &[u8; PAL_SCANLINES_PER_FRAME as usize] = &[
        0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1A, 0x1C,
        0x1E, 0x20, 0x22, 0x24, 0x26, 0x28, 0x2A, 0x2C, 0x2E, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3A,
        0x3C, 0x3E, 0x40, 0x42, 0x44, 0x46, 0x48, 0x4A, 0x4C, 0x4E, 0x50, 0x52, 0x54, 0x56, 0x58,
        0x5A, 0x5C, 0x5E, 0x60, 0x62, 0x64, 0x66, 0x68, 0x6A, 0x6C, 0x6E, 0x70, 0x72, 0x74, 0x76,
        0x78, 0x7A, 0x7C, 0x7E, 0x80, 0x82, 0x84, 0x86, 0x88, 0x8A, 0x8C, 0x8E, 0x90, 0x92, 0x94,
        0x96, 0x98, 0x9A, 0x9C, 0x9E, 0xA0, 0xA2, 0xA4, 0xA6, 0xA8, 0xAA, 0xAC, 0xAE, 0xB0, 0xB2,
        0xB4, 0xB6, 0xB8, 0xBA, 0xBC, 0xBE, 0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE, 0xD0,
        0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE, 0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE,
        0xF0, 0xF2, 0xF4, 0xF6, 0xF8, 0xFA, 0xFC, 0xFE, 0x01, 0x03, 0x05, 0x07, 0x09, 0x0B, 0x0D,
        0x0F, 0x11, 0x13, 0x15, 0x17, 0x19, 0x1B, 0x1D, 0x1F, 0x21, 0x23, 0x25, 0x27, 0x29, 0x2B,
        0x2D, 0x2F, 0x31, 0x33, 0x35, 0x37, 0x39, 0x3B, 0x3D, 0x3F, 0x41, 0x43, 0x45, 0x47, 0x49,
        0x4B, 0x4D, 0x4F, 0x51, 0x53, 0x55, 0x57, 0x59, 0x5B, 0x5D, 0x5F, 0x61, 0x63, 0x65, 0x67,
        0x69, 0x6B, 0x6D, 0x6F, 0x71, 0x73, 0x75, 0x77, 0x79, 0x7B, 0x7D, 0x7F, 0x81, 0x83, 0x85,
        0x87, 0x89, 0x8B, 0x8D, 0x8F, 0x91, 0x93, 0x95, 0x97, 0x99, 0x9B, 0x9D, 0x9F, 0xA1, 0xA3,
        0xA5, 0xA7, 0xA9, 0xAB, 0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1,
        0xC3, 0xC5, 0xC7, 0xC9, 0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF,
        0xE1, 0xE3, 0xE5, 0xE7, 0xE9, 0xEB, 0xED, 0xEF, 0xF1, 0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD,
        0xFF, 0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0xA3, 0xA5, 0xA7, 0xA9,
        0xAB, 0xAD, 0xAF, 0xB1, 0xB3, 0xB5, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC1, 0xC3, 0xC5, 0xC7,
        0xC9, 0xCB, 0xCD, 0xCF, 0xD1, 0xD3, 0xD5, 0xD7, 0xD9, 0xDB, 0xDD, 0xDF, 0xE1, 0xE3, 0xE5,
        0xE7, 0xE9, 0xEB, 0xED, 0xEF, 0xF1, 0xF3, 0xF5, 0xF7, 0xF9, 0xFB, 0xFD, 0xFF,
    ];

    let mut vdp = new_vdp();
    vdp.timing_mode = TimingMode::Pal;
    vdp.registers.vertical_display_size = VerticalDisplaySize::ThirtyCell;
    vdp.registers.interlacing_mode = InterlacingMode::InterlacedDouble;
    vdp.state.interlaced_frame = true;
    vdp.state.interlaced_odd = true;
    v_counter_test(EXPECTED, &mut vdp);
}

fn h_counter_test(
    expected_pixels: &[u16; 3420],
    expected_h: &[u16; 3420],
    pixel_fn: fn(u64) -> u16,
    h_fn: fn(u16) -> u16,
) {
    for mclk in 0..3420 {
        let pixel = pixel_fn(mclk);
        let h = h_fn(pixel);

        assert_eq!(
            pixel, expected_pixels[mclk as usize],
            "Pixel mismatch for mclk={mclk}: expected {}, got {pixel}",
            expected_pixels[mclk as usize]
        );
        assert_eq!(
            h, expected_h[mclk as usize],
            "Internal H mismatch for mclk={mclk}: expected 0x{:03X}, got 0x{h:03X}",
            expected_h[mclk as usize]
        );
    }
}

#[test]
fn h_counter_h32() {
    const EXPECTED_PIXELS: &[u16; 3420] = &[
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
        6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
        9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11,
        11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13,
        13, 13, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 16,
        16, 16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 18, 18, 18, 18,
        18, 18, 18, 18, 18, 18, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 20, 20, 20, 20, 20, 20, 20,
        20, 20, 20, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22,
        23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 25, 25, 25,
        25, 25, 25, 25, 25, 25, 25, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 27, 27, 27, 27, 27, 27,
        27, 27, 27, 27, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 29, 29, 29, 29, 29, 29, 29, 29, 29,
        29, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 32, 32,
        32, 32, 32, 32, 32, 32, 32, 32, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 34, 34, 34, 34, 34,
        34, 34, 34, 34, 34, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 36, 36, 36, 36, 36, 36, 36, 36,
        36, 36, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 39,
        39, 39, 39, 39, 39, 39, 39, 39, 39, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 41, 41, 41, 41,
        41, 41, 41, 41, 41, 41, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 43, 43, 43, 43, 43, 43, 43,
        43, 43, 43, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45,
        46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 48, 48, 48,
        48, 48, 48, 48, 48, 48, 48, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 50, 50, 50, 50, 50, 50,
        50, 50, 50, 50, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 52, 52, 52, 52, 52, 52, 52, 52, 52,
        52, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 55, 55,
        55, 55, 55, 55, 55, 55, 55, 55, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 57, 57, 57, 57, 57,
        57, 57, 57, 57, 57, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 59, 59, 59, 59, 59, 59, 59, 59,
        59, 59, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 62,
        62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64,
        64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66,
        66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68,
        69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 71, 71, 71,
        71, 71, 71, 71, 71, 71, 71, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 73, 73, 73, 73, 73, 73,
        73, 73, 73, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 75, 75, 75, 75, 75, 75, 75, 75, 75,
        75, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 78, 78,
        78, 78, 78, 78, 78, 78, 78, 78, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 80, 80, 80, 80,
        80, 80, 80, 80, 80, 81, 81, 81, 81, 81, 81, 81, 81, 81, 81, 82, 82, 82, 82, 82, 82, 82, 82,
        82, 82, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 85,
        85, 85, 85, 85, 85, 85, 85, 85, 85, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 87, 87, 87, 87,
        87, 87, 87, 87, 87, 87, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 89, 89, 89, 89, 89, 89,
        89, 89, 89, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91,
        92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 94, 94, 94,
        94, 94, 94, 94, 94, 94, 94, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 96, 96, 96, 96, 96, 96,
        96, 96, 96, 96, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 98, 98, 98, 98, 98, 98, 98, 98, 98,
        98, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 100, 100, 100, 100, 100, 100, 100, 100, 100,
        100, 101, 101, 101, 101, 101, 101, 101, 101, 101, 101, 102, 102, 102, 102, 102, 102, 102,
        102, 102, 102, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 104, 104, 104, 104,
        104, 104, 104, 104, 104, 105, 105, 105, 105, 105, 105, 105, 105, 105, 105, 106, 106, 106,
        106, 106, 106, 106, 106, 106, 106, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 108,
        108, 108, 108, 108, 108, 108, 108, 108, 108, 109, 109, 109, 109, 109, 109, 109, 109, 109,
        109, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 111, 111, 111, 111, 111, 111, 111,
        111, 111, 111, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 113, 113, 113, 113, 113,
        113, 113, 113, 113, 113, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 115, 115, 115,
        115, 115, 115, 115, 115, 115, 115, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 117,
        117, 117, 117, 117, 117, 117, 117, 117, 117, 118, 118, 118, 118, 118, 118, 118, 118, 118,
        118, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 120, 120, 120, 120, 120, 120, 120,
        120, 120, 120, 121, 121, 121, 121, 121, 121, 121, 121, 121, 121, 122, 122, 122, 122, 122,
        122, 122, 122, 122, 122, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 124, 124, 124,
        124, 124, 124, 124, 124, 124, 124, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 126,
        126, 126, 126, 126, 126, 126, 126, 126, 126, 127, 127, 127, 127, 127, 127, 127, 127, 127,
        127, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 129, 129, 129, 129, 129, 129, 129,
        129, 129, 129, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 131, 131, 131, 131, 131,
        131, 131, 131, 131, 131, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 133, 133, 133,
        133, 133, 133, 133, 133, 133, 133, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 135,
        135, 135, 135, 135, 135, 135, 135, 135, 135, 136, 136, 136, 136, 136, 136, 136, 136, 136,
        136, 137, 137, 137, 137, 137, 137, 137, 137, 137, 137, 138, 138, 138, 138, 138, 138, 138,
        138, 138, 138, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 140, 140, 140, 140, 140,
        140, 140, 140, 140, 140, 141, 141, 141, 141, 141, 141, 141, 141, 141, 141, 142, 142, 142,
        142, 142, 142, 142, 142, 142, 142, 143, 143, 143, 143, 143, 143, 143, 143, 143, 143, 144,
        144, 144, 144, 144, 144, 144, 144, 144, 144, 145, 145, 145, 145, 145, 145, 145, 145, 145,
        145, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 147, 147, 147, 147, 147, 147, 147,
        147, 147, 147, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 149, 149, 149, 149, 149,
        149, 149, 149, 149, 149, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 151, 151, 151,
        151, 151, 151, 151, 151, 151, 151, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 153,
        153, 153, 153, 153, 153, 153, 153, 153, 153, 154, 154, 154, 154, 154, 154, 154, 154, 154,
        154, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 156, 156, 156, 156, 156, 156, 156,
        156, 156, 156, 157, 157, 157, 157, 157, 157, 157, 157, 157, 157, 158, 158, 158, 158, 158,
        158, 158, 158, 158, 158, 159, 159, 159, 159, 159, 159, 159, 159, 159, 159, 160, 160, 160,
        160, 160, 160, 160, 160, 160, 160, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 162,
        162, 162, 162, 162, 162, 162, 162, 162, 162, 163, 163, 163, 163, 163, 163, 163, 163, 163,
        163, 164, 164, 164, 164, 164, 164, 164, 164, 164, 164, 165, 165, 165, 165, 165, 165, 165,
        165, 165, 165, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 167, 167, 167, 167, 167,
        167, 167, 167, 167, 167, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 169, 169, 169,
        169, 169, 169, 169, 169, 169, 169, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 171,
        171, 171, 171, 171, 171, 171, 171, 171, 171, 172, 172, 172, 172, 172, 172, 172, 172, 172,
        172, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 174, 174, 174, 174, 174, 174, 174,
        174, 174, 174, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 176, 176, 176, 176, 176,
        176, 176, 176, 176, 176, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 178, 178, 178,
        178, 178, 178, 178, 178, 178, 178, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 180,
        180, 180, 180, 180, 180, 180, 180, 180, 180, 181, 181, 181, 181, 181, 181, 181, 181, 181,
        181, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 183, 183, 183, 183, 183, 183, 183,
        183, 183, 183, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 185, 185, 185, 185, 185,
        185, 185, 185, 185, 185, 186, 186, 186, 186, 186, 186, 186, 186, 186, 186, 187, 187, 187,
        187, 187, 187, 187, 187, 187, 187, 188, 188, 188, 188, 188, 188, 188, 188, 188, 188, 189,
        189, 189, 189, 189, 189, 189, 189, 189, 189, 190, 190, 190, 190, 190, 190, 190, 190, 190,
        190, 191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 192, 192, 192, 192, 192, 192, 192,
        192, 192, 192, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 194, 194, 194, 194, 194,
        194, 194, 194, 194, 194, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 196, 196, 196,
        196, 196, 196, 196, 196, 196, 196, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 198,
        198, 198, 198, 198, 198, 198, 198, 198, 198, 199, 199, 199, 199, 199, 199, 199, 199, 199,
        199, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 201, 201, 201, 201, 201, 201, 201,
        201, 201, 201, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 203, 203, 203, 203, 203,
        203, 203, 203, 203, 203, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 205, 205, 205,
        205, 205, 205, 205, 205, 205, 205, 206, 206, 206, 206, 206, 206, 206, 206, 206, 206, 207,
        207, 207, 207, 207, 207, 207, 207, 207, 207, 208, 208, 208, 208, 208, 208, 208, 208, 208,
        208, 209, 209, 209, 209, 209, 209, 209, 209, 209, 209, 210, 210, 210, 210, 210, 210, 210,
        210, 210, 210, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 212, 212, 212, 212, 212,
        212, 212, 212, 212, 212, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 214, 214, 214,
        214, 214, 214, 214, 214, 214, 214, 215, 215, 215, 215, 215, 215, 215, 215, 215, 215, 216,
        216, 216, 216, 216, 216, 216, 216, 216, 216, 217, 217, 217, 217, 217, 217, 217, 217, 217,
        217, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 219, 219, 219, 219, 219, 219, 219,
        219, 219, 219, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 221, 221, 221, 221, 221,
        221, 221, 221, 221, 221, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 223, 223, 223,
        223, 223, 223, 223, 223, 223, 223, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 225,
        225, 225, 225, 225, 225, 225, 225, 225, 225, 226, 226, 226, 226, 226, 226, 226, 226, 226,
        226, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 228, 228, 228, 228, 228, 228, 228,
        228, 228, 228, 229, 229, 229, 229, 229, 229, 229, 229, 229, 229, 230, 230, 230, 230, 230,
        230, 230, 230, 230, 230, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 232, 232, 232,
        232, 232, 232, 232, 232, 232, 232, 233, 233, 233, 233, 233, 233, 233, 233, 233, 233, 234,
        234, 234, 234, 234, 234, 234, 234, 234, 234, 235, 235, 235, 235, 235, 235, 235, 235, 235,
        235, 236, 236, 236, 236, 236, 236, 236, 236, 236, 236, 237, 237, 237, 237, 237, 237, 237,
        237, 237, 237, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 239, 239, 239, 239, 239,
        239, 239, 239, 239, 239, 240, 240, 240, 240, 240, 240, 240, 240, 240, 240, 241, 241, 241,
        241, 241, 241, 241, 241, 241, 241, 242, 242, 242, 242, 242, 242, 242, 242, 242, 242, 243,
        243, 243, 243, 243, 243, 243, 243, 243, 243, 244, 244, 244, 244, 244, 244, 244, 244, 244,
        244, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 246, 246, 246, 246, 246, 246, 246,
        246, 246, 246, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 248, 248, 248, 248, 248,
        248, 248, 248, 248, 248, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 250, 250, 250,
        250, 250, 250, 250, 250, 250, 250, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 252,
        252, 252, 252, 252, 252, 252, 252, 252, 252, 253, 253, 253, 253, 253, 253, 253, 253, 253,
        253, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 255, 255, 255, 255, 255, 255, 255,
        255, 255, 255, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 257, 257, 257, 257, 257,
        257, 257, 257, 257, 257, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 259, 259, 259,
        259, 259, 259, 259, 259, 259, 259, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 261,
        261, 261, 261, 261, 261, 261, 261, 261, 261, 262, 262, 262, 262, 262, 262, 262, 262, 262,
        262, 263, 263, 263, 263, 263, 263, 263, 263, 263, 263, 264, 264, 264, 264, 264, 264, 264,
        264, 264, 264, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 266, 266, 266, 266, 266,
        266, 266, 266, 266, 266, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 268, 268, 268,
        268, 268, 268, 268, 268, 268, 268, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 270,
        270, 270, 270, 270, 270, 270, 270, 270, 270, 271, 271, 271, 271, 271, 271, 271, 271, 271,
        271, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 273, 273, 273, 273, 273, 273, 273,
        273, 273, 273, 274, 274, 274, 274, 274, 274, 274, 274, 274, 274, 275, 275, 275, 275, 275,
        275, 275, 275, 275, 275, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 277, 277, 277,
        277, 277, 277, 277, 277, 277, 277, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 279,
        279, 279, 279, 279, 279, 279, 279, 279, 279, 280, 280, 280, 280, 280, 280, 280, 280, 280,
        280, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 282, 282, 282, 282, 282, 282, 282,
        282, 282, 282, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 284, 284, 284, 284, 284,
        284, 284, 284, 284, 284, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 286, 286, 286,
        286, 286, 286, 286, 286, 286, 286, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 288,
        288, 288, 288, 288, 288, 288, 288, 288, 288, 289, 289, 289, 289, 289, 289, 289, 289, 289,
        289, 290, 290, 290, 290, 290, 290, 290, 290, 290, 290, 291, 291, 291, 291, 291, 291, 291,
        291, 291, 291, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 293, 293, 293, 293, 293,
        293, 293, 293, 293, 293, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 295, 295, 295,
        295, 295, 295, 295, 295, 295, 295, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 297,
        297, 297, 297, 297, 297, 297, 297, 297, 297, 298, 298, 298, 298, 298, 298, 298, 298, 298,
        298, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 300, 300, 300, 300, 300, 300, 300,
        300, 300, 300, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 302, 302, 302, 302, 302,
        302, 302, 302, 302, 302, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 304, 304, 304,
        304, 304, 304, 304, 304, 304, 304, 305, 305, 305, 305, 305, 305, 305, 305, 305, 305, 306,
        306, 306, 306, 306, 306, 306, 306, 306, 306, 307, 307, 307, 307, 307, 307, 307, 307, 307,
        307, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 309, 309, 309, 309, 309, 309, 309,
        309, 309, 309, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 311, 311, 311, 311, 311,
        311, 311, 311, 311, 311, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 313, 313, 313,
        313, 313, 313, 313, 313, 313, 313, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 315,
        315, 315, 315, 315, 315, 315, 315, 315, 315, 316, 316, 316, 316, 316, 316, 316, 316, 316,
        316, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 318, 318, 318, 318, 318, 318, 318,
        318, 318, 318, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 320, 320, 320, 320, 320,
        320, 320, 320, 320, 320, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 322, 322, 322,
        322, 322, 322, 322, 322, 322, 322, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 324,
        324, 324, 324, 324, 324, 324, 324, 324, 324, 325, 325, 325, 325, 325, 325, 325, 325, 325,
        325, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 327, 327, 327, 327, 327, 327, 327,
        327, 327, 327, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 329, 329, 329, 329, 329,
        329, 329, 329, 329, 329, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 331, 331, 331,
        331, 331, 331, 331, 331, 331, 331, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 333,
        333, 333, 333, 333, 333, 333, 333, 333, 333, 334, 334, 334, 334, 334, 334, 334, 334, 334,
        334, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 336, 336, 336, 336, 336, 336, 336,
        336, 336, 336, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 338, 338, 338, 338, 338,
        338, 338, 338, 338, 338, 339, 339, 339, 339, 339, 339, 339, 339, 339, 339, 340, 340, 340,
        340, 340, 340, 340, 340, 340, 340, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341,
    ];
    const EXPECTED_H: &[u16; 3420] = &[
        0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x001, 0x001, 0x001,
        0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
        0x002, 0x002, 0x002, 0x002, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
        0x003, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x005, 0x005,
        0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x006, 0x006, 0x006, 0x006, 0x006,
        0x006, 0x006, 0x006, 0x006, 0x006, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
        0x007, 0x007, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x009,
        0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x00A, 0x00A, 0x00A, 0x00A,
        0x00A, 0x00A, 0x00A, 0x00A, 0x00A, 0x00A, 0x00B, 0x00B, 0x00B, 0x00B, 0x00B, 0x00B, 0x00B,
        0x00B, 0x00B, 0x00B, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C,
        0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00E, 0x00E, 0x00E,
        0x00E, 0x00E, 0x00E, 0x00E, 0x00E, 0x00E, 0x00E, 0x00F, 0x00F, 0x00F, 0x00F, 0x00F, 0x00F,
        0x00F, 0x00F, 0x00F, 0x00F, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
        0x010, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x012, 0x012,
        0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x013, 0x013, 0x013, 0x013, 0x013,
        0x013, 0x013, 0x013, 0x013, 0x013, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014,
        0x014, 0x014, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x016,
        0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x017, 0x017, 0x017, 0x017,
        0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018,
        0x018, 0x018, 0x018, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019,
        0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01B, 0x01B, 0x01B,
        0x01B, 0x01B, 0x01B, 0x01B, 0x01B, 0x01B, 0x01B, 0x01C, 0x01C, 0x01C, 0x01C, 0x01C, 0x01C,
        0x01C, 0x01C, 0x01C, 0x01C, 0x01D, 0x01D, 0x01D, 0x01D, 0x01D, 0x01D, 0x01D, 0x01D, 0x01D,
        0x01D, 0x01E, 0x01E, 0x01E, 0x01E, 0x01E, 0x01E, 0x01E, 0x01E, 0x01E, 0x01E, 0x01F, 0x01F,
        0x01F, 0x01F, 0x01F, 0x01F, 0x01F, 0x01F, 0x01F, 0x01F, 0x020, 0x020, 0x020, 0x020, 0x020,
        0x020, 0x020, 0x020, 0x020, 0x020, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021,
        0x021, 0x021, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x023,
        0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x024, 0x024, 0x024, 0x024,
        0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025,
        0x025, 0x025, 0x025, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026,
        0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x028, 0x028, 0x028,
        0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029,
        0x029, 0x029, 0x029, 0x029, 0x02A, 0x02A, 0x02A, 0x02A, 0x02A, 0x02A, 0x02A, 0x02A, 0x02A,
        0x02A, 0x02B, 0x02B, 0x02B, 0x02B, 0x02B, 0x02B, 0x02B, 0x02B, 0x02B, 0x02B, 0x02C, 0x02C,
        0x02C, 0x02C, 0x02C, 0x02C, 0x02C, 0x02C, 0x02C, 0x02C, 0x02D, 0x02D, 0x02D, 0x02D, 0x02D,
        0x02D, 0x02D, 0x02D, 0x02D, 0x02D, 0x02E, 0x02E, 0x02E, 0x02E, 0x02E, 0x02E, 0x02E, 0x02E,
        0x02E, 0x02E, 0x02F, 0x02F, 0x02F, 0x02F, 0x02F, 0x02F, 0x02F, 0x02F, 0x02F, 0x02F, 0x030,
        0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x031, 0x031, 0x031, 0x031,
        0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x032,
        0x032, 0x032, 0x032, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033,
        0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x035, 0x035, 0x035,
        0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036,
        0x036, 0x036, 0x036, 0x036, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037,
        0x037, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038, 0x039, 0x039,
        0x039, 0x039, 0x039, 0x039, 0x039, 0x039, 0x039, 0x039, 0x03A, 0x03A, 0x03A, 0x03A, 0x03A,
        0x03A, 0x03A, 0x03A, 0x03A, 0x03A, 0x03B, 0x03B, 0x03B, 0x03B, 0x03B, 0x03B, 0x03B, 0x03B,
        0x03B, 0x03B, 0x03C, 0x03C, 0x03C, 0x03C, 0x03C, 0x03C, 0x03C, 0x03C, 0x03C, 0x03C, 0x03D,
        0x03D, 0x03D, 0x03D, 0x03D, 0x03D, 0x03D, 0x03D, 0x03D, 0x03D, 0x03E, 0x03E, 0x03E, 0x03E,
        0x03E, 0x03E, 0x03E, 0x03E, 0x03E, 0x03E, 0x03F, 0x03F, 0x03F, 0x03F, 0x03F, 0x03F, 0x03F,
        0x03F, 0x03F, 0x03F, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040,
        0x041, 0x041, 0x041, 0x041, 0x041, 0x041, 0x041, 0x041, 0x041, 0x041, 0x042, 0x042, 0x042,
        0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043,
        0x043, 0x043, 0x043, 0x043, 0x044, 0x044, 0x044, 0x044, 0x044, 0x044, 0x044, 0x044, 0x044,
        0x044, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x046, 0x046,
        0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x047, 0x047, 0x047, 0x047, 0x047,
        0x047, 0x047, 0x047, 0x047, 0x047, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048,
        0x048, 0x048, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x04A,
        0x04A, 0x04A, 0x04A, 0x04A, 0x04A, 0x04A, 0x04A, 0x04A, 0x04A, 0x04B, 0x04B, 0x04B, 0x04B,
        0x04B, 0x04B, 0x04B, 0x04B, 0x04B, 0x04B, 0x04C, 0x04C, 0x04C, 0x04C, 0x04C, 0x04C, 0x04C,
        0x04C, 0x04C, 0x04C, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D,
        0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04F, 0x04F, 0x04F,
        0x04F, 0x04F, 0x04F, 0x04F, 0x04F, 0x04F, 0x04F, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050,
        0x050, 0x050, 0x050, 0x050, 0x051, 0x051, 0x051, 0x051, 0x051, 0x051, 0x051, 0x051, 0x051,
        0x051, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052, 0x053, 0x053,
        0x053, 0x053, 0x053, 0x053, 0x053, 0x053, 0x053, 0x053, 0x054, 0x054, 0x054, 0x054, 0x054,
        0x054, 0x054, 0x054, 0x054, 0x054, 0x055, 0x055, 0x055, 0x055, 0x055, 0x055, 0x055, 0x055,
        0x055, 0x055, 0x056, 0x056, 0x056, 0x056, 0x056, 0x056, 0x056, 0x056, 0x056, 0x056, 0x057,
        0x057, 0x057, 0x057, 0x057, 0x057, 0x057, 0x057, 0x057, 0x057, 0x058, 0x058, 0x058, 0x058,
        0x058, 0x058, 0x058, 0x058, 0x058, 0x058, 0x059, 0x059, 0x059, 0x059, 0x059, 0x059, 0x059,
        0x059, 0x059, 0x059, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A,
        0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05C, 0x05C, 0x05C,
        0x05C, 0x05C, 0x05C, 0x05C, 0x05C, 0x05C, 0x05C, 0x05D, 0x05D, 0x05D, 0x05D, 0x05D, 0x05D,
        0x05D, 0x05D, 0x05D, 0x05D, 0x05E, 0x05E, 0x05E, 0x05E, 0x05E, 0x05E, 0x05E, 0x05E, 0x05E,
        0x05E, 0x05F, 0x05F, 0x05F, 0x05F, 0x05F, 0x05F, 0x05F, 0x05F, 0x05F, 0x05F, 0x060, 0x060,
        0x060, 0x060, 0x060, 0x060, 0x060, 0x060, 0x060, 0x060, 0x061, 0x061, 0x061, 0x061, 0x061,
        0x061, 0x061, 0x061, 0x061, 0x061, 0x062, 0x062, 0x062, 0x062, 0x062, 0x062, 0x062, 0x062,
        0x062, 0x062, 0x063, 0x063, 0x063, 0x063, 0x063, 0x063, 0x063, 0x063, 0x063, 0x063, 0x064,
        0x064, 0x064, 0x064, 0x064, 0x064, 0x064, 0x064, 0x064, 0x064, 0x065, 0x065, 0x065, 0x065,
        0x065, 0x065, 0x065, 0x065, 0x065, 0x065, 0x066, 0x066, 0x066, 0x066, 0x066, 0x066, 0x066,
        0x066, 0x066, 0x066, 0x067, 0x067, 0x067, 0x067, 0x067, 0x067, 0x067, 0x067, 0x067, 0x067,
        0x068, 0x068, 0x068, 0x068, 0x068, 0x068, 0x068, 0x068, 0x068, 0x068, 0x069, 0x069, 0x069,
        0x069, 0x069, 0x069, 0x069, 0x069, 0x069, 0x069, 0x06A, 0x06A, 0x06A, 0x06A, 0x06A, 0x06A,
        0x06A, 0x06A, 0x06A, 0x06A, 0x06B, 0x06B, 0x06B, 0x06B, 0x06B, 0x06B, 0x06B, 0x06B, 0x06B,
        0x06B, 0x06C, 0x06C, 0x06C, 0x06C, 0x06C, 0x06C, 0x06C, 0x06C, 0x06C, 0x06C, 0x06D, 0x06D,
        0x06D, 0x06D, 0x06D, 0x06D, 0x06D, 0x06D, 0x06D, 0x06D, 0x06E, 0x06E, 0x06E, 0x06E, 0x06E,
        0x06E, 0x06E, 0x06E, 0x06E, 0x06E, 0x06F, 0x06F, 0x06F, 0x06F, 0x06F, 0x06F, 0x06F, 0x06F,
        0x06F, 0x06F, 0x070, 0x070, 0x070, 0x070, 0x070, 0x070, 0x070, 0x070, 0x070, 0x070, 0x071,
        0x071, 0x071, 0x071, 0x071, 0x071, 0x071, 0x071, 0x071, 0x071, 0x072, 0x072, 0x072, 0x072,
        0x072, 0x072, 0x072, 0x072, 0x072, 0x072, 0x073, 0x073, 0x073, 0x073, 0x073, 0x073, 0x073,
        0x073, 0x073, 0x073, 0x074, 0x074, 0x074, 0x074, 0x074, 0x074, 0x074, 0x074, 0x074, 0x074,
        0x075, 0x075, 0x075, 0x075, 0x075, 0x075, 0x075, 0x075, 0x075, 0x075, 0x076, 0x076, 0x076,
        0x076, 0x076, 0x076, 0x076, 0x076, 0x076, 0x076, 0x077, 0x077, 0x077, 0x077, 0x077, 0x077,
        0x077, 0x077, 0x077, 0x077, 0x078, 0x078, 0x078, 0x078, 0x078, 0x078, 0x078, 0x078, 0x078,
        0x078, 0x079, 0x079, 0x079, 0x079, 0x079, 0x079, 0x079, 0x079, 0x079, 0x079, 0x07A, 0x07A,
        0x07A, 0x07A, 0x07A, 0x07A, 0x07A, 0x07A, 0x07A, 0x07A, 0x07B, 0x07B, 0x07B, 0x07B, 0x07B,
        0x07B, 0x07B, 0x07B, 0x07B, 0x07B, 0x07C, 0x07C, 0x07C, 0x07C, 0x07C, 0x07C, 0x07C, 0x07C,
        0x07C, 0x07C, 0x07D, 0x07D, 0x07D, 0x07D, 0x07D, 0x07D, 0x07D, 0x07D, 0x07D, 0x07D, 0x07E,
        0x07E, 0x07E, 0x07E, 0x07E, 0x07E, 0x07E, 0x07E, 0x07E, 0x07E, 0x07F, 0x07F, 0x07F, 0x07F,
        0x07F, 0x07F, 0x07F, 0x07F, 0x07F, 0x07F, 0x080, 0x080, 0x080, 0x080, 0x080, 0x080, 0x080,
        0x080, 0x080, 0x080, 0x081, 0x081, 0x081, 0x081, 0x081, 0x081, 0x081, 0x081, 0x081, 0x081,
        0x082, 0x082, 0x082, 0x082, 0x082, 0x082, 0x082, 0x082, 0x082, 0x082, 0x083, 0x083, 0x083,
        0x083, 0x083, 0x083, 0x083, 0x083, 0x083, 0x083, 0x084, 0x084, 0x084, 0x084, 0x084, 0x084,
        0x084, 0x084, 0x084, 0x084, 0x085, 0x085, 0x085, 0x085, 0x085, 0x085, 0x085, 0x085, 0x085,
        0x085, 0x086, 0x086, 0x086, 0x086, 0x086, 0x086, 0x086, 0x086, 0x086, 0x086, 0x087, 0x087,
        0x087, 0x087, 0x087, 0x087, 0x087, 0x087, 0x087, 0x087, 0x088, 0x088, 0x088, 0x088, 0x088,
        0x088, 0x088, 0x088, 0x088, 0x088, 0x089, 0x089, 0x089, 0x089, 0x089, 0x089, 0x089, 0x089,
        0x089, 0x089, 0x08A, 0x08A, 0x08A, 0x08A, 0x08A, 0x08A, 0x08A, 0x08A, 0x08A, 0x08A, 0x08B,
        0x08B, 0x08B, 0x08B, 0x08B, 0x08B, 0x08B, 0x08B, 0x08B, 0x08B, 0x08C, 0x08C, 0x08C, 0x08C,
        0x08C, 0x08C, 0x08C, 0x08C, 0x08C, 0x08C, 0x08D, 0x08D, 0x08D, 0x08D, 0x08D, 0x08D, 0x08D,
        0x08D, 0x08D, 0x08D, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E,
        0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x090, 0x090, 0x090,
        0x090, 0x090, 0x090, 0x090, 0x090, 0x090, 0x090, 0x091, 0x091, 0x091, 0x091, 0x091, 0x091,
        0x091, 0x091, 0x091, 0x091, 0x092, 0x092, 0x092, 0x092, 0x092, 0x092, 0x092, 0x092, 0x092,
        0x092, 0x093, 0x093, 0x093, 0x093, 0x093, 0x093, 0x093, 0x093, 0x093, 0x093, 0x094, 0x094,
        0x094, 0x094, 0x094, 0x094, 0x094, 0x094, 0x094, 0x094, 0x095, 0x095, 0x095, 0x095, 0x095,
        0x095, 0x095, 0x095, 0x095, 0x095, 0x096, 0x096, 0x096, 0x096, 0x096, 0x096, 0x096, 0x096,
        0x096, 0x096, 0x097, 0x097, 0x097, 0x097, 0x097, 0x097, 0x097, 0x097, 0x097, 0x097, 0x098,
        0x098, 0x098, 0x098, 0x098, 0x098, 0x098, 0x098, 0x098, 0x098, 0x099, 0x099, 0x099, 0x099,
        0x099, 0x099, 0x099, 0x099, 0x099, 0x099, 0x09A, 0x09A, 0x09A, 0x09A, 0x09A, 0x09A, 0x09A,
        0x09A, 0x09A, 0x09A, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B,
        0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09D, 0x09D, 0x09D,
        0x09D, 0x09D, 0x09D, 0x09D, 0x09D, 0x09D, 0x09D, 0x09E, 0x09E, 0x09E, 0x09E, 0x09E, 0x09E,
        0x09E, 0x09E, 0x09E, 0x09E, 0x09F, 0x09F, 0x09F, 0x09F, 0x09F, 0x09F, 0x09F, 0x09F, 0x09F,
        0x09F, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A1, 0x0A1,
        0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A2, 0x0A2, 0x0A2, 0x0A2, 0x0A2,
        0x0A2, 0x0A2, 0x0A2, 0x0A2, 0x0A2, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3,
        0x0A3, 0x0A3, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A5,
        0x0A5, 0x0A5, 0x0A5, 0x0A5, 0x0A5, 0x0A5, 0x0A5, 0x0A5, 0x0A5, 0x0A6, 0x0A6, 0x0A6, 0x0A6,
        0x0A6, 0x0A6, 0x0A6, 0x0A6, 0x0A6, 0x0A6, 0x0A7, 0x0A7, 0x0A7, 0x0A7, 0x0A7, 0x0A7, 0x0A7,
        0x0A7, 0x0A7, 0x0A7, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8,
        0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0AA, 0x0AA, 0x0AA,
        0x0AA, 0x0AA, 0x0AA, 0x0AA, 0x0AA, 0x0AA, 0x0AA, 0x0AB, 0x0AB, 0x0AB, 0x0AB, 0x0AB, 0x0AB,
        0x0AB, 0x0AB, 0x0AB, 0x0AB, 0x0AC, 0x0AC, 0x0AC, 0x0AC, 0x0AC, 0x0AC, 0x0AC, 0x0AC, 0x0AC,
        0x0AC, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AE, 0x0AE,
        0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AF, 0x0AF, 0x0AF, 0x0AF, 0x0AF,
        0x0AF, 0x0AF, 0x0AF, 0x0AF, 0x0AF, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0,
        0x0B0, 0x0B0, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B2,
        0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B3, 0x0B3, 0x0B3, 0x0B3,
        0x0B3, 0x0B3, 0x0B3, 0x0B3, 0x0B3, 0x0B3, 0x0B4, 0x0B4, 0x0B4, 0x0B4, 0x0B4, 0x0B4, 0x0B4,
        0x0B4, 0x0B4, 0x0B4, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5,
        0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B7, 0x0B7, 0x0B7,
        0x0B7, 0x0B7, 0x0B7, 0x0B7, 0x0B7, 0x0B7, 0x0B7, 0x0B8, 0x0B8, 0x0B8, 0x0B8, 0x0B8, 0x0B8,
        0x0B8, 0x0B8, 0x0B8, 0x0B8, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0B9,
        0x0B9, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BB, 0x0BB,
        0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BC, 0x0BC, 0x0BC, 0x0BC, 0x0BC,
        0x0BC, 0x0BC, 0x0BC, 0x0BC, 0x0BC, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD,
        0x0BD, 0x0BD, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BF,
        0x0BF, 0x0BF, 0x0BF, 0x0BF, 0x0BF, 0x0BF, 0x0BF, 0x0BF, 0x0BF, 0x0C0, 0x0C0, 0x0C0, 0x0C0,
        0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C1, 0x0C1, 0x0C1, 0x0C1, 0x0C1, 0x0C1, 0x0C1,
        0x0C1, 0x0C1, 0x0C1, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2,
        0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C4, 0x0C4, 0x0C4,
        0x0C4, 0x0C4, 0x0C4, 0x0C4, 0x0C4, 0x0C4, 0x0C4, 0x0C5, 0x0C5, 0x0C5, 0x0C5, 0x0C5, 0x0C5,
        0x0C5, 0x0C5, 0x0C5, 0x0C5, 0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C6,
        0x0C6, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C8, 0x0C8,
        0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C9, 0x0C9, 0x0C9, 0x0C9, 0x0C9,
        0x0C9, 0x0C9, 0x0C9, 0x0C9, 0x0C9, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CA,
        0x0CA, 0x0CA, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CC,
        0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CD, 0x0CD, 0x0CD, 0x0CD,
        0x0CD, 0x0CD, 0x0CD, 0x0CD, 0x0CD, 0x0CD, 0x0CE, 0x0CE, 0x0CE, 0x0CE, 0x0CE, 0x0CE, 0x0CE,
        0x0CE, 0x0CE, 0x0CE, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF,
        0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D1, 0x0D1, 0x0D1,
        0x0D1, 0x0D1, 0x0D1, 0x0D1, 0x0D1, 0x0D1, 0x0D1, 0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D2,
        0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D3, 0x0D3, 0x0D3, 0x0D3, 0x0D3, 0x0D3, 0x0D3, 0x0D3, 0x0D3,
        0x0D3, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D5, 0x0D5,
        0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D6, 0x0D6, 0x0D6, 0x0D6, 0x0D6,
        0x0D6, 0x0D6, 0x0D6, 0x0D6, 0x0D6, 0x0D7, 0x0D7, 0x0D7, 0x0D7, 0x0D7, 0x0D7, 0x0D7, 0x0D7,
        0x0D7, 0x0D7, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D9,
        0x0D9, 0x0D9, 0x0D9, 0x0D9, 0x0D9, 0x0D9, 0x0D9, 0x0D9, 0x0D9, 0x0DA, 0x0DA, 0x0DA, 0x0DA,
        0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DB, 0x0DB, 0x0DB, 0x0DB, 0x0DB, 0x0DB, 0x0DB,
        0x0DB, 0x0DB, 0x0DB, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC,
        0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DE, 0x0DE, 0x0DE,
        0x0DE, 0x0DE, 0x0DE, 0x0DE, 0x0DE, 0x0DE, 0x0DE, 0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0DF,
        0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0,
        0x0E0, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E2, 0x0E2,
        0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E3, 0x0E3, 0x0E3, 0x0E3, 0x0E3,
        0x0E3, 0x0E3, 0x0E3, 0x0E3, 0x0E3, 0x0E4, 0x0E4, 0x0E4, 0x0E4, 0x0E4, 0x0E4, 0x0E4, 0x0E4,
        0x0E4, 0x0E4, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E6,
        0x0E6, 0x0E6, 0x0E6, 0x0E6, 0x0E6, 0x0E6, 0x0E6, 0x0E6, 0x0E6, 0x0E7, 0x0E7, 0x0E7, 0x0E7,
        0x0E7, 0x0E7, 0x0E7, 0x0E7, 0x0E7, 0x0E7, 0x0E8, 0x0E8, 0x0E8, 0x0E8, 0x0E8, 0x0E8, 0x0E8,
        0x0E8, 0x0E8, 0x0E8, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9,
        0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EB, 0x0EB, 0x0EB,
        0x0EB, 0x0EB, 0x0EB, 0x0EB, 0x0EB, 0x0EB, 0x0EB, 0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0EC,
        0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0ED, 0x0ED, 0x0ED, 0x0ED, 0x0ED, 0x0ED, 0x0ED, 0x0ED, 0x0ED,
        0x0ED, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EF, 0x0EF,
        0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0F0, 0x0F0, 0x0F0, 0x0F0, 0x0F0,
        0x0F0, 0x0F0, 0x0F0, 0x0F0, 0x0F0, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F1,
        0x0F1, 0x0F1, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F3,
        0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F4, 0x0F4, 0x0F4, 0x0F4,
        0x0F4, 0x0F4, 0x0F4, 0x0F4, 0x0F4, 0x0F4, 0x0F5, 0x0F5, 0x0F5, 0x0F5, 0x0F5, 0x0F5, 0x0F5,
        0x0F5, 0x0F5, 0x0F5, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6,
        0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F8, 0x0F8, 0x0F8,
        0x0F8, 0x0F8, 0x0F8, 0x0F8, 0x0F8, 0x0F8, 0x0F8, 0x0F9, 0x0F9, 0x0F9, 0x0F9, 0x0F9, 0x0F9,
        0x0F9, 0x0F9, 0x0F9, 0x0F9, 0x0FA, 0x0FA, 0x0FA, 0x0FA, 0x0FA, 0x0FA, 0x0FA, 0x0FA, 0x0FA,
        0x0FA, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FC, 0x0FC,
        0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FD, 0x0FD, 0x0FD, 0x0FD, 0x0FD,
        0x0FD, 0x0FD, 0x0FD, 0x0FD, 0x0FD, 0x0FE, 0x0FE, 0x0FE, 0x0FE, 0x0FE, 0x0FE, 0x0FE, 0x0FE,
        0x0FE, 0x0FE, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x100,
        0x100, 0x100, 0x100, 0x100, 0x100, 0x100, 0x100, 0x100, 0x100, 0x101, 0x101, 0x101, 0x101,
        0x101, 0x101, 0x101, 0x101, 0x101, 0x101, 0x102, 0x102, 0x102, 0x102, 0x102, 0x102, 0x102,
        0x102, 0x102, 0x102, 0x103, 0x103, 0x103, 0x103, 0x103, 0x103, 0x103, 0x103, 0x103, 0x103,
        0x104, 0x104, 0x104, 0x104, 0x104, 0x104, 0x104, 0x104, 0x104, 0x104, 0x105, 0x105, 0x105,
        0x105, 0x105, 0x105, 0x105, 0x105, 0x105, 0x105, 0x106, 0x106, 0x106, 0x106, 0x106, 0x106,
        0x106, 0x106, 0x106, 0x106, 0x107, 0x107, 0x107, 0x107, 0x107, 0x107, 0x107, 0x107, 0x107,
        0x107, 0x108, 0x108, 0x108, 0x108, 0x108, 0x108, 0x108, 0x108, 0x108, 0x108, 0x109, 0x109,
        0x109, 0x109, 0x109, 0x109, 0x109, 0x109, 0x109, 0x109, 0x10A, 0x10A, 0x10A, 0x10A, 0x10A,
        0x10A, 0x10A, 0x10A, 0x10A, 0x10A, 0x10B, 0x10B, 0x10B, 0x10B, 0x10B, 0x10B, 0x10B, 0x10B,
        0x10B, 0x10B, 0x10C, 0x10C, 0x10C, 0x10C, 0x10C, 0x10C, 0x10C, 0x10C, 0x10C, 0x10C, 0x10D,
        0x10D, 0x10D, 0x10D, 0x10D, 0x10D, 0x10D, 0x10D, 0x10D, 0x10D, 0x10E, 0x10E, 0x10E, 0x10E,
        0x10E, 0x10E, 0x10E, 0x10E, 0x10E, 0x10E, 0x10F, 0x10F, 0x10F, 0x10F, 0x10F, 0x10F, 0x10F,
        0x10F, 0x10F, 0x10F, 0x110, 0x110, 0x110, 0x110, 0x110, 0x110, 0x110, 0x110, 0x110, 0x110,
        0x111, 0x111, 0x111, 0x111, 0x111, 0x111, 0x111, 0x111, 0x111, 0x111, 0x112, 0x112, 0x112,
        0x112, 0x112, 0x112, 0x112, 0x112, 0x112, 0x112, 0x113, 0x113, 0x113, 0x113, 0x113, 0x113,
        0x113, 0x113, 0x113, 0x113, 0x114, 0x114, 0x114, 0x114, 0x114, 0x114, 0x114, 0x114, 0x114,
        0x114, 0x115, 0x115, 0x115, 0x115, 0x115, 0x115, 0x115, 0x115, 0x115, 0x115, 0x116, 0x116,
        0x116, 0x116, 0x116, 0x116, 0x116, 0x116, 0x116, 0x116, 0x117, 0x117, 0x117, 0x117, 0x117,
        0x117, 0x117, 0x117, 0x117, 0x117, 0x118, 0x118, 0x118, 0x118, 0x118, 0x118, 0x118, 0x118,
        0x118, 0x118, 0x119, 0x119, 0x119, 0x119, 0x119, 0x119, 0x119, 0x119, 0x119, 0x119, 0x11A,
        0x11A, 0x11A, 0x11A, 0x11A, 0x11A, 0x11A, 0x11A, 0x11A, 0x11A, 0x11B, 0x11B, 0x11B, 0x11B,
        0x11B, 0x11B, 0x11B, 0x11B, 0x11B, 0x11B, 0x11C, 0x11C, 0x11C, 0x11C, 0x11C, 0x11C, 0x11C,
        0x11C, 0x11C, 0x11C, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D,
        0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11F, 0x11F, 0x11F,
        0x11F, 0x11F, 0x11F, 0x11F, 0x11F, 0x11F, 0x11F, 0x120, 0x120, 0x120, 0x120, 0x120, 0x120,
        0x120, 0x120, 0x120, 0x120, 0x121, 0x121, 0x121, 0x121, 0x121, 0x121, 0x121, 0x121, 0x121,
        0x121, 0x122, 0x122, 0x122, 0x122, 0x122, 0x122, 0x122, 0x122, 0x122, 0x122, 0x123, 0x123,
        0x123, 0x123, 0x123, 0x123, 0x123, 0x123, 0x123, 0x123, 0x124, 0x124, 0x124, 0x124, 0x124,
        0x124, 0x124, 0x124, 0x124, 0x124, 0x125, 0x125, 0x125, 0x125, 0x125, 0x125, 0x125, 0x125,
        0x125, 0x125, 0x126, 0x126, 0x126, 0x126, 0x126, 0x126, 0x126, 0x126, 0x126, 0x126, 0x127,
        0x127, 0x127, 0x127, 0x127, 0x127, 0x127, 0x127, 0x127, 0x127, 0x1D2, 0x1D2, 0x1D2, 0x1D2,
        0x1D2, 0x1D2, 0x1D2, 0x1D2, 0x1D2, 0x1D2, 0x1D3, 0x1D3, 0x1D3, 0x1D3, 0x1D3, 0x1D3, 0x1D3,
        0x1D3, 0x1D3, 0x1D3, 0x1D4, 0x1D4, 0x1D4, 0x1D4, 0x1D4, 0x1D4, 0x1D4, 0x1D4, 0x1D4, 0x1D4,
        0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D6, 0x1D6, 0x1D6,
        0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D7, 0x1D7, 0x1D7, 0x1D7, 0x1D7, 0x1D7,
        0x1D7, 0x1D7, 0x1D7, 0x1D7, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D8,
        0x1D8, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1DA, 0x1DA,
        0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DB, 0x1DB, 0x1DB, 0x1DB, 0x1DB,
        0x1DB, 0x1DB, 0x1DB, 0x1DB, 0x1DB, 0x1DC, 0x1DC, 0x1DC, 0x1DC, 0x1DC, 0x1DC, 0x1DC, 0x1DC,
        0x1DC, 0x1DC, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DE,
        0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DF, 0x1DF, 0x1DF, 0x1DF,
        0x1DF, 0x1DF, 0x1DF, 0x1DF, 0x1DF, 0x1DF, 0x1E0, 0x1E0, 0x1E0, 0x1E0, 0x1E0, 0x1E0, 0x1E0,
        0x1E0, 0x1E0, 0x1E0, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1,
        0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E3, 0x1E3, 0x1E3,
        0x1E3, 0x1E3, 0x1E3, 0x1E3, 0x1E3, 0x1E3, 0x1E3, 0x1E4, 0x1E4, 0x1E4, 0x1E4, 0x1E4, 0x1E4,
        0x1E4, 0x1E4, 0x1E4, 0x1E4, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5,
        0x1E5, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E7, 0x1E7,
        0x1E7, 0x1E7, 0x1E7, 0x1E7, 0x1E7, 0x1E7, 0x1E7, 0x1E7, 0x1E8, 0x1E8, 0x1E8, 0x1E8, 0x1E8,
        0x1E8, 0x1E8, 0x1E8, 0x1E8, 0x1E8, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9,
        0x1E9, 0x1E9, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EB,
        0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EC, 0x1EC, 0x1EC, 0x1EC,
        0x1EC, 0x1EC, 0x1EC, 0x1EC, 0x1EC, 0x1EC, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1ED,
        0x1ED, 0x1ED, 0x1ED, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE,
        0x1EF, 0x1EF, 0x1EF, 0x1EF, 0x1EF, 0x1EF, 0x1EF, 0x1EF, 0x1EF, 0x1EF, 0x1F0, 0x1F0, 0x1F0,
        0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F1, 0x1F1, 0x1F1, 0x1F1, 0x1F1, 0x1F1,
        0x1F1, 0x1F1, 0x1F1, 0x1F1, 0x1F2, 0x1F2, 0x1F2, 0x1F2, 0x1F2, 0x1F2, 0x1F2, 0x1F2, 0x1F2,
        0x1F2, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F4, 0x1F4,
        0x1F4, 0x1F4, 0x1F4, 0x1F4, 0x1F4, 0x1F4, 0x1F4, 0x1F4, 0x1F5, 0x1F5, 0x1F5, 0x1F5, 0x1F5,
        0x1F5, 0x1F5, 0x1F5, 0x1F5, 0x1F5, 0x1F6, 0x1F6, 0x1F6, 0x1F6, 0x1F6, 0x1F6, 0x1F6, 0x1F6,
        0x1F6, 0x1F6, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F8,
        0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F9, 0x1F9, 0x1F9, 0x1F9,
        0x1F9, 0x1F9, 0x1F9, 0x1F9, 0x1F9, 0x1F9, 0x1FA, 0x1FA, 0x1FA, 0x1FA, 0x1FA, 0x1FA, 0x1FA,
        0x1FA, 0x1FA, 0x1FA, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB,
        0x1FC, 0x1FC, 0x1FC, 0x1FC, 0x1FC, 0x1FC, 0x1FC, 0x1FC, 0x1FC, 0x1FC, 0x1FD, 0x1FD, 0x1FD,
        0x1FD, 0x1FD, 0x1FD, 0x1FD, 0x1FD, 0x1FD, 0x1FD, 0x1FE, 0x1FE, 0x1FE, 0x1FE, 0x1FE, 0x1FE,
        0x1FE, 0x1FE, 0x1FE, 0x1FE, 0x1FF, 0x1FF, 0x1FF, 0x1FF, 0x1FF, 0x1FF, 0x1FF, 0x1FF, 0x1FF,
        0x1FF,
    ];

    h_counter_test(
        EXPECTED_PIXELS,
        EXPECTED_H,
        scanline_mclk_to_pixel_h32,
        pixel_to_internal_h_h32,
    );
}

#[test]
fn h_counter_h40() {
    const EXPECTED_PIXELS: &[u16; 3420] = &[
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3,
        3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7,
        7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 10, 10, 10, 10, 10, 10, 10, 10,
        11, 11, 11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13,
        13, 14, 14, 14, 14, 14, 14, 14, 14, 15, 15, 15, 15, 15, 15, 15, 15, 16, 16, 16, 16, 16, 16,
        16, 16, 17, 17, 17, 17, 17, 17, 17, 17, 18, 18, 18, 18, 18, 18, 18, 18, 19, 19, 19, 19, 19,
        19, 19, 19, 20, 20, 20, 20, 20, 20, 20, 20, 21, 21, 21, 21, 21, 21, 21, 21, 22, 22, 22, 22,
        22, 22, 22, 22, 23, 23, 23, 23, 23, 23, 23, 23, 24, 24, 24, 24, 24, 24, 24, 24, 25, 25, 25,
        25, 25, 25, 25, 25, 26, 26, 26, 26, 26, 26, 26, 26, 27, 27, 27, 27, 27, 27, 27, 27, 28, 28,
        28, 28, 28, 28, 28, 28, 29, 29, 29, 29, 29, 29, 29, 29, 30, 30, 30, 30, 30, 30, 30, 30, 31,
        31, 31, 31, 31, 31, 31, 31, 32, 32, 32, 32, 32, 32, 32, 32, 33, 33, 33, 33, 33, 33, 33, 33,
        34, 34, 34, 34, 34, 34, 34, 34, 35, 35, 35, 35, 35, 35, 35, 35, 36, 36, 36, 36, 36, 36, 36,
        36, 37, 37, 37, 37, 37, 37, 37, 37, 38, 38, 38, 38, 38, 38, 38, 38, 39, 39, 39, 39, 39, 39,
        39, 39, 40, 40, 40, 40, 40, 40, 40, 40, 41, 41, 41, 41, 41, 41, 41, 41, 42, 42, 42, 42, 42,
        42, 42, 42, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 44, 44, 44, 44, 44, 45, 45, 45, 45,
        45, 45, 45, 45, 46, 46, 46, 46, 46, 46, 46, 46, 47, 47, 47, 47, 47, 47, 47, 47, 48, 48, 48,
        48, 48, 48, 48, 48, 49, 49, 49, 49, 49, 49, 49, 49, 50, 50, 50, 50, 50, 50, 50, 50, 51, 51,
        51, 51, 51, 51, 51, 51, 52, 52, 52, 52, 52, 52, 52, 52, 53, 53, 53, 53, 53, 53, 53, 53, 54,
        54, 54, 54, 54, 54, 54, 54, 55, 55, 55, 55, 55, 55, 55, 55, 56, 56, 56, 56, 56, 56, 56, 56,
        57, 57, 57, 57, 57, 57, 57, 57, 58, 58, 58, 58, 58, 58, 58, 58, 59, 59, 59, 59, 59, 59, 59,
        59, 60, 60, 60, 60, 60, 60, 60, 60, 61, 61, 61, 61, 61, 61, 61, 61, 62, 62, 62, 62, 62, 62,
        62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65,
        65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 67, 68, 68, 68, 68,
        68, 68, 68, 68, 69, 69, 69, 69, 69, 69, 69, 69, 70, 70, 70, 70, 70, 70, 70, 70, 71, 71, 71,
        71, 71, 71, 71, 71, 72, 72, 72, 72, 72, 72, 72, 72, 73, 73, 73, 73, 73, 73, 73, 73, 74, 74,
        74, 74, 74, 74, 74, 74, 75, 75, 75, 75, 75, 75, 75, 75, 76, 76, 76, 76, 76, 76, 76, 76, 77,
        77, 77, 77, 77, 77, 77, 77, 78, 78, 78, 78, 78, 78, 78, 78, 79, 79, 79, 79, 79, 79, 79, 79,
        80, 80, 80, 80, 80, 80, 80, 80, 81, 81, 81, 81, 81, 81, 81, 81, 82, 82, 82, 82, 82, 82, 82,
        82, 83, 83, 83, 83, 83, 83, 83, 83, 84, 84, 84, 84, 84, 84, 84, 84, 85, 85, 85, 85, 85, 85,
        85, 85, 86, 86, 86, 86, 86, 86, 86, 86, 87, 87, 87, 87, 87, 87, 87, 87, 88, 88, 88, 88, 88,
        88, 88, 88, 89, 89, 89, 89, 89, 89, 89, 89, 90, 90, 90, 90, 90, 90, 90, 90, 91, 91, 91, 91,
        91, 91, 91, 91, 92, 92, 92, 92, 92, 92, 92, 92, 93, 93, 93, 93, 93, 93, 93, 93, 94, 94, 94,
        94, 94, 94, 94, 94, 95, 95, 95, 95, 95, 95, 95, 95, 96, 96, 96, 96, 96, 96, 96, 96, 97, 97,
        97, 97, 97, 97, 97, 97, 98, 98, 98, 98, 98, 98, 98, 98, 99, 99, 99, 99, 99, 99, 99, 99,
        100, 100, 100, 100, 100, 100, 100, 100, 101, 101, 101, 101, 101, 101, 101, 101, 102, 102,
        102, 102, 102, 102, 102, 102, 103, 103, 103, 103, 103, 103, 103, 103, 104, 104, 104, 104,
        104, 104, 104, 104, 105, 105, 105, 105, 105, 105, 105, 105, 106, 106, 106, 106, 106, 106,
        106, 106, 107, 107, 107, 107, 107, 107, 107, 107, 108, 108, 108, 108, 108, 108, 108, 108,
        109, 109, 109, 109, 109, 109, 109, 109, 110, 110, 110, 110, 110, 110, 110, 110, 111, 111,
        111, 111, 111, 111, 111, 111, 112, 112, 112, 112, 112, 112, 112, 112, 113, 113, 113, 113,
        113, 113, 113, 113, 114, 114, 114, 114, 114, 114, 114, 114, 115, 115, 115, 115, 115, 115,
        115, 115, 116, 116, 116, 116, 116, 116, 116, 116, 117, 117, 117, 117, 117, 117, 117, 117,
        118, 118, 118, 118, 118, 118, 118, 118, 119, 119, 119, 119, 119, 119, 119, 119, 120, 120,
        120, 120, 120, 120, 120, 120, 121, 121, 121, 121, 121, 121, 121, 121, 122, 122, 122, 122,
        122, 122, 122, 122, 123, 123, 123, 123, 123, 123, 123, 123, 124, 124, 124, 124, 124, 124,
        124, 124, 125, 125, 125, 125, 125, 125, 125, 125, 126, 126, 126, 126, 126, 126, 126, 126,
        127, 127, 127, 127, 127, 127, 127, 127, 128, 128, 128, 128, 128, 128, 128, 128, 129, 129,
        129, 129, 129, 129, 129, 129, 130, 130, 130, 130, 130, 130, 130, 130, 131, 131, 131, 131,
        131, 131, 131, 131, 132, 132, 132, 132, 132, 132, 132, 132, 133, 133, 133, 133, 133, 133,
        133, 133, 134, 134, 134, 134, 134, 134, 134, 134, 135, 135, 135, 135, 135, 135, 135, 135,
        136, 136, 136, 136, 136, 136, 136, 136, 137, 137, 137, 137, 137, 137, 137, 137, 138, 138,
        138, 138, 138, 138, 138, 138, 139, 139, 139, 139, 139, 139, 139, 139, 140, 140, 140, 140,
        140, 140, 140, 140, 141, 141, 141, 141, 141, 141, 141, 141, 142, 142, 142, 142, 142, 142,
        142, 142, 143, 143, 143, 143, 143, 143, 143, 143, 144, 144, 144, 144, 144, 144, 144, 144,
        145, 145, 145, 145, 145, 145, 145, 145, 146, 146, 146, 146, 146, 146, 146, 146, 147, 147,
        147, 147, 147, 147, 147, 147, 148, 148, 148, 148, 148, 148, 148, 148, 149, 149, 149, 149,
        149, 149, 149, 149, 150, 150, 150, 150, 150, 150, 150, 150, 151, 151, 151, 151, 151, 151,
        151, 151, 152, 152, 152, 152, 152, 152, 152, 152, 153, 153, 153, 153, 153, 153, 153, 153,
        154, 154, 154, 154, 154, 154, 154, 154, 155, 155, 155, 155, 155, 155, 155, 155, 156, 156,
        156, 156, 156, 156, 156, 156, 157, 157, 157, 157, 157, 157, 157, 157, 158, 158, 158, 158,
        158, 158, 158, 158, 159, 159, 159, 159, 159, 159, 159, 159, 160, 160, 160, 160, 160, 160,
        160, 160, 161, 161, 161, 161, 161, 161, 161, 161, 162, 162, 162, 162, 162, 162, 162, 162,
        163, 163, 163, 163, 163, 163, 163, 163, 164, 164, 164, 164, 164, 164, 164, 164, 165, 165,
        165, 165, 165, 165, 165, 165, 166, 166, 166, 166, 166, 166, 166, 166, 167, 167, 167, 167,
        167, 167, 167, 167, 168, 168, 168, 168, 168, 168, 168, 168, 169, 169, 169, 169, 169, 169,
        169, 169, 170, 170, 170, 170, 170, 170, 170, 170, 171, 171, 171, 171, 171, 171, 171, 171,
        172, 172, 172, 172, 172, 172, 172, 172, 173, 173, 173, 173, 173, 173, 173, 173, 174, 174,
        174, 174, 174, 174, 174, 174, 175, 175, 175, 175, 175, 175, 175, 175, 176, 176, 176, 176,
        176, 176, 176, 176, 177, 177, 177, 177, 177, 177, 177, 177, 178, 178, 178, 178, 178, 178,
        178, 178, 179, 179, 179, 179, 179, 179, 179, 179, 180, 180, 180, 180, 180, 180, 180, 180,
        181, 181, 181, 181, 181, 181, 181, 181, 182, 182, 182, 182, 182, 182, 182, 182, 183, 183,
        183, 183, 183, 183, 183, 183, 184, 184, 184, 184, 184, 184, 184, 184, 185, 185, 185, 185,
        185, 185, 185, 185, 186, 186, 186, 186, 186, 186, 186, 186, 187, 187, 187, 187, 187, 187,
        187, 187, 188, 188, 188, 188, 188, 188, 188, 188, 189, 189, 189, 189, 189, 189, 189, 189,
        190, 190, 190, 190, 190, 190, 190, 190, 191, 191, 191, 191, 191, 191, 191, 191, 192, 192,
        192, 192, 192, 192, 192, 192, 193, 193, 193, 193, 193, 193, 193, 193, 194, 194, 194, 194,
        194, 194, 194, 194, 195, 195, 195, 195, 195, 195, 195, 195, 196, 196, 196, 196, 196, 196,
        196, 196, 197, 197, 197, 197, 197, 197, 197, 197, 198, 198, 198, 198, 198, 198, 198, 198,
        199, 199, 199, 199, 199, 199, 199, 199, 200, 200, 200, 200, 200, 200, 200, 200, 201, 201,
        201, 201, 201, 201, 201, 201, 202, 202, 202, 202, 202, 202, 202, 202, 203, 203, 203, 203,
        203, 203, 203, 203, 204, 204, 204, 204, 204, 204, 204, 204, 205, 205, 205, 205, 205, 205,
        205, 205, 206, 206, 206, 206, 206, 206, 206, 206, 207, 207, 207, 207, 207, 207, 207, 207,
        208, 208, 208, 208, 208, 208, 208, 208, 209, 209, 209, 209, 209, 209, 209, 209, 210, 210,
        210, 210, 210, 210, 210, 210, 211, 211, 211, 211, 211, 211, 211, 211, 212, 212, 212, 212,
        212, 212, 212, 212, 213, 213, 213, 213, 213, 213, 213, 213, 214, 214, 214, 214, 214, 214,
        214, 214, 215, 215, 215, 215, 215, 215, 215, 215, 216, 216, 216, 216, 216, 216, 216, 216,
        217, 217, 217, 217, 217, 217, 217, 217, 218, 218, 218, 218, 218, 218, 218, 218, 219, 219,
        219, 219, 219, 219, 219, 219, 220, 220, 220, 220, 220, 220, 220, 220, 221, 221, 221, 221,
        221, 221, 221, 221, 222, 222, 222, 222, 222, 222, 222, 222, 223, 223, 223, 223, 223, 223,
        223, 223, 224, 224, 224, 224, 224, 224, 224, 224, 225, 225, 225, 225, 225, 225, 225, 225,
        226, 226, 226, 226, 226, 226, 226, 226, 227, 227, 227, 227, 227, 227, 227, 227, 228, 228,
        228, 228, 228, 228, 228, 228, 229, 229, 229, 229, 229, 229, 229, 229, 230, 230, 230, 230,
        230, 230, 230, 230, 231, 231, 231, 231, 231, 231, 231, 231, 232, 232, 232, 232, 232, 232,
        232, 232, 233, 233, 233, 233, 233, 233, 233, 233, 234, 234, 234, 234, 234, 234, 234, 234,
        235, 235, 235, 235, 235, 235, 235, 235, 236, 236, 236, 236, 236, 236, 236, 236, 237, 237,
        237, 237, 237, 237, 237, 237, 238, 238, 238, 238, 238, 238, 238, 238, 239, 239, 239, 239,
        239, 239, 239, 239, 240, 240, 240, 240, 240, 240, 240, 240, 241, 241, 241, 241, 241, 241,
        241, 241, 242, 242, 242, 242, 242, 242, 242, 242, 243, 243, 243, 243, 243, 243, 243, 243,
        244, 244, 244, 244, 244, 244, 244, 244, 245, 245, 245, 245, 245, 245, 245, 245, 246, 246,
        246, 246, 246, 246, 246, 246, 247, 247, 247, 247, 247, 247, 247, 247, 248, 248, 248, 248,
        248, 248, 248, 248, 249, 249, 249, 249, 249, 249, 249, 249, 250, 250, 250, 250, 250, 250,
        250, 250, 251, 251, 251, 251, 251, 251, 251, 251, 252, 252, 252, 252, 252, 252, 252, 252,
        253, 253, 253, 253, 253, 253, 253, 253, 254, 254, 254, 254, 254, 254, 254, 254, 255, 255,
        255, 255, 255, 255, 255, 255, 256, 256, 256, 256, 256, 256, 256, 256, 257, 257, 257, 257,
        257, 257, 257, 257, 258, 258, 258, 258, 258, 258, 258, 258, 259, 259, 259, 259, 259, 259,
        259, 259, 260, 260, 260, 260, 260, 260, 260, 260, 261, 261, 261, 261, 261, 261, 261, 261,
        262, 262, 262, 262, 262, 262, 262, 262, 263, 263, 263, 263, 263, 263, 263, 263, 264, 264,
        264, 264, 264, 264, 264, 264, 265, 265, 265, 265, 265, 265, 265, 265, 266, 266, 266, 266,
        266, 266, 266, 266, 267, 267, 267, 267, 267, 267, 267, 267, 268, 268, 268, 268, 268, 268,
        268, 268, 269, 269, 269, 269, 269, 269, 269, 269, 270, 270, 270, 270, 270, 270, 270, 270,
        271, 271, 271, 271, 271, 271, 271, 271, 272, 272, 272, 272, 272, 272, 272, 272, 273, 273,
        273, 273, 273, 273, 273, 273, 274, 274, 274, 274, 274, 274, 274, 274, 275, 275, 275, 275,
        275, 275, 275, 275, 276, 276, 276, 276, 276, 276, 276, 276, 277, 277, 277, 277, 277, 277,
        277, 277, 278, 278, 278, 278, 278, 278, 278, 278, 279, 279, 279, 279, 279, 279, 279, 279,
        280, 280, 280, 280, 280, 280, 280, 280, 281, 281, 281, 281, 281, 281, 281, 281, 282, 282,
        282, 282, 282, 282, 282, 282, 283, 283, 283, 283, 283, 283, 283, 283, 284, 284, 284, 284,
        284, 284, 284, 284, 285, 285, 285, 285, 285, 285, 285, 285, 286, 286, 286, 286, 286, 286,
        286, 286, 287, 287, 287, 287, 287, 287, 287, 287, 288, 288, 288, 288, 288, 288, 288, 288,
        289, 289, 289, 289, 289, 289, 289, 289, 290, 290, 290, 290, 290, 290, 290, 290, 291, 291,
        291, 291, 291, 291, 291, 291, 292, 292, 292, 292, 292, 292, 292, 292, 293, 293, 293, 293,
        293, 293, 293, 293, 294, 294, 294, 294, 294, 294, 294, 294, 295, 295, 295, 295, 295, 295,
        295, 295, 296, 296, 296, 296, 296, 296, 296, 296, 297, 297, 297, 297, 297, 297, 297, 297,
        298, 298, 298, 298, 298, 298, 298, 298, 299, 299, 299, 299, 299, 299, 299, 299, 300, 300,
        300, 300, 300, 300, 300, 300, 301, 301, 301, 301, 301, 301, 301, 301, 302, 302, 302, 302,
        302, 302, 302, 302, 303, 303, 303, 303, 303, 303, 303, 303, 304, 304, 304, 304, 304, 304,
        304, 304, 305, 305, 305, 305, 305, 305, 305, 305, 306, 306, 306, 306, 306, 306, 306, 306,
        307, 307, 307, 307, 307, 307, 307, 307, 308, 308, 308, 308, 308, 308, 308, 308, 309, 309,
        309, 309, 309, 309, 309, 309, 310, 310, 310, 310, 310, 310, 310, 310, 311, 311, 311, 311,
        311, 311, 311, 311, 312, 312, 312, 312, 312, 312, 312, 312, 313, 313, 313, 313, 313, 313,
        313, 313, 314, 314, 314, 314, 314, 314, 314, 314, 315, 315, 315, 315, 315, 315, 315, 315,
        316, 316, 316, 316, 316, 316, 316, 316, 317, 317, 317, 317, 317, 317, 317, 317, 318, 318,
        318, 318, 318, 318, 318, 318, 319, 319, 319, 319, 319, 319, 319, 319, 320, 320, 320, 320,
        320, 320, 320, 320, 321, 321, 321, 321, 321, 321, 321, 321, 322, 322, 322, 322, 322, 322,
        322, 322, 323, 323, 323, 323, 323, 323, 323, 323, 324, 324, 324, 324, 324, 324, 324, 324,
        325, 325, 325, 325, 325, 325, 325, 325, 326, 326, 326, 326, 326, 326, 326, 326, 327, 327,
        327, 327, 327, 327, 327, 327, 328, 328, 328, 328, 328, 328, 328, 328, 329, 329, 329, 329,
        329, 329, 329, 329, 330, 330, 330, 330, 330, 330, 330, 330, 331, 331, 331, 331, 331, 331,
        331, 331, 332, 332, 332, 332, 332, 332, 332, 332, 333, 333, 333, 333, 333, 333, 333, 333,
        334, 334, 334, 334, 334, 334, 334, 334, 335, 335, 335, 335, 335, 335, 335, 335, 336, 336,
        336, 336, 336, 336, 336, 336, 337, 337, 337, 337, 337, 337, 337, 337, 338, 338, 338, 338,
        338, 338, 338, 338, 339, 339, 339, 339, 339, 339, 339, 339, 340, 340, 340, 340, 340, 340,
        340, 340, 341, 341, 341, 341, 341, 341, 341, 341, 342, 342, 342, 342, 342, 342, 342, 342,
        343, 343, 343, 343, 343, 343, 343, 343, 344, 344, 344, 344, 344, 344, 344, 344, 345, 345,
        345, 345, 345, 345, 345, 345, 346, 346, 346, 346, 346, 346, 346, 346, 347, 347, 347, 347,
        347, 347, 347, 347, 348, 348, 348, 348, 348, 348, 348, 348, 349, 349, 349, 349, 349, 349,
        349, 349, 350, 350, 350, 350, 350, 350, 350, 350, 351, 351, 351, 351, 351, 351, 351, 351,
        352, 352, 352, 352, 352, 352, 352, 352, 353, 353, 353, 353, 353, 353, 353, 353, 354, 354,
        354, 354, 354, 354, 354, 354, 355, 355, 355, 355, 355, 355, 355, 355, 356, 356, 356, 356,
        356, 356, 356, 356, 357, 357, 357, 357, 357, 357, 357, 357, 358, 358, 358, 358, 358, 358,
        358, 358, 359, 359, 359, 359, 359, 359, 359, 359, 360, 360, 360, 360, 360, 360, 360, 360,
        361, 361, 361, 361, 361, 361, 361, 361, 362, 362, 362, 362, 362, 362, 362, 362, 363, 363,
        363, 363, 363, 363, 363, 363, 364, 364, 364, 364, 364, 364, 364, 364, 365, 365, 365, 365,
        365, 365, 365, 365, 366, 366, 366, 366, 366, 366, 366, 366, 367, 367, 367, 367, 367, 367,
        367, 367, 368, 368, 368, 368, 368, 368, 368, 368, 369, 369, 369, 369, 369, 369, 369, 369,
        369, 369, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 371, 371, 371, 371, 371, 371,
        371, 371, 371, 371, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 373, 373, 373, 373,
        373, 373, 373, 373, 373, 373, 374, 374, 374, 374, 374, 374, 374, 374, 374, 374, 375, 375,
        375, 375, 375, 375, 375, 375, 375, 375, 376, 376, 376, 376, 376, 376, 376, 376, 376, 377,
        377, 377, 377, 377, 377, 377, 377, 377, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378,
        379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 380, 380, 380, 380, 380, 380, 380, 380,
        380, 380, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 382, 382, 382, 382, 382, 382,
        382, 382, 382, 382, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 384, 384, 384, 384,
        384, 384, 384, 384, 384, 384, 385, 385, 385, 385, 385, 385, 385, 385, 386, 386, 386, 386,
        386, 386, 386, 386, 386, 386, 387, 387, 387, 387, 387, 387, 387, 387, 387, 387, 388, 388,
        388, 388, 388, 388, 388, 388, 388, 388, 389, 389, 389, 389, 389, 389, 389, 389, 389, 389,
        390, 390, 390, 390, 390, 390, 390, 390, 390, 390, 391, 391, 391, 391, 391, 391, 391, 391,
        391, 391, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 393, 393, 393, 393, 393, 393,
        393, 393, 393, 394, 394, 394, 394, 394, 394, 394, 394, 394, 395, 395, 395, 395, 395, 395,
        395, 395, 395, 395, 396, 396, 396, 396, 396, 396, 396, 396, 396, 396, 397, 397, 397, 397,
        397, 397, 397, 397, 397, 397, 398, 398, 398, 398, 398, 398, 398, 398, 398, 398, 399, 399,
        399, 399, 399, 399, 399, 399, 399, 399, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400,
        401, 401, 401, 401, 401, 401, 401, 401, 401, 401, 402, 402, 402, 402, 402, 402, 402, 402,
        403, 403, 403, 403, 403, 403, 403, 403, 404, 404, 404, 404, 404, 404, 404, 404, 405, 405,
        405, 405, 405, 405, 405, 405, 406, 406, 406, 406, 406, 406, 406, 406, 407, 407, 407, 407,
        407, 407, 407, 407, 408, 408, 408, 408, 408, 408, 408, 408, 409, 409, 409, 409, 409, 409,
        409, 409, 410, 410, 410, 410, 410, 410, 410, 410, 411, 411, 411, 411, 411, 411, 411, 411,
        412, 412, 412, 412, 412, 412, 412, 412, 413, 413, 413, 413, 413, 413, 413, 413, 414, 414,
        414, 414, 414, 414, 414, 414, 415, 415, 415, 415, 415, 415, 415, 415, 416, 416, 416, 416,
        416, 416, 416, 416, 417, 417, 417, 417, 417, 417, 417, 417, 418, 418, 418, 418, 418, 418,
        418, 418, 419, 419, 419, 419, 419, 419, 419, 419,
    ];
    const EXPECTED_H: &[u16; 3420] = &[
        0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x001, 0x001, 0x001, 0x001, 0x001,
        0x001, 0x001, 0x001, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x003, 0x003,
        0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
        0x004, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x006, 0x006, 0x006, 0x006,
        0x006, 0x006, 0x006, 0x006, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x008,
        0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
        0x009, 0x009, 0x00A, 0x00A, 0x00A, 0x00A, 0x00A, 0x00A, 0x00A, 0x00A, 0x00B, 0x00B, 0x00B,
        0x00B, 0x00B, 0x00B, 0x00B, 0x00B, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C, 0x00C,
        0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00D, 0x00E, 0x00E, 0x00E, 0x00E, 0x00E,
        0x00E, 0x00E, 0x00E, 0x00F, 0x00F, 0x00F, 0x00F, 0x00F, 0x00F, 0x00F, 0x00F, 0x010, 0x010,
        0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011,
        0x011, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x013, 0x013, 0x013, 0x013,
        0x013, 0x013, 0x013, 0x013, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x015,
        0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016,
        0x016, 0x016, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x018, 0x018, 0x018,
        0x018, 0x018, 0x018, 0x018, 0x018, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019,
        0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01A, 0x01B, 0x01B, 0x01B, 0x01B, 0x01B,
        0x01B, 0x01B, 0x01B, 0x01C, 0x01C, 0x01C, 0x01C, 0x01C, 0x01C, 0x01C, 0x01C, 0x01D, 0x01D,
        0x01D, 0x01D, 0x01D, 0x01D, 0x01D, 0x01D, 0x01E, 0x01E, 0x01E, 0x01E, 0x01E, 0x01E, 0x01E,
        0x01E, 0x01F, 0x01F, 0x01F, 0x01F, 0x01F, 0x01F, 0x01F, 0x01F, 0x020, 0x020, 0x020, 0x020,
        0x020, 0x020, 0x020, 0x020, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x022,
        0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023,
        0x023, 0x023, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x025, 0x025, 0x025,
        0x025, 0x025, 0x025, 0x025, 0x025, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026,
        0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x028, 0x028, 0x028, 0x028, 0x028,
        0x028, 0x028, 0x028, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x02A, 0x02A,
        0x02A, 0x02A, 0x02A, 0x02A, 0x02A, 0x02A, 0x02B, 0x02B, 0x02B, 0x02B, 0x02B, 0x02B, 0x02B,
        0x02B, 0x02C, 0x02C, 0x02C, 0x02C, 0x02C, 0x02C, 0x02C, 0x02C, 0x02D, 0x02D, 0x02D, 0x02D,
        0x02D, 0x02D, 0x02D, 0x02D, 0x02E, 0x02E, 0x02E, 0x02E, 0x02E, 0x02E, 0x02E, 0x02E, 0x02F,
        0x02F, 0x02F, 0x02F, 0x02F, 0x02F, 0x02F, 0x02F, 0x030, 0x030, 0x030, 0x030, 0x030, 0x030,
        0x030, 0x030, 0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x032, 0x032, 0x032,
        0x032, 0x032, 0x032, 0x032, 0x032, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033,
        0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x035, 0x035, 0x035, 0x035, 0x035,
        0x035, 0x035, 0x035, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x037, 0x037,
        0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038,
        0x038, 0x039, 0x039, 0x039, 0x039, 0x039, 0x039, 0x039, 0x039, 0x03A, 0x03A, 0x03A, 0x03A,
        0x03A, 0x03A, 0x03A, 0x03A, 0x03B, 0x03B, 0x03B, 0x03B, 0x03B, 0x03B, 0x03B, 0x03B, 0x03C,
        0x03C, 0x03C, 0x03C, 0x03C, 0x03C, 0x03C, 0x03C, 0x03D, 0x03D, 0x03D, 0x03D, 0x03D, 0x03D,
        0x03D, 0x03D, 0x03E, 0x03E, 0x03E, 0x03E, 0x03E, 0x03E, 0x03E, 0x03E, 0x03F, 0x03F, 0x03F,
        0x03F, 0x03F, 0x03F, 0x03F, 0x03F, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040,
        0x041, 0x041, 0x041, 0x041, 0x041, 0x041, 0x041, 0x041, 0x042, 0x042, 0x042, 0x042, 0x042,
        0x042, 0x042, 0x042, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043, 0x044, 0x044,
        0x044, 0x044, 0x044, 0x044, 0x044, 0x044, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045,
        0x045, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x047, 0x047, 0x047, 0x047,
        0x047, 0x047, 0x047, 0x047, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x049,
        0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x04A, 0x04A, 0x04A, 0x04A, 0x04A, 0x04A,
        0x04A, 0x04A, 0x04B, 0x04B, 0x04B, 0x04B, 0x04B, 0x04B, 0x04B, 0x04B, 0x04C, 0x04C, 0x04C,
        0x04C, 0x04C, 0x04C, 0x04C, 0x04C, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D, 0x04D,
        0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04E, 0x04F, 0x04F, 0x04F, 0x04F, 0x04F,
        0x04F, 0x04F, 0x04F, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050, 0x051, 0x051,
        0x051, 0x051, 0x051, 0x051, 0x051, 0x051, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052,
        0x052, 0x053, 0x053, 0x053, 0x053, 0x053, 0x053, 0x053, 0x053, 0x054, 0x054, 0x054, 0x054,
        0x054, 0x054, 0x054, 0x054, 0x055, 0x055, 0x055, 0x055, 0x055, 0x055, 0x055, 0x055, 0x056,
        0x056, 0x056, 0x056, 0x056, 0x056, 0x056, 0x056, 0x057, 0x057, 0x057, 0x057, 0x057, 0x057,
        0x057, 0x057, 0x058, 0x058, 0x058, 0x058, 0x058, 0x058, 0x058, 0x058, 0x059, 0x059, 0x059,
        0x059, 0x059, 0x059, 0x059, 0x059, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A, 0x05A,
        0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05B, 0x05C, 0x05C, 0x05C, 0x05C, 0x05C,
        0x05C, 0x05C, 0x05C, 0x05D, 0x05D, 0x05D, 0x05D, 0x05D, 0x05D, 0x05D, 0x05D, 0x05E, 0x05E,
        0x05E, 0x05E, 0x05E, 0x05E, 0x05E, 0x05E, 0x05F, 0x05F, 0x05F, 0x05F, 0x05F, 0x05F, 0x05F,
        0x05F, 0x060, 0x060, 0x060, 0x060, 0x060, 0x060, 0x060, 0x060, 0x061, 0x061, 0x061, 0x061,
        0x061, 0x061, 0x061, 0x061, 0x062, 0x062, 0x062, 0x062, 0x062, 0x062, 0x062, 0x062, 0x063,
        0x063, 0x063, 0x063, 0x063, 0x063, 0x063, 0x063, 0x064, 0x064, 0x064, 0x064, 0x064, 0x064,
        0x064, 0x064, 0x065, 0x065, 0x065, 0x065, 0x065, 0x065, 0x065, 0x065, 0x066, 0x066, 0x066,
        0x066, 0x066, 0x066, 0x066, 0x066, 0x067, 0x067, 0x067, 0x067, 0x067, 0x067, 0x067, 0x067,
        0x068, 0x068, 0x068, 0x068, 0x068, 0x068, 0x068, 0x068, 0x069, 0x069, 0x069, 0x069, 0x069,
        0x069, 0x069, 0x069, 0x06A, 0x06A, 0x06A, 0x06A, 0x06A, 0x06A, 0x06A, 0x06A, 0x06B, 0x06B,
        0x06B, 0x06B, 0x06B, 0x06B, 0x06B, 0x06B, 0x06C, 0x06C, 0x06C, 0x06C, 0x06C, 0x06C, 0x06C,
        0x06C, 0x06D, 0x06D, 0x06D, 0x06D, 0x06D, 0x06D, 0x06D, 0x06D, 0x06E, 0x06E, 0x06E, 0x06E,
        0x06E, 0x06E, 0x06E, 0x06E, 0x06F, 0x06F, 0x06F, 0x06F, 0x06F, 0x06F, 0x06F, 0x06F, 0x070,
        0x070, 0x070, 0x070, 0x070, 0x070, 0x070, 0x070, 0x071, 0x071, 0x071, 0x071, 0x071, 0x071,
        0x071, 0x071, 0x072, 0x072, 0x072, 0x072, 0x072, 0x072, 0x072, 0x072, 0x073, 0x073, 0x073,
        0x073, 0x073, 0x073, 0x073, 0x073, 0x074, 0x074, 0x074, 0x074, 0x074, 0x074, 0x074, 0x074,
        0x075, 0x075, 0x075, 0x075, 0x075, 0x075, 0x075, 0x075, 0x076, 0x076, 0x076, 0x076, 0x076,
        0x076, 0x076, 0x076, 0x077, 0x077, 0x077, 0x077, 0x077, 0x077, 0x077, 0x077, 0x078, 0x078,
        0x078, 0x078, 0x078, 0x078, 0x078, 0x078, 0x079, 0x079, 0x079, 0x079, 0x079, 0x079, 0x079,
        0x079, 0x07A, 0x07A, 0x07A, 0x07A, 0x07A, 0x07A, 0x07A, 0x07A, 0x07B, 0x07B, 0x07B, 0x07B,
        0x07B, 0x07B, 0x07B, 0x07B, 0x07C, 0x07C, 0x07C, 0x07C, 0x07C, 0x07C, 0x07C, 0x07C, 0x07D,
        0x07D, 0x07D, 0x07D, 0x07D, 0x07D, 0x07D, 0x07D, 0x07E, 0x07E, 0x07E, 0x07E, 0x07E, 0x07E,
        0x07E, 0x07E, 0x07F, 0x07F, 0x07F, 0x07F, 0x07F, 0x07F, 0x07F, 0x07F, 0x080, 0x080, 0x080,
        0x080, 0x080, 0x080, 0x080, 0x080, 0x081, 0x081, 0x081, 0x081, 0x081, 0x081, 0x081, 0x081,
        0x082, 0x082, 0x082, 0x082, 0x082, 0x082, 0x082, 0x082, 0x083, 0x083, 0x083, 0x083, 0x083,
        0x083, 0x083, 0x083, 0x084, 0x084, 0x084, 0x084, 0x084, 0x084, 0x084, 0x084, 0x085, 0x085,
        0x085, 0x085, 0x085, 0x085, 0x085, 0x085, 0x086, 0x086, 0x086, 0x086, 0x086, 0x086, 0x086,
        0x086, 0x087, 0x087, 0x087, 0x087, 0x087, 0x087, 0x087, 0x087, 0x088, 0x088, 0x088, 0x088,
        0x088, 0x088, 0x088, 0x088, 0x089, 0x089, 0x089, 0x089, 0x089, 0x089, 0x089, 0x089, 0x08A,
        0x08A, 0x08A, 0x08A, 0x08A, 0x08A, 0x08A, 0x08A, 0x08B, 0x08B, 0x08B, 0x08B, 0x08B, 0x08B,
        0x08B, 0x08B, 0x08C, 0x08C, 0x08C, 0x08C, 0x08C, 0x08C, 0x08C, 0x08C, 0x08D, 0x08D, 0x08D,
        0x08D, 0x08D, 0x08D, 0x08D, 0x08D, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E, 0x08E,
        0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x08F, 0x090, 0x090, 0x090, 0x090, 0x090,
        0x090, 0x090, 0x090, 0x091, 0x091, 0x091, 0x091, 0x091, 0x091, 0x091, 0x091, 0x092, 0x092,
        0x092, 0x092, 0x092, 0x092, 0x092, 0x092, 0x093, 0x093, 0x093, 0x093, 0x093, 0x093, 0x093,
        0x093, 0x094, 0x094, 0x094, 0x094, 0x094, 0x094, 0x094, 0x094, 0x095, 0x095, 0x095, 0x095,
        0x095, 0x095, 0x095, 0x095, 0x096, 0x096, 0x096, 0x096, 0x096, 0x096, 0x096, 0x096, 0x097,
        0x097, 0x097, 0x097, 0x097, 0x097, 0x097, 0x097, 0x098, 0x098, 0x098, 0x098, 0x098, 0x098,
        0x098, 0x098, 0x099, 0x099, 0x099, 0x099, 0x099, 0x099, 0x099, 0x099, 0x09A, 0x09A, 0x09A,
        0x09A, 0x09A, 0x09A, 0x09A, 0x09A, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B, 0x09B,
        0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09C, 0x09D, 0x09D, 0x09D, 0x09D, 0x09D,
        0x09D, 0x09D, 0x09D, 0x09E, 0x09E, 0x09E, 0x09E, 0x09E, 0x09E, 0x09E, 0x09E, 0x09F, 0x09F,
        0x09F, 0x09F, 0x09F, 0x09F, 0x09F, 0x09F, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A0, 0x0A0,
        0x0A0, 0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A1, 0x0A2, 0x0A2, 0x0A2, 0x0A2,
        0x0A2, 0x0A2, 0x0A2, 0x0A2, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A3, 0x0A4,
        0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A4, 0x0A5, 0x0A5, 0x0A5, 0x0A5, 0x0A5, 0x0A5,
        0x0A5, 0x0A5, 0x0A6, 0x0A6, 0x0A6, 0x0A6, 0x0A6, 0x0A6, 0x0A6, 0x0A6, 0x0A7, 0x0A7, 0x0A7,
        0x0A7, 0x0A7, 0x0A7, 0x0A7, 0x0A7, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8, 0x0A8,
        0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0A9, 0x0AA, 0x0AA, 0x0AA, 0x0AA, 0x0AA,
        0x0AA, 0x0AA, 0x0AA, 0x0AB, 0x0AB, 0x0AB, 0x0AB, 0x0AB, 0x0AB, 0x0AB, 0x0AB, 0x0AC, 0x0AC,
        0x0AC, 0x0AC, 0x0AC, 0x0AC, 0x0AC, 0x0AC, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD,
        0x0AD, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AF, 0x0AF, 0x0AF, 0x0AF,
        0x0AF, 0x0AF, 0x0AF, 0x0AF, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B1,
        0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B1, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B2,
        0x0B2, 0x0B2, 0x0B3, 0x0B3, 0x0B3, 0x0B3, 0x0B3, 0x0B3, 0x0B3, 0x0B3, 0x0B4, 0x0B4, 0x0B4,
        0x0B4, 0x0B4, 0x0B4, 0x0B4, 0x0B4, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5,
        0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B6, 0x0B7, 0x0B7, 0x0B7, 0x0B7, 0x0B7,
        0x0B7, 0x0B7, 0x0B7, 0x0B8, 0x0B8, 0x0B8, 0x0B8, 0x0B8, 0x0B8, 0x0B8, 0x0B8, 0x0B9, 0x0B9,
        0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BA, 0x0BA,
        0x0BA, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BC, 0x0BC, 0x0BC, 0x0BC,
        0x0BC, 0x0BC, 0x0BC, 0x0BC, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BE,
        0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BF, 0x0BF, 0x0BF, 0x0BF, 0x0BF, 0x0BF,
        0x0BF, 0x0BF, 0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C1, 0x0C1, 0x0C1,
        0x0C1, 0x0C1, 0x0C1, 0x0C1, 0x0C1, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2,
        0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C3, 0x0C4, 0x0C4, 0x0C4, 0x0C4, 0x0C4,
        0x0C4, 0x0C4, 0x0C4, 0x0C5, 0x0C5, 0x0C5, 0x0C5, 0x0C5, 0x0C5, 0x0C5, 0x0C5, 0x0C6, 0x0C6,
        0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C7, 0x0C7,
        0x0C7, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C9, 0x0C9, 0x0C9, 0x0C9,
        0x0C9, 0x0C9, 0x0C9, 0x0C9, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CB,
        0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CB, 0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CC,
        0x0CC, 0x0CC, 0x0CD, 0x0CD, 0x0CD, 0x0CD, 0x0CD, 0x0CD, 0x0CD, 0x0CD, 0x0CE, 0x0CE, 0x0CE,
        0x0CE, 0x0CE, 0x0CE, 0x0CE, 0x0CE, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF, 0x0CF,
        0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D1, 0x0D1, 0x0D1, 0x0D1, 0x0D1,
        0x0D1, 0x0D1, 0x0D1, 0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D3, 0x0D3,
        0x0D3, 0x0D3, 0x0D3, 0x0D3, 0x0D3, 0x0D3, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4,
        0x0D4, 0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D5, 0x0D6, 0x0D6, 0x0D6, 0x0D6,
        0x0D6, 0x0D6, 0x0D6, 0x0D6, 0x0D7, 0x0D7, 0x0D7, 0x0D7, 0x0D7, 0x0D7, 0x0D7, 0x0D7, 0x0D8,
        0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D9, 0x0D9, 0x0D9, 0x0D9, 0x0D9, 0x0D9,
        0x0D9, 0x0D9, 0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DB, 0x0DB, 0x0DB,
        0x0DB, 0x0DB, 0x0DB, 0x0DB, 0x0DB, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DC,
        0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DE, 0x0DE, 0x0DE, 0x0DE, 0x0DE,
        0x0DE, 0x0DE, 0x0DE, 0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0E0, 0x0E0,
        0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E0, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E1,
        0x0E1, 0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E2, 0x0E3, 0x0E3, 0x0E3, 0x0E3,
        0x0E3, 0x0E3, 0x0E3, 0x0E3, 0x0E4, 0x0E4, 0x0E4, 0x0E4, 0x0E4, 0x0E4, 0x0E4, 0x0E4, 0x0E5,
        0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E6, 0x0E6, 0x0E6, 0x0E6, 0x0E6, 0x0E6,
        0x0E6, 0x0E6, 0x0E7, 0x0E7, 0x0E7, 0x0E7, 0x0E7, 0x0E7, 0x0E7, 0x0E7, 0x0E8, 0x0E8, 0x0E8,
        0x0E8, 0x0E8, 0x0E8, 0x0E8, 0x0E8, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9, 0x0E9,
        0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EB, 0x0EB, 0x0EB, 0x0EB, 0x0EB,
        0x0EB, 0x0EB, 0x0EB, 0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0ED, 0x0ED,
        0x0ED, 0x0ED, 0x0ED, 0x0ED, 0x0ED, 0x0ED, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EE,
        0x0EE, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0F0, 0x0F0, 0x0F0, 0x0F0,
        0x0F0, 0x0F0, 0x0F0, 0x0F0, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F2,
        0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F2, 0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F3,
        0x0F3, 0x0F3, 0x0F4, 0x0F4, 0x0F4, 0x0F4, 0x0F4, 0x0F4, 0x0F4, 0x0F4, 0x0F5, 0x0F5, 0x0F5,
        0x0F5, 0x0F5, 0x0F5, 0x0F5, 0x0F5, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F6,
        0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F7, 0x0F8, 0x0F8, 0x0F8, 0x0F8, 0x0F8,
        0x0F8, 0x0F8, 0x0F8, 0x0F9, 0x0F9, 0x0F9, 0x0F9, 0x0F9, 0x0F9, 0x0F9, 0x0F9, 0x0FA, 0x0FA,
        0x0FA, 0x0FA, 0x0FA, 0x0FA, 0x0FA, 0x0FA, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB,
        0x0FB, 0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FC, 0x0FD, 0x0FD, 0x0FD, 0x0FD,
        0x0FD, 0x0FD, 0x0FD, 0x0FD, 0x0FE, 0x0FE, 0x0FE, 0x0FE, 0x0FE, 0x0FE, 0x0FE, 0x0FE, 0x0FF,
        0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x100, 0x100, 0x100, 0x100, 0x100, 0x100,
        0x100, 0x100, 0x101, 0x101, 0x101, 0x101, 0x101, 0x101, 0x101, 0x101, 0x102, 0x102, 0x102,
        0x102, 0x102, 0x102, 0x102, 0x102, 0x103, 0x103, 0x103, 0x103, 0x103, 0x103, 0x103, 0x103,
        0x104, 0x104, 0x104, 0x104, 0x104, 0x104, 0x104, 0x104, 0x105, 0x105, 0x105, 0x105, 0x105,
        0x105, 0x105, 0x105, 0x106, 0x106, 0x106, 0x106, 0x106, 0x106, 0x106, 0x106, 0x107, 0x107,
        0x107, 0x107, 0x107, 0x107, 0x107, 0x107, 0x108, 0x108, 0x108, 0x108, 0x108, 0x108, 0x108,
        0x108, 0x109, 0x109, 0x109, 0x109, 0x109, 0x109, 0x109, 0x109, 0x10A, 0x10A, 0x10A, 0x10A,
        0x10A, 0x10A, 0x10A, 0x10A, 0x10B, 0x10B, 0x10B, 0x10B, 0x10B, 0x10B, 0x10B, 0x10B, 0x10C,
        0x10C, 0x10C, 0x10C, 0x10C, 0x10C, 0x10C, 0x10C, 0x10D, 0x10D, 0x10D, 0x10D, 0x10D, 0x10D,
        0x10D, 0x10D, 0x10E, 0x10E, 0x10E, 0x10E, 0x10E, 0x10E, 0x10E, 0x10E, 0x10F, 0x10F, 0x10F,
        0x10F, 0x10F, 0x10F, 0x10F, 0x10F, 0x110, 0x110, 0x110, 0x110, 0x110, 0x110, 0x110, 0x110,
        0x111, 0x111, 0x111, 0x111, 0x111, 0x111, 0x111, 0x111, 0x112, 0x112, 0x112, 0x112, 0x112,
        0x112, 0x112, 0x112, 0x113, 0x113, 0x113, 0x113, 0x113, 0x113, 0x113, 0x113, 0x114, 0x114,
        0x114, 0x114, 0x114, 0x114, 0x114, 0x114, 0x115, 0x115, 0x115, 0x115, 0x115, 0x115, 0x115,
        0x115, 0x116, 0x116, 0x116, 0x116, 0x116, 0x116, 0x116, 0x116, 0x117, 0x117, 0x117, 0x117,
        0x117, 0x117, 0x117, 0x117, 0x118, 0x118, 0x118, 0x118, 0x118, 0x118, 0x118, 0x118, 0x119,
        0x119, 0x119, 0x119, 0x119, 0x119, 0x119, 0x119, 0x11A, 0x11A, 0x11A, 0x11A, 0x11A, 0x11A,
        0x11A, 0x11A, 0x11B, 0x11B, 0x11B, 0x11B, 0x11B, 0x11B, 0x11B, 0x11B, 0x11C, 0x11C, 0x11C,
        0x11C, 0x11C, 0x11C, 0x11C, 0x11C, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D, 0x11D,
        0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11E, 0x11F, 0x11F, 0x11F, 0x11F, 0x11F,
        0x11F, 0x11F, 0x11F, 0x120, 0x120, 0x120, 0x120, 0x120, 0x120, 0x120, 0x120, 0x121, 0x121,
        0x121, 0x121, 0x121, 0x121, 0x121, 0x121, 0x122, 0x122, 0x122, 0x122, 0x122, 0x122, 0x122,
        0x122, 0x123, 0x123, 0x123, 0x123, 0x123, 0x123, 0x123, 0x123, 0x124, 0x124, 0x124, 0x124,
        0x124, 0x124, 0x124, 0x124, 0x125, 0x125, 0x125, 0x125, 0x125, 0x125, 0x125, 0x125, 0x126,
        0x126, 0x126, 0x126, 0x126, 0x126, 0x126, 0x126, 0x127, 0x127, 0x127, 0x127, 0x127, 0x127,
        0x127, 0x127, 0x128, 0x128, 0x128, 0x128, 0x128, 0x128, 0x128, 0x128, 0x129, 0x129, 0x129,
        0x129, 0x129, 0x129, 0x129, 0x129, 0x12A, 0x12A, 0x12A, 0x12A, 0x12A, 0x12A, 0x12A, 0x12A,
        0x12B, 0x12B, 0x12B, 0x12B, 0x12B, 0x12B, 0x12B, 0x12B, 0x12C, 0x12C, 0x12C, 0x12C, 0x12C,
        0x12C, 0x12C, 0x12C, 0x12D, 0x12D, 0x12D, 0x12D, 0x12D, 0x12D, 0x12D, 0x12D, 0x12E, 0x12E,
        0x12E, 0x12E, 0x12E, 0x12E, 0x12E, 0x12E, 0x12F, 0x12F, 0x12F, 0x12F, 0x12F, 0x12F, 0x12F,
        0x12F, 0x130, 0x130, 0x130, 0x130, 0x130, 0x130, 0x130, 0x130, 0x131, 0x131, 0x131, 0x131,
        0x131, 0x131, 0x131, 0x131, 0x132, 0x132, 0x132, 0x132, 0x132, 0x132, 0x132, 0x132, 0x133,
        0x133, 0x133, 0x133, 0x133, 0x133, 0x133, 0x133, 0x134, 0x134, 0x134, 0x134, 0x134, 0x134,
        0x134, 0x134, 0x135, 0x135, 0x135, 0x135, 0x135, 0x135, 0x135, 0x135, 0x136, 0x136, 0x136,
        0x136, 0x136, 0x136, 0x136, 0x136, 0x137, 0x137, 0x137, 0x137, 0x137, 0x137, 0x137, 0x137,
        0x138, 0x138, 0x138, 0x138, 0x138, 0x138, 0x138, 0x138, 0x139, 0x139, 0x139, 0x139, 0x139,
        0x139, 0x139, 0x139, 0x13A, 0x13A, 0x13A, 0x13A, 0x13A, 0x13A, 0x13A, 0x13A, 0x13B, 0x13B,
        0x13B, 0x13B, 0x13B, 0x13B, 0x13B, 0x13B, 0x13C, 0x13C, 0x13C, 0x13C, 0x13C, 0x13C, 0x13C,
        0x13C, 0x13D, 0x13D, 0x13D, 0x13D, 0x13D, 0x13D, 0x13D, 0x13D, 0x13E, 0x13E, 0x13E, 0x13E,
        0x13E, 0x13E, 0x13E, 0x13E, 0x13F, 0x13F, 0x13F, 0x13F, 0x13F, 0x13F, 0x13F, 0x13F, 0x140,
        0x140, 0x140, 0x140, 0x140, 0x140, 0x140, 0x140, 0x141, 0x141, 0x141, 0x141, 0x141, 0x141,
        0x141, 0x141, 0x142, 0x142, 0x142, 0x142, 0x142, 0x142, 0x142, 0x142, 0x143, 0x143, 0x143,
        0x143, 0x143, 0x143, 0x143, 0x143, 0x144, 0x144, 0x144, 0x144, 0x144, 0x144, 0x144, 0x144,
        0x145, 0x145, 0x145, 0x145, 0x145, 0x145, 0x145, 0x145, 0x146, 0x146, 0x146, 0x146, 0x146,
        0x146, 0x146, 0x146, 0x147, 0x147, 0x147, 0x147, 0x147, 0x147, 0x147, 0x147, 0x148, 0x148,
        0x148, 0x148, 0x148, 0x148, 0x148, 0x148, 0x149, 0x149, 0x149, 0x149, 0x149, 0x149, 0x149,
        0x149, 0x14A, 0x14A, 0x14A, 0x14A, 0x14A, 0x14A, 0x14A, 0x14A, 0x14B, 0x14B, 0x14B, 0x14B,
        0x14B, 0x14B, 0x14B, 0x14B, 0x14C, 0x14C, 0x14C, 0x14C, 0x14C, 0x14C, 0x14C, 0x14C, 0x14D,
        0x14D, 0x14D, 0x14D, 0x14D, 0x14D, 0x14D, 0x14D, 0x14E, 0x14E, 0x14E, 0x14E, 0x14E, 0x14E,
        0x14E, 0x14E, 0x14F, 0x14F, 0x14F, 0x14F, 0x14F, 0x14F, 0x14F, 0x14F, 0x150, 0x150, 0x150,
        0x150, 0x150, 0x150, 0x150, 0x150, 0x151, 0x151, 0x151, 0x151, 0x151, 0x151, 0x151, 0x151,
        0x152, 0x152, 0x152, 0x152, 0x152, 0x152, 0x152, 0x152, 0x153, 0x153, 0x153, 0x153, 0x153,
        0x153, 0x153, 0x153, 0x154, 0x154, 0x154, 0x154, 0x154, 0x154, 0x154, 0x154, 0x155, 0x155,
        0x155, 0x155, 0x155, 0x155, 0x155, 0x155, 0x156, 0x156, 0x156, 0x156, 0x156, 0x156, 0x156,
        0x156, 0x157, 0x157, 0x157, 0x157, 0x157, 0x157, 0x157, 0x157, 0x158, 0x158, 0x158, 0x158,
        0x158, 0x158, 0x158, 0x158, 0x159, 0x159, 0x159, 0x159, 0x159, 0x159, 0x159, 0x159, 0x15A,
        0x15A, 0x15A, 0x15A, 0x15A, 0x15A, 0x15A, 0x15A, 0x15B, 0x15B, 0x15B, 0x15B, 0x15B, 0x15B,
        0x15B, 0x15B, 0x15C, 0x15C, 0x15C, 0x15C, 0x15C, 0x15C, 0x15C, 0x15C, 0x15D, 0x15D, 0x15D,
        0x15D, 0x15D, 0x15D, 0x15D, 0x15D, 0x15E, 0x15E, 0x15E, 0x15E, 0x15E, 0x15E, 0x15E, 0x15E,
        0x15F, 0x15F, 0x15F, 0x15F, 0x15F, 0x15F, 0x15F, 0x15F, 0x160, 0x160, 0x160, 0x160, 0x160,
        0x160, 0x160, 0x160, 0x161, 0x161, 0x161, 0x161, 0x161, 0x161, 0x161, 0x161, 0x162, 0x162,
        0x162, 0x162, 0x162, 0x162, 0x162, 0x162, 0x163, 0x163, 0x163, 0x163, 0x163, 0x163, 0x163,
        0x163, 0x164, 0x164, 0x164, 0x164, 0x164, 0x164, 0x164, 0x164, 0x165, 0x165, 0x165, 0x165,
        0x165, 0x165, 0x165, 0x165, 0x166, 0x166, 0x166, 0x166, 0x166, 0x166, 0x166, 0x166, 0x167,
        0x167, 0x167, 0x167, 0x167, 0x167, 0x167, 0x167, 0x168, 0x168, 0x168, 0x168, 0x168, 0x168,
        0x168, 0x168, 0x169, 0x169, 0x169, 0x169, 0x169, 0x169, 0x169, 0x169, 0x16A, 0x16A, 0x16A,
        0x16A, 0x16A, 0x16A, 0x16A, 0x16A, 0x16B, 0x16B, 0x16B, 0x16B, 0x16B, 0x16B, 0x16B, 0x16B,
        0x16C, 0x16C, 0x16C, 0x16C, 0x16C, 0x16C, 0x16C, 0x16C, 0x1C9, 0x1C9, 0x1C9, 0x1C9, 0x1C9,
        0x1C9, 0x1C9, 0x1C9, 0x1CA, 0x1CA, 0x1CA, 0x1CA, 0x1CA, 0x1CA, 0x1CA, 0x1CA, 0x1CB, 0x1CB,
        0x1CB, 0x1CB, 0x1CB, 0x1CB, 0x1CB, 0x1CB, 0x1CC, 0x1CC, 0x1CC, 0x1CC, 0x1CC, 0x1CC, 0x1CC,
        0x1CC, 0x1CD, 0x1CD, 0x1CD, 0x1CD, 0x1CD, 0x1CD, 0x1CD, 0x1CD, 0x1CD, 0x1CD, 0x1CE, 0x1CE,
        0x1CE, 0x1CE, 0x1CE, 0x1CE, 0x1CE, 0x1CE, 0x1CE, 0x1CE, 0x1CF, 0x1CF, 0x1CF, 0x1CF, 0x1CF,
        0x1CF, 0x1CF, 0x1CF, 0x1CF, 0x1CF, 0x1D0, 0x1D0, 0x1D0, 0x1D0, 0x1D0, 0x1D0, 0x1D0, 0x1D0,
        0x1D0, 0x1D0, 0x1D1, 0x1D1, 0x1D1, 0x1D1, 0x1D1, 0x1D1, 0x1D1, 0x1D1, 0x1D1, 0x1D1, 0x1D2,
        0x1D2, 0x1D2, 0x1D2, 0x1D2, 0x1D2, 0x1D2, 0x1D2, 0x1D2, 0x1D2, 0x1D3, 0x1D3, 0x1D3, 0x1D3,
        0x1D3, 0x1D3, 0x1D3, 0x1D3, 0x1D3, 0x1D3, 0x1D4, 0x1D4, 0x1D4, 0x1D4, 0x1D4, 0x1D4, 0x1D4,
        0x1D4, 0x1D4, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D6, 0x1D6,
        0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D7, 0x1D7, 0x1D7, 0x1D7, 0x1D7,
        0x1D7, 0x1D7, 0x1D7, 0x1D7, 0x1D7, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D8,
        0x1D8, 0x1D8, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1D9, 0x1DA,
        0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DB, 0x1DB, 0x1DB, 0x1DB,
        0x1DB, 0x1DB, 0x1DB, 0x1DB, 0x1DB, 0x1DB, 0x1DC, 0x1DC, 0x1DC, 0x1DC, 0x1DC, 0x1DC, 0x1DC,
        0x1DC, 0x1DC, 0x1DC, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DE, 0x1DE,
        0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DF, 0x1DF, 0x1DF, 0x1DF, 0x1DF,
        0x1DF, 0x1DF, 0x1DF, 0x1DF, 0x1DF, 0x1E0, 0x1E0, 0x1E0, 0x1E0, 0x1E0, 0x1E0, 0x1E0, 0x1E0,
        0x1E0, 0x1E0, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E2,
        0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E2, 0x1E3, 0x1E3, 0x1E3, 0x1E3,
        0x1E3, 0x1E3, 0x1E3, 0x1E3, 0x1E3, 0x1E3, 0x1E4, 0x1E4, 0x1E4, 0x1E4, 0x1E4, 0x1E4, 0x1E4,
        0x1E4, 0x1E4, 0x1E4, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E5, 0x1E6,
        0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E7, 0x1E7, 0x1E7, 0x1E7, 0x1E7,
        0x1E7, 0x1E7, 0x1E7, 0x1E7, 0x1E7, 0x1E8, 0x1E8, 0x1E8, 0x1E8, 0x1E8, 0x1E8, 0x1E8, 0x1E8,
        0x1E8, 0x1E8, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1E9, 0x1EA,
        0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EB, 0x1EB, 0x1EB, 0x1EB,
        0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EC, 0x1EC, 0x1EC, 0x1EC, 0x1EC, 0x1EC, 0x1EC,
        0x1EC, 0x1EC, 0x1EC, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1ED,
        0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EE, 0x1EF, 0x1EF, 0x1EF, 0x1EF, 0x1EF,
        0x1EF, 0x1EF, 0x1EF, 0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F1, 0x1F1,
        0x1F1, 0x1F1, 0x1F1, 0x1F1, 0x1F1, 0x1F1, 0x1F2, 0x1F2, 0x1F2, 0x1F2, 0x1F2, 0x1F2, 0x1F2,
        0x1F2, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F3, 0x1F4, 0x1F4, 0x1F4, 0x1F4,
        0x1F4, 0x1F4, 0x1F4, 0x1F4, 0x1F5, 0x1F5, 0x1F5, 0x1F5, 0x1F5, 0x1F5, 0x1F5, 0x1F5, 0x1F6,
        0x1F6, 0x1F6, 0x1F6, 0x1F6, 0x1F6, 0x1F6, 0x1F6, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F7,
        0x1F7, 0x1F7, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F8, 0x1F9, 0x1F9, 0x1F9,
        0x1F9, 0x1F9, 0x1F9, 0x1F9, 0x1F9, 0x1FA, 0x1FA, 0x1FA, 0x1FA, 0x1FA, 0x1FA, 0x1FA, 0x1FA,
        0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FC, 0x1FC, 0x1FC, 0x1FC, 0x1FC,
        0x1FC, 0x1FC, 0x1FC, 0x1FD, 0x1FD, 0x1FD, 0x1FD, 0x1FD, 0x1FD, 0x1FD, 0x1FD, 0x1FE, 0x1FE,
        0x1FE, 0x1FE, 0x1FE, 0x1FE, 0x1FE, 0x1FE, 0x1FF, 0x1FF, 0x1FF, 0x1FF, 0x1FF, 0x1FF, 0x1FF,
        0x1FF,
    ];

    h_counter_test(
        EXPECTED_PIXELS,
        EXPECTED_H,
        scanline_mclk_to_pixel_h40,
        pixel_to_internal_h_h40,
    );
}
